/*******************************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 ******************************************************************************/
package org.apache.olingo.odata2.jpa.processor.core.access.data;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringWriter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.olingo.odata2.api.edm.*;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.core.edm.AbstractSimpleType;
import org.apache.olingo.odata2.core.edm.provider.EdmPropertyImplProv;
import org.apache.olingo.odata2.core.edm.provider.EdmSimplePropertyImplProv;
import org.apache.olingo.odata2.jpa.processor.api.ODataJPAContext;
import org.apache.olingo.odata2.jpa.processor.api.ODataJPAQueryExtensionEntityListener;
import org.apache.olingo.odata2.jpa.processor.api.ODataJPATombstoneEntityListener;
import org.apache.olingo.odata2.jpa.processor.api.exception.ODataJPARuntimeException;
import org.apache.olingo.odata2.jpa.processor.api.model.JPAEdmMapping;
import org.apache.olingo.odata2.jpa.processor.core.ODataJPAConfig;
import org.apache.olingo.odata2.jpa.processor.core.model.JPAEdmMappingImpl;

public final class JPAEntityParser {

  /*
   * List of buffers used by the Parser
   */
  private static short MAX_SIZE = 10;
  public static final String ACCESS_MODIFIER_GET = "get";
  public static final String ACCESS_MODIFIER_SET = "set";
  private static final String ACCESS_MODIFIER_IS = "is";
  private final UriInfo infoView;

  private HashMap<String, HashMap<String, Method>> jpaEntityAccessMap = null;
  private HashMap<String, HashMap<String, String>> jpaEmbeddableKeyMap = null;


  private final ODataJPAContext oDataJPAContext;

  public JPAEntityParser(final ODataJPAContext context, final UriInfo infoView) {
    oDataJPAContext = context;
    this.infoView = infoView;
    jpaEntityAccessMap = new HashMap<String, HashMap<String, Method>>(
        MAX_SIZE);
    jpaEmbeddableKeyMap = new HashMap<String, HashMap<String, String>>();
  };

  public HashMap<String, Method> getJPAEntityAccessMap(final String jpaEntityName) {
    return jpaEntityAccessMap.get(jpaEntityName);
  }

  public HashMap<String, String> getJPAEmbeddableKeyMap(final String jpaEntityName) {
    return jpaEmbeddableKeyMap.get(jpaEntityName);
  }

  public List<Map<String, Object>> parse2EdmEntityList(final UriInfo infoView, final Collection<Object> jpaEntityList,
      final List<EdmProperty> properties, final EdmEntityType entity)
      throws ODataJPARuntimeException {

    if (jpaEntityList == null) {
      return null;
    }
    List<Map<String, Object>> edmEntityList = new ArrayList<Map<String, Object>>();
    for (Object item : jpaEntityList) {
      edmEntityList.add(parse2EdmPropertyValueMap(item, properties, entity));
    }

    return edmEntityList;
  }

  private String getMethodFromProperty(String propertyName) {
    return String.format("get%s", propertyName.substring(0,1).toUpperCase() + propertyName.substring(1));
  }

  public final HashMap<String, Object> parse2EdmPropertyValueMap(final Object jpaEntity,
                                                                 final List<EdmProperty> selectPropertyList, final EdmStructuralType structuralType) throws ODataJPARuntimeException {

    HashMap<String, Object> edmEntity = new HashMap<String, Object>();
    HashMap<String, Method> accessModifierMap = null;
    Object propertyValue = null;
    String jpaEntityAccessKey = null;

    jpaEntityAccessKey = jpaEntity.getClass().getName();
    if (!jpaEntityAccessMap.containsKey(jpaEntityAccessKey)) {
      accessModifierMap =
          getAccessModifiers((List<EdmProperty>) selectPropertyList, jpaEntity.getClass(), ACCESS_MODIFIER_GET);
    } else {
      accessModifierMap = jpaEntityAccessMap.get(jpaEntityAccessKey);
    }

    ODataJPAQueryExtensionEntityListener queryListener = null;
    try {
      if (((JPAEdmMapping) ((EdmEntityType) structuralType).getMapping()).getODataJPATombstoneEntityListener() != null) {

        ODataJPATombstoneEntityListener listener = ((JPAEdmMapping) ((EdmEntityType) structuralType).getMapping()).getODataJPATombstoneEntityListener().newInstance();

        if (listener instanceof ODataJPAQueryExtensionEntityListener) {
          queryListener = (ODataJPAQueryExtensionEntityListener) listener;
        }

      } else {
        if (oDataJPAContext != null)
          queryListener = oDataJPAContext.getODataJPAQueryExtensionEntityListener();
      }
    } catch (Exception e) {
      e.printStackTrace();
    }

    boolean hasCalc = false;

    Map<String, Object> defaults = null;

    if (infoView != null && infoView.isNew() && queryListener != null) {
      defaults = queryListener.getDefaultFieldValues((EdmEntityType) structuralType, edmEntity);
    }

    for (EdmProperty property : selectPropertyList) {
      try {
        if (property.getMapping().isCalculated()) {
          hasCalc = true;
        }

        if (queryListener != null) {
          if (!queryListener.authorizeProperty((EdmEntityType) structuralType, property, ACCESS_MODIFIER_GET.toUpperCase())) {
            continue;
          }
        }

        propertyValue = jpaEntity;
        String propertyName = property.getName();
        if (((EdmSimplePropertyImplProv) property).getComposite() != null) {
          propertyValue = "";
          boolean isNull = true;
          for(EdmProperty p: ((EdmSimplePropertyImplProv) property).getComposite()) {
            if (propertyValue != null) {
              String propertyValueStr = propertyValue.toString();
              if (!propertyValueStr.isEmpty()) {
                propertyValue = propertyValueStr + ODataJPAConfig.COMPOSITE_SEPARATOR;
              }
            }

            Object value = null;
            if (((EdmSimplePropertyImplProv) p).getProperty().isForeignKey()) {
              String methodName = null;
              methodName = jpaEmbeddableKeyMap.get(jpaEntityAccessKey).get(p.getName());
              boolean isVirtualAccess = false;
              if (property.getMapping() != null && p.getMapping() instanceof JPAEdmMappingImpl) {
                isVirtualAccess = ((JPAEdmMappingImpl) property.getMapping()).isVirtualAccess();
              }
              if (isVirtualAccess) {
                value = getEmbeddablePropertyValue(methodName, jpaEntity, true);
              } else {
                value = getEmbeddablePropertyValue(methodName, jpaEntity);
              }
            } else {
              value = getPropertyValue(accessModifierMap.get(p.getName()), jpaEntity, p.getName());
            }
            if (value != null) {
              isNull = false;
            }
            if (value != null) {
              String valueStr = ((AbstractSimpleType) p.getType()).valueToObjectKeyString(value, EdmLiteralKind.URI, p.getFacets());

              String propertyValueStr = propertyValue.toString();
              propertyValue = propertyValueStr + valueStr;
            } else {
              String propertyValueStr = propertyValue.toString();
              propertyValue = propertyValueStr + "null";
            }
          }
          if (isNull) {
            propertyValue = null;
          }
        } else {

          Method method = accessModifierMap.get(propertyName);
          if (method == null && !property.getMapping().isCalculated()) {
            String methodName = null;
            if (jpaEmbeddableKeyMap.get(jpaEntityAccessKey) != null)
              methodName = jpaEmbeddableKeyMap.get(jpaEntityAccessKey).get(propertyName);
            if (methodName != null) {
              boolean isVirtualAccess = false;
              if (property.getMapping() != null && property.getMapping() instanceof JPAEdmMappingImpl) {
                isVirtualAccess = ((JPAEdmMappingImpl) property.getMapping()).isVirtualAccess();
              }
              if (isVirtualAccess) {
                propertyValue = getEmbeddablePropertyValue(methodName, propertyValue, true);
                //Trying to get the value if the object is a hashmap and the previously obtained value is null
                if (propertyValue == null && ((VirtualClass) jpaEntity).getObject() instanceof HashMap) {
                  try {
                    methodName = getMethodFromProperty(propertyName);
                    propertyValue = getEmbeddablePropertyValue(methodName, jpaEntity, true);
                    try {
                      //if is Virtual access, the value never can be a same object of mapping obj
                      if (propertyValue != null && propertyValue.getClass().equals(((JPAEdmMappingImpl)structuralType.getMapping()).getOriginaType())) {
                        propertyValue = null;
                      }
                    }
                    catch (Exception e) {

                    }

                  }
                  catch (Exception e) {
                    //do nothing
                  }
                }
              } else {
                propertyValue = getEmbeddablePropertyValue(methodName, propertyValue);
              }
            }
            else {
              methodName = getMethodFromProperty(propertyName);
              propertyValue = getEmbeddablePropertyValue(methodName, propertyValue);
            }
          } else {
            propertyValue = getPropertyValue(accessModifierMap.get(propertyName), propertyValue, propertyName);
          }
          if (property.getType().getKind()
              .equals(EdmTypeKind.COMPLEX)) {
            propertyValue = parse2EdmPropertyValueMap(propertyValue, (EdmStructuralType) property.getType());
          }

          if (defaults != null && propertyValue == null) {
            propertyValue = defaults.get(propertyName);
          }
        }

        edmEntity.put(propertyName, propertyValue);
      } catch (EdmException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      } catch (SecurityException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      } catch (IllegalArgumentException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      }
    }

    if (hasCalc) {
      Map<String, Object> calcs = queryListener.getCalcFieldValues((EdmEntityType) structuralType, edmEntity);
      if (calcs != null) {
        edmEntity.putAll(calcs);
      }
    }

    if (queryListener != null) {
      queryListener.execEvent(infoView, (EdmEntityType) structuralType, "onNavigate", edmEntity);
    }


    return edmEntity;
  }

  public final List<Map<String, Object>> parse2EdmEntityList(final Collection<Object> jpaEntityList,
      final EdmStructuralType structuralType) throws ODataJPARuntimeException {
    if (jpaEntityList == null || structuralType == null) {
      return null;
    }
    List<EdmProperty> edmProperties = getEdmProperties(structuralType);
    List<Map<String, Object>> edmEntityList = new ArrayList<Map<String, Object>>();
    for (Object jpaEntity : jpaEntityList) {
      edmEntityList.add(parse2EdmPropertyValueMap(jpaEntity, edmProperties, structuralType));
    }

    return edmEntityList;
  }

  public final HashMap<String, Object> parse2EdmPropertyValueMap(final Object jpaEntity,
      final EdmStructuralType structuralType) throws ODataJPARuntimeException {
    if (jpaEntity == null || structuralType == null) {
      return null;
    }
    return parse2EdmPropertyValueMap(jpaEntity, getEdmProperties(structuralType), structuralType);
  }

  public final HashMap<String, Object> parse2EdmNavigationValueMap(
      final Object jpaEntity, final List<EdmNavigationProperty> navigationPropertyList)
      throws ODataJPARuntimeException {
    Object result = null;
    String methodName = null;
    HashMap<String, Object> navigationMap = new HashMap<String, Object>();
    if (jpaEntity == null) {
      return navigationMap;
    }
    if (navigationPropertyList != null
        && !navigationPropertyList.isEmpty()) {

    	try {
    		for (EdmNavigationProperty navigationProperty : navigationPropertyList) {
    			methodName = getAccessModifierName(navigationProperty.getName(),
    					navigationProperty.getMapping(), ACCESS_MODIFIER_GET);
    			Method getterMethod = null;
    			JPAEdmMapping jpaEdmMapping = (JPAEdmMapping)navigationProperty.getMapping();
    			if(jpaEdmMapping != null && jpaEdmMapping.isVirtualAccess()) {
    				getterMethod = jpaEntity.getClass().getMethod(ACCESS_MODIFIER_GET, String.class);
    			}else{
    				getterMethod = jpaEntity.getClass()
    						.getMethod(methodName, (Class<?>[]) null);
    			}

    			getterMethod.setAccessible(true);
    			result = getPropertyValue(getterMethod, jpaEntity,
    					navigationProperty.getMapping().getInternalName());
    			navigationMap.put(navigationProperty.getName(), result);
    		}
    	} catch (IllegalArgumentException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      } catch (EdmException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      } catch (SecurityException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      } catch (NoSuchMethodException e) {
        throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
      }
    }
    return navigationMap;
  }

  public Method getAccessModifierSet(final Object jpaEntity, final String methodName) throws ODataJPARuntimeException {
    Class<?> jpaType = jpaEntity.getClass();
    String methodNameGet = ACCESS_MODIFIER_GET + methodName.substring(3);
    Method method = null;

    try {
      method = jpaType.getMethod(methodNameGet, (Class<?>[]) null);
      Class<?> parameterType = method.getReturnType();
      method = jpaType.getMethod(methodName, new Class<?>[] { parameterType });
    } catch (NoSuchMethodException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (SecurityException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    }

    return method;
  }

  public HashMap<String, Method> getAccessModifiers(final Object jpaEntity,
      final EdmStructuralType structuralType, final String accessModifier) throws ODataJPARuntimeException {
    return getAccessModifiers(getEdmProperties(structuralType), jpaEntity.getClass(), accessModifier);
  }

  public static Object getPropertyValue(final Method method, final Object entity, String propertyName) 
		  throws ODataJPARuntimeException {
    Object propertyValue = null;
    if (method == null) {
      return null;
    }
    try {
      method.setAccessible(true);
      Class<?> returnType = method.getReturnType();

      if (returnType.equals(char[].class)) {
        char[] ch = (char[]) method.invoke(entity);
        if (ch != null) {
          propertyValue = (String) String.valueOf((char[]) method.invoke(entity));
        }
      } else if (returnType.equals(Character[].class)) {
        propertyValue = (String) toString((Character[]) method.invoke(entity));
      } else if (returnType.equals(char.class)) {
        char c = (Character) method.invoke(entity);
        if (c != '\u0000') {
          propertyValue = (String) String.valueOf(c);
        }
      } else if (returnType.equals(Character.class)) {
        Character c = (Character) method.invoke(entity);
        if (c != null) {
          propertyValue = toString(new Character[] { c });
        }
      } else if (returnType.equals(Blob.class)) {
        propertyValue = getBytes((Blob) method.invoke(entity));
      } else if (returnType.equals(Clob.class)) {
        propertyValue = getString((Clob) method.invoke(entity));
      } else {
    	  if(method.getParameterTypes().length>0) {
    		  propertyValue = method.invoke(entity,propertyName);
    	  } else {
    		  propertyValue = method.invoke(entity);
    	  }
      }
    } catch (IllegalAccessException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (IllegalArgumentException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (InvocationTargetException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (SecurityException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    }

    return propertyValue;
  }

  public static String getString(final Clob clob) throws ODataJPARuntimeException {
    Reader stringReader = null;
    try {

      if (clob == null) {
        return null;
      }

      stringReader = clob.getCharacterStream();
      StringWriter buffer = null;
      long clobSize = clob.length();
      long remainingClobSize = clobSize;
      int len = 0;
      int off = 0;
      boolean bufferNotEmpty = false;
      if (clobSize > Integer.MAX_VALUE) {
        buffer = new StringWriter(Integer.MAX_VALUE);
        len = Integer.MAX_VALUE;
        bufferNotEmpty = true;
      } else {
        buffer = new StringWriter((int) clobSize);
        len = (int) clobSize;
      }
      char c[] = new char[len];
      while (remainingClobSize > len) {
        stringReader.read(c, off, len);
        buffer.write(c);
        off = len + 1;
        remainingClobSize = remainingClobSize - Integer.MAX_VALUE;
        if (remainingClobSize > Integer.MAX_VALUE) {
          len = Integer.MAX_VALUE;
        } else {
          len = (int) remainingClobSize;
        }
      }
      if (remainingClobSize <= len) {
        stringReader.read(c, off, len);
      }
      if (bufferNotEmpty) {
        return buffer.toString();
      } else {
        return new String(c);
      }

    } catch (SQLException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (IOException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } finally {
      if (stringReader != null) {
        try {
          stringReader.close();
        } catch (IOException e) {
          //do nothing
        }
      }
    }

  }

  public static byte[] getBytes(final Blob blob) throws ODataJPARuntimeException {
    InputStream is = null;
    ByteArrayOutputStream buffer = null;
    try {

      if (blob == null) {
        return null;
      }

      long blobSize = blob.length();
      long remainingBlobSize = blobSize;
      int len = 0;
      int off = 0;
      boolean bufferNotEmpty = false;
      if (blobSize > Integer.MAX_VALUE) { // Edge case when the Blob Size is more than 2GB
        buffer = new ByteArrayOutputStream(Integer.MAX_VALUE);
        len = Integer.MAX_VALUE;
        bufferNotEmpty = true;
      } else {
        buffer = new ByteArrayOutputStream((int) blobSize);
        len = (int) blobSize;
      }

      is = blob.getBinaryStream();
      byte b[] = new byte[len];
      while (remainingBlobSize > len) {
        is.read(b, off, len);
        buffer.write(b);
        off = len + 1;
        remainingBlobSize = remainingBlobSize - Integer.MAX_VALUE;
        if (remainingBlobSize > Integer.MAX_VALUE) {
          len = Integer.MAX_VALUE;
        } else {
          len = (int) remainingBlobSize;
        }
      }
      if (remainingBlobSize <= len) {
        is.read(b, off, len);
      }
      if (bufferNotEmpty) {
        return buffer.toByteArray();
      } else {
        return b;
      }
    } catch (SQLException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (IOException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } finally {
      if (is != null) {
        try {
          is.close();
        } catch (IOException e) {
          // do nothing
        }
      }
    }
  }

  public Object getEmbeddablePropertyValue(final String methodName, final Object jpaEntity, boolean isVirtualAccess)
      throws ODataJPARuntimeException {

    String[] nameParts = methodName.split("\\.");
    Object propertyValue = jpaEntity;
    Method method = null;
    try {
      for (String namePart : nameParts) {
        if (propertyValue == null) {
          break;
        }
        if (isVirtualAccess) {

        	method = propertyValue.getClass().getMethod(ACCESS_MODIFIER_GET, String.class);
        	namePart = namePart.replaceFirst(ACCESS_MODIFIER_GET, "");
        } else {
        	method = propertyValue.getClass().getMethod(namePart, (Class<?>[]) null);
        }
        method.setAccessible(true);
        propertyValue = getPropertyValue(method, propertyValue,namePart);
        isVirtualAccess = false;
      }
    } catch (NoSuchMethodException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (SecurityException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    }
    return propertyValue;
  }

  public Object getEmbeddablePropertyValue(final String methodName, final Object jpaEntity)
	      throws ODataJPARuntimeException {

	  return getEmbeddablePropertyValue(methodName, jpaEntity, false);
  }
  
  public static String toString(final Character[] input) {
    if (input == null) {
      return null;
    }

    StringBuilder builder = new StringBuilder();
    for (Character element : input) {
      if (element == null) {
        continue;
      }
      builder.append(element.charValue());
    }
    return builder.toString();

  }

  public static Character[] toCharacterArray(final String input) {
    if (input == null) {
      return null;
    }

    Character[] characters = new Character[input.length()];
    char[] chars = ((String) input).toCharArray();
    for (int i = 0; i < input.length(); i++) {
      characters[i] = new Character(chars[i]);
    }

    return characters;
  }

  public static String getAccessModifierName(final String propertyName, final EdmMapping mapping,
      final String accessModifier)
      throws ODataJPARuntimeException {
    String name = null;
    StringBuilder builder = new StringBuilder();
    String[] nameParts = {};
    if (mapping == null || mapping.getInternalName() == null) {
      name = propertyName;
    } else {
      name = mapping.getInternalName();
    }
    if (name != null) {
      nameParts = name.split("\\.");
    }
    if (nameParts.length == 1) {
      if (name != null) {
        char c = Character.toUpperCase(name.charAt(0));

        builder.append(accessModifier).append(c).append(name.substring(1))
            .toString();
      }
    } else if (nameParts.length > 1) {

      for (int i = 0; i < nameParts.length; i++) {
        name = nameParts[i];
        char c = Character.toUpperCase(name.charAt(0));
        if (i == 0) {
          builder.append(accessModifier).append(c).append(name.substring(1));
        } else {
          builder.append(".").append(accessModifier).append(c)
              .append(name.substring(1));
        }
      }
    } else {
      return null;
    }

    if (builder.length() > 0) {
      return builder.toString();
    } else {
      return null;
    }

  }

  public Method getAccessModifier(final Class<?> jpaEntityType, final EdmNavigationProperty navigationProperty,
      final String accessModifier)
      throws ODataJPARuntimeException {

    try {

      JPAEdmMapping navPropMapping = (JPAEdmMapping) navigationProperty.getMapping();

      String name;
      Class<?>[] params = null;
      if (navPropMapping != null && navPropMapping.isVirtualAccess()) {
  
    	  return jpaEntityType.getMethod(ACCESS_MODIFIER_SET, String.class, Object.class);
      } else {
	      name = getAccessModifierName(navigationProperty.getName(), (EdmMapping) navPropMapping, accessModifier);
	
	      if (accessModifier.equals(ACCESS_MODIFIER_SET)) {
	        EdmAssociationEnd end = navigationProperty.getRelationship().getEnd(navigationProperty.getToRole());
	        switch (end.getMultiplicity()) {
	        case MANY:
	          params = new Class<?>[] { navPropMapping != null ? navPropMapping.getJPAType() : null };
	          break;
	        case ONE:
          case ZERO_TO_ONE:
	          params = new Class<?>[] { ((JPAEdmMapping) end.getEntityType().getMapping()).getJPAType() };
	          break;
	        default:
	          break;
	        }
	      }
	      return jpaEntityType.getMethod(name, params);
      }

    } catch (NoSuchMethodException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (SecurityException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    } catch (EdmException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    }
  }

  private HashMap<String, Method> getAccessModifiers(final List<EdmProperty> edmProperties,
      final Class<?> jpaEntityType,
      final String accessModifier) throws ODataJPARuntimeException {

    HashMap<String, Method> accessModifierMap = jpaEntityAccessMap.get(jpaEntityType.getName());
    if (accessModifierMap == null) {
      accessModifierMap = new HashMap<String, Method>();
      jpaEntityAccessMap.put(jpaEntityType.getName(), accessModifierMap);
    }
    HashMap<String, String> embeddableKey = jpaEmbeddableKeyMap.get(jpaEntityType.getName());
    if (embeddableKey == null) {
      embeddableKey = new HashMap<String, String>();
    }

    Method method = null;
    try {
      for (EdmProperty property : edmProperties) {
        if (((EdmSimplePropertyImplProv) property).getComposite() != null) {
          accessModifierMap.putAll(getAccessModifiers(((EdmSimplePropertyImplProv) property).getComposite(), jpaEntityType, accessModifier));
        }
        method = null;
        String propertyName = property.getName();
        if (accessModifierMap.containsKey(propertyName)) {
          continue;
        }
        String methodName = getAccessModifierName(property.getName(), property.getMapping(), accessModifier);
        String[] nameParts = methodName != null ? methodName.split("\\.") : new String[0];
        try {
          if (nameParts.length > 1) {
            if (!embeddableKey.containsKey(propertyName)) {
              embeddableKey.put(propertyName, methodName);
              continue;
            }
          } else {
        	  if (accessModifier.equals(ACCESS_MODIFIER_SET)) {
        		  JPAEdmMapping jpaEdmMapping = (JPAEdmMapping) property.getMapping();
        		  if(jpaEdmMapping != null && jpaEdmMapping.isVirtualAccess()) {
        		    try {
                  accessModifierMap.put(propertyName, jpaEntityType.getMethod(ACCESS_MODIFIER_SET,
                      new Class<?>[]{String.class, Object.class}));
                } catch(Exception e) {
        		      try {
                    accessModifierMap.put(propertyName, ReflectionUtil.getMethod(jpaEntityType, methodName));
                  } catch(Exception e2) {
                    accessModifierMap.put(propertyName, null);
                  }
                }
        		  }else {
        			  accessModifierMap.put(propertyName,  ReflectionUtil.getMethod(jpaEntityType, methodName));
        		  }
        	  } else {
        		  JPAEdmMapping jpaEdmMapping = (JPAEdmMapping) property.getMapping();
        		  if(jpaEdmMapping != null && jpaEdmMapping.isVirtualAccess()) {
        		    try {
                  method = jpaEntityType.getMethod(ACCESS_MODIFIER_GET, String.class);
                } catch(Exception e) {
        		      //Abafa
                }
        		  }else{
        			  method =  ReflectionUtil.getMethod(jpaEntityType, methodName);
        		  }
        	  }
          }
        } catch (EdmException exp) {
          throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, exp);
        } catch (NoSuchMethodException e1) {
          try {
            EdmSimpleType edmSimpleType = (EdmSimpleType) property.getType();
            if (edmSimpleType == EdmSimpleTypeKind.Boolean.getEdmSimpleTypeInstance()
                && accessModifier.equals(ACCESS_MODIFIER_GET)) {
              String nameWithIs = getAccessModifierName(property.getName(),
                  property.getMapping(), ACCESS_MODIFIER_IS);
              method = jpaEntityType.getMethod(nameWithIs, (Class<?>[]) null);
            } else {
              throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e1);
            }
          } catch (EdmException exp) {
            throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, exp);
          } catch (NoSuchMethodException exp) {
            throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, exp);
          } catch (SecurityException exp) {
            throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, exp);
          }
        } catch (SecurityException e1) {
          throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e1);
        }
        if (method != null) {
          accessModifierMap.put(propertyName, method);
        }
      }
    } catch (EdmException exp) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, exp);
    }
    if (!embeddableKey.isEmpty()) {
      if (!jpaEmbeddableKeyMap.containsKey(jpaEntityType.getName())) {
        jpaEmbeddableKeyMap.put(jpaEntityType.getName(), embeddableKey);
      } else {
        jpaEmbeddableKeyMap.get(jpaEntityType.getName()).putAll(embeddableKey);
      }
    }
    return accessModifierMap;
  }

  private List<EdmProperty> getEdmProperties(final EdmStructuralType structuralType) throws ODataJPARuntimeException {
    List<EdmProperty> edmProperties = new ArrayList<EdmProperty>();
    try {
      for (String propertyName : structuralType.getPropertyNames()) {
        edmProperties.add((EdmProperty) structuralType.getProperty(propertyName));
      }
    } catch (EdmException e) {
      throw ODataJPARuntimeException.throwException(ODataJPARuntimeException.INNER_EXCEPTION, e);
    }
    return edmProperties;
  }

}
