package org.apache.olingo.odata2.core.servlet;

import org.apache.olingo.odata2.api.processor.ODataRequest;

import javax.servlet.RequestDispatcher;
import javax.servlet.ServletInputStream;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.Principal;
import java.util.*;

public class ODataRequestWrapper implements HttpServletRequest {
  private ODataRequest oDataRequest;
  private HttpServletRequest request;

  public ODataRequestWrapper(ODataRequest oDataRequest, HttpServletRequest request) {
    this.oDataRequest = oDataRequest;
    this.request = request;
  }

  @Override
  public String getAuthType() {
    return request.getAuthType();
  }

  @Override
  public Cookie[] getCookies() {
    return request.getCookies();
  }

  @Override
  public long getDateHeader(String name) {
    return -1;
  }

  @Override
  public String getHeader(String name) {
    List<String> values = oDataRequest.getRequestHeaders().get(name);
    if (values != null && !values.isEmpty()) {
      String result = "";
      for (String v : values) {
        if (!result.isEmpty()) {
          result += ", ";
        }
        result += v;
      }
      return result;
    }

    return null;
  }

  @Override
  public Enumeration<String> getHeaders(String name) {
    return Collections.enumeration(oDataRequest.getRequestHeaders().get(name));
  }

  @Override
  public Enumeration<String> getHeaderNames() {
    return Collections.enumeration(oDataRequest.getRequestHeaders().keySet());
  }

  @Override
  public int getIntHeader(String name) {
    return Integer.valueOf(getHeader(name));
  }

  @Override
  public String getMethod() {
    return oDataRequest.getMethod().toString();
  }

  @Override
  public String getPathInfo() {
    return oDataRequest.getPathInfo().toString();
  }

  @Override
  public String getPathTranslated() {
    return oDataRequest.getPathInfo().toString();
  }

  @Override
  public String getContextPath() {
    return request.getContextPath();
  }

  static String urlEncodeUTF8(String s) {
    try {
      return URLEncoder.encode(s, "UTF-8");
    } catch (UnsupportedEncodingException e) {
      throw new UnsupportedOperationException(e);
    }
  }

  static String urlEncodeUTF8(Map<?, ?> map) {
    StringBuilder sb = new StringBuilder();
    for (Map.Entry<?, ?> entry : map.entrySet()) {
      if (sb.length() > 0) {
        sb.append("&");
      }
      sb.append(String.format("%s=%s",
          urlEncodeUTF8(entry.getKey().toString()),
          urlEncodeUTF8(entry.getValue().toString())
      ));
    }
    return sb.toString();
  }

  @Override
  public String getQueryString() {
    return urlEncodeUTF8(oDataRequest.getQueryParameters());
  }

  @Override
  public String getRemoteUser() {
    return request.getRemoteUser();
  }

  @Override
  public boolean isUserInRole(String role) {
    return request.isUserInRole(role);
  }

  @Override
  public Principal getUserPrincipal() {
    return request.getUserPrincipal();
  }

  @Override
  public String getRequestedSessionId() {
    return request.getRequestedSessionId();
  }

  @Override
  public String getRequestURI() {
    return oDataRequest.getPathInfo().getRequestUri().toString();
  }

  @Override
  public StringBuffer getRequestURL() {
    return new StringBuffer(oDataRequest.getPathInfo().getRequestUri().toString());
  }

  @Override
  public String getServletPath() {
    return request.getServletPath();
  }

  @Override
  public HttpSession getSession(boolean create) {
    return request.getSession(create);
  }

  @Override
  public HttpSession getSession() {
    return request.getSession();
  }

  @Override
  public boolean isRequestedSessionIdValid() {
    return request.isRequestedSessionIdValid();
  }

  @Override
  public boolean isRequestedSessionIdFromCookie() {
    return request.isRequestedSessionIdFromCookie();
  }

  @Override
  public boolean isRequestedSessionIdFromURL() {
    return request.isRequestedSessionIdFromURL();
  }

  @Override
  public boolean isRequestedSessionIdFromUrl() {
    return request.isRequestedSessionIdFromUrl();
  }

  @Override
  public Object getAttribute(String name) {
    return request.getAttribute(name);
  }

  @Override
  public Enumeration<String> getAttributeNames() {
    return request.getAttributeNames();
  }

  @Override
  public String getCharacterEncoding() {
    return request.getCharacterEncoding();
  }

  @Override
  public void setCharacterEncoding(String env) throws UnsupportedEncodingException {
    request.setCharacterEncoding(env);
  }

  @Override
  public int getContentLength() {
    return request.getContentLength();
  }


  @Override
  public String getContentType() {
    return request.getContentType();
  }

  @Override
  public ServletInputStream getInputStream() throws IOException {
    return request.getInputStream();
  }

  @Override
  public String getParameter(String name) {
    return oDataRequest.getQueryParameters().get(name);
  }

  @Override
  public Enumeration<String> getParameterNames() {
    return Collections.enumeration(oDataRequest.getQueryParameters().keySet());
  }

  @Override
  public String[] getParameterValues(String name) {
    List<String> values = oDataRequest.getAllQueryParameters().get(name);
    if (values != null) {
      return values.toArray(new String[0]);
    }
    return new String[0];
  }

  @Override
  public Map<String, String[]> getParameterMap() {
    Map<String, String[]> map = new LinkedHashMap<String, String[]>();
    for (Map.Entry<String, List<String>> entry : oDataRequest.getAllQueryParameters().entrySet()) {
      if (entry.getValue() != null) {
        map.put(entry.getKey(), entry.getValue().toArray(new String[0]));
      } else {
        map.put(entry.getKey(), new String[0]);
      }
    }
    return map;
  }

  @Override
  public String getProtocol() {
    return request.getProtocol();
  }

  @Override
  public String getScheme() {
    return request.getScheme();
  }

  @Override
  public String getServerName() {
    return request.getServerName();
  }

  @Override
  public int getServerPort() {
    return request.getServerPort();
  }

  @Override
  public BufferedReader getReader() throws IOException {
    return request.getReader();
  }

  @Override
  public String getRemoteAddr() {
    return request.getRemoteAddr();
  }

  @Override
  public String getRemoteHost() {
    return request.getRemoteHost();
  }

  @Override
  public void setAttribute(String name, Object o) {
    request.setAttribute(name, o);
  }

  @Override
  public void removeAttribute(String name) {
    request.removeAttribute(name);
  }

  @Override
  public Locale getLocale() {
    return request.getLocale();
  }

  @Override
  public Enumeration<Locale> getLocales() {
    return request.getLocales();
  }

  @Override
  public boolean isSecure() {
    return request.isSecure();
  }

  @Override
  public RequestDispatcher getRequestDispatcher(String path) {
    return request.getRequestDispatcher(path);
  }

  @Override
  public String getRealPath(String path) {
    return request.getRealPath(path);
  }

  @Override
  public int getRemotePort() {
    return request.getRemotePort();
  }

  @Override
  public String getLocalName() {
    return request.getLocalName();
  }

  @Override
  public String getLocalAddr() {
    return request.getLocalAddr();
  }

  @Override
  public int getLocalPort() {
    return request.getLocalPort();
  }


}
