package cronapp.reports.j4c.dataset;

import java.io.Serializable;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;

import cronapp.reports.commons.Geleia;
import cronapp.reports.j4c.dataset.jdbc.SqlOperatorType;
import io.zatarox.squiggle.*;
import io.zatarox.squiggle.criteria.CriteriaExpression;
import io.zatarox.squiggle.criteria.InCriteria;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.swt.widgets.Control;

import cronapp.reports.j4c.commons.J4CMatchOperator;
import io.zatarox.squiggle.criteria.BetweenCriteria;
import io.zatarox.squiggle.criteria.MatchCriteria;

/**
 * Classe de representação de uma condição do relatório.
 * <p>
 * Created by arthemus on 05/05/16.
 */
public class J4CWhereCondition implements Serializable {

  private transient final LinkedHashMap<Control, Object> betweenComponents;

  private transient J4CDataset parent;

  private J4CColumn column;

  private String operator;

  private Set<Object> values;

  private String aspasQuery = "\"";

  private Object betweenVal1;

  private Object betweenVal2;


  private J4CWhereCondition() {
    this.betweenComponents = new LinkedHashMap<>();
  }

  public J4CWhereCondition(J4CDataset parent, J4CColumn column, String operator, Object... values) {
    this();
    this.parent = parent;
    this.column = column;
    this.operator = operator;
    this.values = new LinkedHashSet<>();
    this.values.addAll(Arrays.asList(ArrayUtils.nullToEmpty(values)));
  }

  public J4CWhereCondition(J4CDataset parent, J4CColumn column, J4CMatchOperator operator, Object... values) {
    this();
    this.parent = parent;
    this.column = column;
    this.operator = operator.getOperator();
    this.values = new LinkedHashSet<>();
    this.values.addAll(Arrays.asList(ArrayUtils.nullToEmpty(values)));
  }

  public J4CDataset getParent() {
    return parent;
  }

  public J4CColumn getColumn() {
    return column;
  }

  public void setColumn(J4CColumn column) {
    this.column = column;
  }

  public String getOperator() {
    return operator;
  }

  public void setOperator(String operator) {
    this.operator = operator;
  }

  public Collection<Object> getValues() {
    if (isBetween())
      return this.betweenComponents.values();
    return this.values;
  }

  public Object[] getValuesArray() {
    if (isBetween())
      return this.betweenComponents.values().toArray();
    return ArrayUtils.nullToEmpty(this.values.toArray());
  }

  private Object getUniqueValue() {
    Object[] array = this.getValuesArray();
    if (array != null && array.length > 0)
      return array[0];
    return null;
  }

  public Object getValue() {
    Object[] array = this.getValuesArray();
    if (array != null && array.length > 0)
      return array[0];
    return null;
  }

  public void addValues(Object ... values) {
    if(isCommonComparation()) {
      this.values.clear();
      this.values.addAll(Arrays.asList(ArrayUtils.nullToEmpty(values)));
    }
  }

  public boolean isBetween() {
    return SqlOperatorType.BETWEEN.getDescription().equals(this.operator);
  }

  public boolean isIn() {
    return SqlOperatorType.IN.getDescription().equals(this.operator);
  }

  public String getAspasQuery() {
    return aspasQuery;
  }

  public void setAspasQuery(String aspasQuery) {
    this.aspasQuery = aspasQuery;
  }

  public boolean isCommonComparation() {
    return !SqlOperatorType.BETWEEN.getDescription().equals(this.operator);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (o == null || getClass() != o.getClass())
      return false;
    J4CWhereCondition that = (J4CWhereCondition) o;
    return column != null ? column.equals(that.column)
        : that.column == null &&
        (operator != null ? operator.equals(that.operator)
            : that.operator == null && (values != null ? values.equals(that.values)
            : that.values == null && (betweenComponents != null
            ? betweenComponents.equals(that.betweenComponents)
            : that.betweenComponents == null)));
  }

  @Override
  public int hashCode() {
    int result = column != null ? column.hashCode() : 0;
    result = 31 * result + (operator != null ? operator.hashCode() : 0);
    result = 31 * result + (values != null ? values.hashCode() : 0);
    result = 31 * result + (betweenComponents != null ? betweenComponents.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {

    String whereText = "";
    String columnName = aspasQuery + this.column.getParent().getName() + aspasQuery + "." + this.column.getName();
    if (isCommonComparation()) {
      whereText = columnName + " [" + this.operator + "]";
      Object[] values = this.getValuesArray();
      if (values.length > 0)
        whereText = whereText + " [" + values[0] + "]";
    } else if (isBetween()) {
      whereText = columnName + " [" + this.operator + "]";
      whereText = whereText + " [" + getBetweenVal1() + "]";
      whereText = whereText + " AND [" + getBetweenVal2() + "]";
    }
    return whereText;
  }

  public void addBetweenComponent(Control component) {
    this.betweenComponents.put(component, null);
  }

  public void addBetweenValue(Control component, Object value) {
    this.betweenComponents.put(component, value);
  }

  Criteria newCriteria() {
    J4CTable parent = this.column.getParent();
    Table table = new Table(aspasQuery + parent.getName() + aspasQuery, parent.getAs());
    Column column = table.getColumn(this.column.getName());
    if (isBetween()) {
      return new BetweenCriteria(column, CriteriaUtil.newMatch(getBetweenVal1()), CriteriaUtil.newMatch(getBetweenVal2()));
    }
    if (isIn()) {
      return getInCriteria(table);
    }
    return new MatchCriteria(column, this.operator, CriteriaUtil.newMatch(this.getUniqueValue()));

  }

  private Criteria getInCriteria(Table table) {

    String type = this.getColumn().getTypeJava();

    switch (type) {
      case "Short":
      case "Integer":
      case "Long":
        return new InCriteria(table, this.column.getName(), getvaluesToLong());


      case "Float":
      case "Double":
        return new InCriteria(table, this.column.getName(), getvaluesToDouble());

      case "Date":
      case "Time":
      case "Timestamp":
        return new InCriteria(table, this.column.getName(), getvaluesTodate());

      default:
        return new InCriteria(table, this.column.getName(), this.getUniqueValue().toString().split(","));
    }


  }

  private String[] getvaluesTodate() {
    String[] values = this.getUniqueValue().toString().split(",");
    String[] dateFormated = new String[values.length];
    try {
      for (int i = 0; i < values.length; i++) {
        String data = values[i];
        if (data.contains("-")) {
          dateFormated[i] = new SimpleDateFormat("yyyy-MM-dd").format(new SimpleDateFormat("yyyy-MM-dd").parse(data));
        } else {
          dateFormated[i] = new SimpleDateFormat("yyyy-MM-dd").format(new SimpleDateFormat("dd/MM/yyyy").parse(data));
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
    }
    return dateFormated;
  }

  private long[] getvaluesToLong() {
    long[] valuesConverted = null;
    if (this.getUniqueValue().toString().contains(",")) {
      String[] values = this.getUniqueValue().toString().split(",");
      valuesConverted = new long[values.length];
      for (int i = 0; i < values.length; i++) {
        if (StringUtils.isNumeric(values[i])) {
          valuesConverted[i] = Long.valueOf(values[i]);
        } else {
          break;
        }
      }
    } else if (StringUtils.isNumeric(this.getUniqueValue().toString())) {
      valuesConverted = new long[1];
      valuesConverted[0] = Long.valueOf(this.getUniqueValue().toString());
    }
    return valuesConverted != null ? valuesConverted : new long[1];
  }

  private double[] getvaluesToDouble() {
    double[] valuesConverted = null;
    if (this.getUniqueValue().toString().contains(",")) {
      String[] values = this.getUniqueValue().toString().split(",");
      valuesConverted = new double[values.length];
      for (int i = 0; i < values.length; i++) {
        if (StringUtils.isNumeric(values[i])) {
          valuesConverted[i] = Double.valueOf(values[i]);
        } else {
          break;
        }
      }
    } else if (StringUtils.isNumeric(this.getUniqueValue().toString())) {
      valuesConverted = new double[1];
      valuesConverted[0] = Double.valueOf(this.getUniqueValue().toString());
    }
    return valuesConverted != null ? valuesConverted : new double[1];
  }

  public Object getBetweenVal1() {
    return betweenVal1;
  }

  public void setBetweenVal1(Object betweenVal1) {
    this.betweenVal1 = betweenVal1;
  }

  public Object getBetweenVal2() {
    return betweenVal2;
  }

  public void setBetweenVal2(Object betweenVal2) {
    this.betweenVal2 = betweenVal2;
  }

}
