package cronapp.reports.j4c.export;

import static ar.com.fdvs.dj.domain.AutoText.ALIGNMENT_CENTER;
import static ar.com.fdvs.dj.domain.AutoText.AUTOTEXT_PAGE_X_SLASH_Y;
import static ar.com.fdvs.dj.domain.AutoText.POSITION_FOOTER;

import java.awt.Color;
import java.sql.Connection;
import java.util.HashMap;
import java.util.Optional;

import ar.com.fdvs.dj.core.DJConstants;
import ar.com.fdvs.dj.domain.DJCalculation;
import ar.com.fdvs.dj.domain.DJGroupLabel;
import ar.com.fdvs.dj.domain.DynamicReport;
import ar.com.fdvs.dj.domain.ImageBanner;
import ar.com.fdvs.dj.domain.Style;
import ar.com.fdvs.dj.domain.builders.ColumnBuilder;
import ar.com.fdvs.dj.domain.builders.DynamicReportBuilder;
import ar.com.fdvs.dj.domain.builders.GroupBuilder;
import ar.com.fdvs.dj.domain.chart.DJChart;
import ar.com.fdvs.dj.domain.chart.builder.AbstractChartBuilder;
import ar.com.fdvs.dj.domain.chart.builder.DJPieChartBuilder;
import ar.com.fdvs.dj.domain.constants.Border;
import ar.com.fdvs.dj.domain.constants.Font;
import ar.com.fdvs.dj.domain.constants.GroupLayout;
import ar.com.fdvs.dj.domain.constants.HorizontalAlign;
import ar.com.fdvs.dj.domain.constants.ImageScaleMode;
import ar.com.fdvs.dj.domain.constants.LabelPosition;
import ar.com.fdvs.dj.domain.constants.Page;
import ar.com.fdvs.dj.domain.constants.VerticalAlign;
import ar.com.fdvs.dj.domain.entities.columns.AbstractColumn;
import ar.com.fdvs.dj.domain.entities.columns.PropertyColumn;
import cronapp.reports.j4c.J4CBorder;
import cronapp.reports.j4c.J4CChart;
import cronapp.reports.j4c.J4CField;
import cronapp.reports.j4c.J4CFont;
import cronapp.reports.j4c.J4CImage;
import cronapp.reports.j4c.J4CObject;
import cronapp.reports.j4c.J4CPage;
import cronapp.reports.j4c.J4CText;
import cronapp.reports.j4c.charts.J4CPieChart;
import cronapp.reports.j4c.commons.J4CCalculation;
import cronapp.reports.j4c.commons.J4CChartPosition;
import cronapp.reports.j4c.commons.J4CConstants;
import cronapp.reports.j4c.commons.J4CGroupLayout;
import cronapp.reports.j4c.commons.J4CPageFormat;
import cronapp.reports.j4c.commons.J4CPosition;
import cronapp.reports.j4c.commons.J4CTextMode;
import net.sf.jasperreports.engine.JRChart;
import net.sf.jasperreports.engine.type.ScaleImageEnum;

/**
 * Exportação do relatório para ser trabalhar com o framework DynamicReports.
 *
 * Created by arthemus on 10/06/16.
 */
public class DynamicReportExport implements J4CExport<DynamicReport> {
  
  private final J4CObject j4CObject;
  private final Connection connection;
  private final HashMap<J4CField, AbstractColumn> mapColumns;
  
  private static final String OPERATOR_BETWEEN = "BETWEEN";
  
  DynamicReportExport(J4CObject j4CObject, Connection connection) {
    this.j4CObject = j4CObject;
    this.connection = connection;
    this.mapColumns = new HashMap<>();
  }
  
  @Override
  public DynamicReport get() {
    
    // Início - Builder...
    final DynamicReportBuilder dynamicReportBuilder = new DynamicReportBuilder();
    dynamicReportBuilder.setReportName(this.j4CObject.getName());
    dynamicReportBuilder.setUseFullPageWidth(true);
    dynamicReportBuilder.setWhenNoDataAllSectionNoDetail();
    
    // 1.1 - Linhas zebradas...
    dynamicReportBuilder.setPrintBackgroundOnOddRows(this.j4CObject.isConditionalLinesColor());
    
    // 1.2 - Paginação...
    if(this.j4CObject.isPagination())
      dynamicReportBuilder.addAutoText(AUTOTEXT_PAGE_X_SLASH_Y, POSITION_FOOTER, ALIGNMENT_CENTER);
    
    // 1.3 - Espaço entre linhas...
    dynamicReportBuilder.setDetailHeight(this.j4CObject.getPage().getLineSpace());
    
    // 1.4 - Margens...
    dynamicReportBuilder.setTopMargin(this.j4CObject.getPage().getTopMargin());
    dynamicReportBuilder.setBottomMargin(this.j4CObject.getPage().getBottomMargin());
    dynamicReportBuilder.setLeftMargin(this.j4CObject.getPage().getLeftMargin());
    dynamicReportBuilder.setRightMargin(this.j4CObject.getPage().getRightMargin());
    
    // 1.5 - Tamanho da página e formato...
    dynamicReportBuilder.setPageSizeAndOrientation(toPage(this.j4CObject.getPage()));
    
    // 2 - Title...
    J4CText title = this.j4CObject.getTitle();
    dynamicReportBuilder.setTitle(title.getValue(), title.isExpressionValue());
    dynamicReportBuilder.setTitleStyle(toStyle(title));
    
    // 3 - SubTitle...
    J4CText subTitle = this.j4CObject.getSubTitle();
    dynamicReportBuilder.setSubtitle(subTitle.getValue());
    dynamicReportBuilder.setSubtitleStyle(toStyle(subTitle));
    
    // 4.1 - Colunas...
    this.j4CObject.getFields().forEach(j4CColumn -> {
      
      final AbstractColumn columnDoes = ColumnBuilder.getNew()
              .setColumnProperty(j4CColumn.getText().getValue(), j4CColumn.getType())
              .setStyle(toStyle(j4CColumn.getText())).setTitle(j4CColumn.getTitle().getValue())
              .setHeaderStyle(toStyle(j4CColumn.getTitle())).build();
      
      columnDoes.setName(j4CColumn.getText().getValue());
      
      dynamicReportBuilder.addColumn(columnDoes);
      
      mapColumns.put(j4CColumn, columnDoes);
    });
    
    // 4.2 - Grupos e sumários...
    mapColumns.entrySet().forEach(entry -> {
      
      final J4CField j4CField = entry.getKey();
      final AbstractColumn abstractColumn = entry.getValue();
      
      // 5 - Grupos...
      Optional.ofNullable(j4CField.getGroup()).ifPresent(j4CGroup -> {
        
        final PropertyColumn propertyColumn = (PropertyColumn)abstractColumn;
        
        GroupBuilder groupBuilder = new GroupBuilder();
        groupBuilder.setCriteriaColumn(propertyColumn);
        groupBuilder.setGroupLayout(toGroupLayout(j4CGroup.getGroupLayout()));
        groupBuilder.setStartInNewPage(j4CGroup.isStartInNewPage());
        
        if(j4CGroup.getGroupLayout().equals(J4CGroupLayout.VALUE_IN_HEADER_WITH_HEADERS))
          dynamicReportBuilder.setPrintColumnNames(false);
        
        // 6 - Sumarios por grupo...
        Optional.ofNullable(j4CGroup.getSummary()).ifPresent(j4CSummary -> {
          
          Style valueStyle = toStyle(j4CSummary.getValue());
          Style labelStyle = toStyle(j4CSummary.getLabel());
          
          DJGroupLabel summaryLabel = new DJGroupLabel(j4CSummary.getLabel().getValue(), labelStyle,
                  toPosition(j4CSummary.getLabel().getHorizontalAlignment()));
          summaryLabel.setJasperExpression(j4CSummary.getValue().isExpressionValue());
          
          final J4CField summaryColumn = this.getJ4CColumn(j4CSummary.getColumn());
          
          groupBuilder.addFooterVariable(mapColumns.get(summaryColumn), toCalculation(j4CSummary.getCalculation()),
                  valueStyle, null, summaryLabel);
        });
        
        dynamicReportBuilder.addGroup(groupBuilder.build());
      });
      
      // 7 - Sumário global no final do relatório...
      Optional.ofNullable(j4CField.getSummary()).ifPresent(j4CSummary -> {
        Style summaryStyle = toStyle(j4CSummary.getValue());
        dynamicReportBuilder.addGlobalFooterVariable(abstractColumn, toCalculation(j4CSummary.getCalculation()),
                summaryStyle);
        dynamicReportBuilder.setGrandTotalLegend("");
      });
    });
    
    // 8 - Query SQL...
    String sql = this.j4CObject.dataset().getSql(connection);
    if(sql != null && sql.length() > 0)
      dynamicReportBuilder.setQuery(sql, DJConstants.QUERY_LANGUAGE_SQL);
    
    // 9 - Logotipo...
    J4CImage image = this.j4CObject.getImage();
    if(image != null) {
      dynamicReportBuilder.addFirstPageImageBanner(image.getPath(), image.getWidth(), image.getHeight(),
              toBannerPosition(image.getPosition()), toScale(image.getImageScale()));
      
      if(image.isRepeatEachPage())
        dynamicReportBuilder.addImageBanner(image.getPath(), image.getWidth(), image.getHeight(),
                toBannerPosition(image.getPosition()), toScale(image.getImageScale()));
    }
    
    // 10 - Gráficos...
    J4CChart chart = this.j4CObject.getChart();
    if(chart != null) {
      DJChart djChart = toChart(chart);
      dynamicReportBuilder.addChart(djChart);
    }
    
    // Final - Propriedades.
    String json = this.j4CObject.build().toJson().get();
    dynamicReportBuilder.setProperty(J4CConstants.DATASOURCE, this.j4CObject.dataset().getJndiConnection());
    dynamicReportBuilder.setProperty(J4CConstants.J4C_OBJECT, json);
    dynamicReportBuilder.setProperty(J4CConstants.ZOOM_FACTOR, "50.0");
    dynamicReportBuilder.setProperty(J4CConstants.JASPER_EDITED, Boolean.FALSE.toString());
    
    return dynamicReportBuilder.build();
  }
  
  private DJChart toChart(J4CChart j4CChart) {
    switch(j4CChart.getJRType()) {
      case JRChart.CHART_TYPE_PIE:
        return toPieChart(new DJPieChartBuilder(), (J4CPieChart)j4CChart);
      default:
        return null;
    }
  }
  
  private DJChart toPieChart(DJPieChartBuilder pieChartBuilder, J4CPieChart j4CChart) {
    this.toBaseChartProperties(pieChartBuilder, j4CChart);
    return pieChartBuilder.setCircular(true)
            .setKey((PropertyColumn)this.mapColumns.get(this.getJ4CColumn(j4CChart.getKey())))
            .addSerie(this.mapColumns.get(this.getJ4CColumn(j4CChart.getSerie()))).build();
  }
  
  private void toBaseChartProperties(AbstractChartBuilder abstractChartBuilder, J4CChart j4CChart) {
    abstractChartBuilder.setWidth(j4CChart.getWidth()).setHeight(j4CChart.getHeight())
            .setCentered(j4CChart.isCentered()).setPosition(toChartPosition(j4CChart.getPosition())).setShowLegend(true)
            .setBackColor(Color.LIGHT_GRAY).setTitlePosition(ar.com.fdvs.dj.domain.chart.DJChartOptions.EDGE_TOP)
            .setTitleColor(Color.DARK_GRAY).setTitleFont(Font.ARIAL_BIG_BOLD).setTitle("").setSubtitle("")
            .setSubtitleColor(Color.DARK_GRAY).setSubtitleFont(Font.COURIER_NEW_BIG_BOLD)
            .setLegendColor(Color.DARK_GRAY).setLegendFont(Font.COURIER_NEW_MEDIUM_BOLD)
            .setLegendBackgroundColor(Color.WHITE)
            .setLegendPosition(ar.com.fdvs.dj.domain.chart.DJChartOptions.EDGE_BOTTOM)
            .setLineStyle(ar.com.fdvs.dj.domain.chart.DJChartOptions.LINE_STYLE_DASHED).setLineWidth(1)
            .setLineColor(Color.DARK_GRAY).setPadding(5);
  }
  
  private byte toChartPosition(J4CChartPosition j4CChartPosition) {
    switch(j4CChartPosition) {
      case HEADER:
        return ar.com.fdvs.dj.domain.chart.DJChartOptions.POSITION_HEADER;
      default:
        return ar.com.fdvs.dj.domain.chart.DJChartOptions.POSITION_FOOTER;
    }
  }
  
  private Page toPage(J4CPage j4CPage) {
    switch(j4CPage.getLeaf()) {
      case A4:
        if(j4CPage.getFormat().equals(J4CPageFormat.PORTRAIT))
          return Page.Page_A4_Portrait();
        else
          return Page.Page_A4_Landscape();
      case LETTER:
        if(j4CPage.getFormat().equals(J4CPageFormat.PORTRAIT))
          return Page.Page_Letter_Portrait();
        else
          return Page.Page_Letter_Landscape();
      default:
        return Page.Page_A4_Portrait();
    }
  }
  
  private ImageScaleMode toScale(ScaleImageEnum scale) {
    switch(scale) {
      case CLIP:
        return ImageScaleMode.NO_RESIZE;
      case FILL_FRAME:
        return ImageScaleMode.FILL;
      case REAL_HEIGHT:
        return ImageScaleMode.REAL_HEIGHT;
      case REAL_SIZE:
        return ImageScaleMode.REAL_SIZE;
      default:
        return ImageScaleMode.FILL_PROPORTIONALLY;
    }
  }
  
  private byte toBannerPosition(J4CPosition position) {
    switch(position) {
      case CENTER:
        return ImageBanner.ALIGN_CENTER;
      case RIGHT:
        return ImageBanner.ALIGN_RIGHT;
      default:
        return ImageBanner.ALIGN_LEFT;
    }
  }
  
  private DJCalculation toCalculation(J4CCalculation j4CCalculation) {
    switch(j4CCalculation) {
      case AVERAGE:
        return DJCalculation.AVERAGE;
      case COUNT:
        return DJCalculation.COUNT;
      case NOTHING:
        return DJCalculation.NOTHING;
      default:
        return DJCalculation.SUM;
    }
  }
  
  private LabelPosition toPosition(J4CPosition j4CPosition) {
    switch(j4CPosition) {
      case BOTTOM:
        return LabelPosition.BOTTOM;
      case RIGHT:
        return LabelPosition.RIGHT;
      case TOP:
        return LabelPosition.TOP;
      default:
        return LabelPosition.LEFT;
    }
  }
  
  private GroupLayout toGroupLayout(J4CGroupLayout j4CGroupLayout) {
    switch(j4CGroupLayout) {
      case VALUE_IN_HEADER_AND_FOR_EACH:
        return GroupLayout.VALUE_IN_HEADER_AND_FOR_EACH;
      case VALUE_IN_HEADER_AND_FOR_EACH_WITH_HEADERS:
        return GroupLayout.VALUE_IN_HEADER_AND_FOR_EACH_WITH_HEADERS;
      case VALUE_IN_HEADER:
        return GroupLayout.VALUE_IN_HEADER;
      case VALUE_IN_HEADER_WITH_HEADERS:
        return GroupLayout.VALUE_IN_HEADER_WITH_HEADERS;
      case VALUE_IN_HEADER_WITH_HEADERS_AND_COLUMN_NAME:
        return GroupLayout.VALUE_IN_HEADER_WITH_HEADERS_AND_COLUMN_NAME;
      case VALUE_FOR_EACH:
        return GroupLayout.VALUE_FOR_EACH;
      case VALUE_FOR_EACH_WITH_HEADERS:
        return GroupLayout.VALUE_FOR_EACH_WITH_HEADERS;
      case EMPTY:
        return GroupLayout.EMPTY;
      case DEFAULT_WITH_HEADER:
        return GroupLayout.DEFAULT_WITH_HEADER;
      default:
        return GroupLayout.DEFAULT;
    }
  }
  
  private Style toStyle(J4CText j4CText) {
    Style style = new Style();
    
    J4CFont j4CFont = j4CText.getFont();
    
    Font font = new Font();
    font.setFontName(j4CFont.getName());
    font.setFontSize(j4CFont.getSize());
    font.setBold(j4CFont.isBold());
    font.setItalic(j4CFont.isItalic());
    font.setUnderline(j4CFont.isUnderline());
    
    style.setFont(font);
    style.setBackgroundColor(j4CText.getBackcolor());
    style.setTextColor(j4CText.getForecolor());
    style.setHorizontalAlign(toHorizontalAlign(j4CText.getHorizontalAlignment()));
    style.setVerticalAlign(toVerticalAlign(j4CText.getVerticalAlignment()));
    style.setTransparent(j4CText.getMode().equals(J4CTextMode.TRANSPARENT));
    style.setBorderTop(toBorder(j4CText.getBorderTop()));
    style.setBorderBottom(toBorder(j4CText.getBorderBottom()));
    style.setBorderLeft(toBorder(j4CText.getBorderLeft()));
    style.setBorderRight(toBorder(j4CText.getBorderRight()));
    style.setPattern(j4CText.getPattern());
    
    return style;
  }
  
  private Border toBorder(J4CBorder j4CBorder) {
    Border border = new Border(Border.BORDER_WIDTH_NONE);
    border.setColor(j4CBorder.getColor());
    border.setLineStyle((byte)j4CBorder.getLineStyle().ordinal());
    border.setWidth(j4CBorder.getWidth().getWidth());
    return border;
  }
  
  private VerticalAlign toVerticalAlign(J4CPosition j4CPosition) {
    switch(j4CPosition) {
      case TOP:
        return VerticalAlign.TOP;
      case BOTTOM:
        return VerticalAlign.BOTTOM;
      default:
        return VerticalAlign.MIDDLE;
    }
  }
  
  private HorizontalAlign toHorizontalAlign(J4CPosition j4CPosition) {
    switch(j4CPosition) {
      case LEFT:
        return HorizontalAlign.LEFT;
      case RIGHT:
        return HorizontalAlign.RIGHT;
      default:
        return HorizontalAlign.CENTER;
    }
  }
  
  private J4CField getJ4CColumn(String columnName) {
    final J4CField[] finalColumn = { null };
    this.j4CObject.getFields().stream().filter(column -> column.getText().getValue().equals(columnName)).findFirst()
            .ifPresent(column -> finalColumn[0] = column);
    return finalColumn[0];
  }
  
}
