package cronapp.reports;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.export.JRXlsExporter;
import net.sf.jasperreports.export.SimpleExporterInput;
import net.sf.jasperreports.export.SimpleOutputStreamExporterOutput;
import net.sf.jasperreports.export.SimpleXlsReportConfiguration;

/**
 * @author arthemus
 * @since 20/05/17
 */
public final class ReportExport {
  
  private static final Logger log = LoggerFactory.getLogger(ReportExport.class);
  
  private final String fileTarget;

  private final JasperPrint jasperPrint;

  ReportExport(String fileTarget, JasperPrint jasperPrint) {
    this.fileTarget = fileTarget;
    this.jasperPrint = jasperPrint;
  }
  
  public String getTargetFile() {
    return this.fileTarget;
  }
  
  public void exportReportToPdfFile() {
    try {
      JasperExportManager.exportReportToPdfFile(jasperPrint, fileTarget);
    } catch (JRException e) {
      log.error("Problemas durante a exportacao do relatorio para o arquivo PDF.");
      throw new RuntimeException(e);
    }
  }
  
  public byte[] toPDF() {
    exportReportToPdfFile();
    return getBytes();
  }
  
  public byte[] toXLS() {
    try {
      JRXlsExporter exporter = new JRXlsExporter();
      
      SimpleXlsReportConfiguration configuration = new SimpleXlsReportConfiguration();
      configuration.setOnePagePerSheet(false);
      configuration.setDetectCellType(true);
      configuration.setIgnoreCellBackground(true);
      configuration.setWrapText(false);
      configuration.setRemoveEmptySpaceBetweenRows(true);
      configuration.setCollapseRowSpan(true);
      configuration.setAutoFitPageHeight(true);
      configuration.setForcePageBreaks(false);
      configuration.setIgnorePageMargins(true);
      configuration.setShowGridLines(true);
      configuration.setWhitePageBackground(false);
      
      SimpleExporterInput simpleExporterInput = new SimpleExporterInput(jasperPrint);
      SimpleOutputStreamExporterOutput simpleOutputStreamExporterOutput = new SimpleOutputStreamExporterOutput(fileTarget);
      
      exporter.setConfiguration(configuration);
      exporter.setExporterInput(simpleExporterInput);
      exporter.setExporterOutput(simpleOutputStreamExporterOutput);
      
      exporter.exportReport();
    }
    catch(JRException e) {
      log.error("Problemas durante a exportacao do relatorio para o arquivo PDF.");
      throw new RuntimeException(e);
    }
    return getBytes();
  }
  
  private byte[] getBytes() {
    Path path = Paths.get(fileTarget);
    try {
      return Files.readAllBytes(path);
    }
    catch(IOException e) {
      log.error("Problemas para obter o array de bytes do relatorio.");
      throw new RuntimeException(e);
    }
  }
  
}
