package cronapp.reports;

import java.net.URL;
import java.sql.Connection;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cronapp.reports.commons.Functions;
import cronapp.reports.commons.Parameter;
import cronapp.reports.j4c.commons.J4CConstants;
import cronapp.reports.j4c.commons.J4CDatasetType;
import cronapp.reports.j4c.dataset.J4CDataset;
import net.sf.jasperreports.engine.JRException;
import net.sf.jasperreports.engine.JRExpression;
import net.sf.jasperreports.engine.JRParameter;
import net.sf.jasperreports.engine.JasperCompileManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRBeanCollectionDataSource;
import net.sf.jasperreports.engine.design.JasperDesign;

/**
 * @author arthemus
 * @since 17/08/17
 */
public class PrintDesign {

  private static final Logger log = LoggerFactory.getLogger(PrintDesign.class);

  private final JasperDesign jasperDesign;
  private final String fileTarget;
  private final List<Parameter> frontParameters;
  private final Map<String, Object> printParameters;
  private final Map<String, JRParameter> designParameters;
  private final ClassLoader loader;

  PrintDesign(JasperDesign jasperDesign, String fileTarget, List<Parameter> frontParameters) {
    this.jasperDesign = jasperDesign;
    this.fileTarget = fileTarget;
    this.frontParameters = frontParameters;
    this.printParameters = new HashMap<>();
    this.designParameters = jasperDesign.getParametersMap();
    this.loader = Thread.currentThread().getContextClassLoader();
  }
  
  public PrintDesign updateParameters() {
    frontParameters.forEach(parameter -> {
      printParameters.put(parameter.getName(), parameter.getValue());
    });

    designParameters.entrySet().stream()
      .filter(p -> !printParameters.containsKey(p.getKey()))
      .forEach(parameter -> {
        String key = parameter.getKey();
        JRParameter value = parameter.getValue();
        if(value == null)
          printParameters.put(key, null);
        else {
          JRExpression valueExpression = value.getDefaultValueExpression();
          if(valueExpression != null)
            printParameters.put(key, valueExpression.getText());
        }
      });
    return this;
  }
  
  public PrintDesign updateImages() {
    printParameters.entrySet().stream()
      .filter(parameter -> parameter.getKey().contains("image_"))
      .forEach(parameter -> {
        JRParameter jrParameter = designParameters.get(parameter.getKey());
        JRExpression defaultValueExpression = jrParameter.getDefaultValueExpression();
        if(defaultValueExpression != null) {
          URL resource = loader.getResource(defaultValueExpression.getText().replaceAll("\"", ""));
          if(resource != null)
            printParameters.put(parameter.getKey(), resource.getPath());
        }
      });
    return this;
  }
  
  public PrintDesign updateSubreports() {
    printParameters.entrySet().stream()
      .filter(parameter -> parameter.getKey().contains("sub_"))
      .forEach(parameter -> {
        JRParameter jrParameter = designParameters.get(parameter.getKey());
        JRExpression defaultValueExpression = jrParameter.getDefaultValueExpression();
        if(defaultValueExpression != null) {
          URL resource = loader.getResource(defaultValueExpression.getText().replaceAll("\"", "").replaceAll(".jrxml", ".jasper"));
          if(resource != null)
            printParameters.put(parameter.getKey(), resource.getPath());
        }
      });
    return this;
  }
  
  public ReportExport print(Connection connection) {
    JasperPrint jasperPrint;
    try {
      JasperReport jasperReport = JasperCompileManager.compileReport(jasperDesign);
      jasperPrint = JasperFillManager.fillReport(jasperReport, printParameters, connection);
    }
    catch(JRException e) {
      log.error("Problems during the compile.");
      throw new RuntimeException(e);
    }
    return new ReportExport(fileTarget, jasperPrint);
  }

  public ReportExport print(Collection<?> objectCollection) {
    JasperPrint jasperPrint;
    try {
      JasperReport jasperReport = JasperCompileManager.compileReport(jasperDesign);
      JRBeanCollectionDataSource dataSource = new JRBeanCollectionDataSource(objectCollection);
      jasperPrint = JasperFillManager.fillReport(jasperReport, printParameters, dataSource);
    }
    catch(JRException e) {
      log.error("Problems during the compile.");
      throw new RuntimeException(e);
    }
    return new ReportExport(fileTarget, jasperPrint);
  }

  public String getDatasource() {
    return jasperDesign.getProperty(J4CConstants.DATASOURCE);
  }
  
  public J4CDataset getCollectionDataset() {
    String property = jasperDesign.getProperty(J4CConstants.DATASET);
    if(Functions.isExists(property)) {
      List<ReportDataset> datasets = ReportDataset.listFromJson(property);
      ReportDataset reportDataset = ReportDataset.getMain(datasets);
      J4CDataset dataset = reportDataset.getJ4CDataset();
      if(dataset.getDatasetType().equals(J4CDatasetType.COLLECTION))
        return dataset;
    }
    return null;
  }

  public Map<String, Object> getPrintParameters() {
    return Collections.unmodifiableMap(printParameters);
  }

}
