package cronapp.reports.j4c;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import org.springframework.util.StringUtils;

import com.google.gson.Gson;

import cronapp.reports.commons.GsonSingleton;
import cronapp.reports.j4c.commons.J4CConstants;
import cronapp.reports.j4c.commons.J4CUtils;
import cronapp.reports.j4c.dataset.J4CColumn;
import cronapp.reports.j4c.dataset.J4CDataset;
import cronapp.reports.j4c.export.ReportMaker;

/**
 * J4C (Jasper 'for' Cronos)
 * <p>
 * Modelo de dados e objeto base para a manipulação de informações para a geração de um relatório.
 * <p>
 * A premissa desse novo pacote j4c é prover uma API genérica e intercambiavel para os relatórios criados com o editor
 * livre para o JasperReports ou com o módulo direcional Adhoc, permitindo a importação e exportação de informações
 * entre os módulos de forma transparente e o desenvolvimento baseado em TDD.
 * <p>
 * Created by arthemus on 07/06/16.
 */
public class J4CObject implements Serializable, Cloneable {
  
  private final String name;
  
  private J4CText title;
  
  private J4CText subTitle;
  
  private boolean conditionalLinesColor;
  
  private J4CDataset dataset;
  
  private boolean pagination;
  
  private List<J4CField> fields;
  
  private J4CImage image;
  
  private J4CPage page;
  
  private J4CChart chart;
  
  private Set<J4CParameter> parameters;
  
  private transient Map<String, Object> parametersValue;
  
  private transient J4CTemplate template;
  
  public J4CObject() {
    this("J4CReport");
  }
  
  public J4CObject(String name) {
    this.name = name;
  }
  
  public static J4CObject fromJSON(final String json) {
    Gson gson = GsonSingleton.INSTANCE.get();
    J4CObject j4CObject = gson.fromJson(json, J4CObject.class);
    j4CObject.dataset().synchronizeParent(j4CObject);
    return j4CObject;
  }
  
  public String getName() {
    return name;
  }
  
  public J4CText getTitle() {
    if(title == null)
      title = new J4CText();
    return title;
  }
  
  public void setTitle(J4CText title) {
    this.title = title;
  }
  
  public J4CText getSubTitle() {
    if(subTitle == null)
      subTitle = new J4CText();
    return subTitle;
  }
  
  public void setSubTitle(J4CText subTitle) {
    this.subTitle = subTitle;
  }
  
  public boolean isConditionalLinesColor() {
    return conditionalLinesColor;
  }
  
  public void setConditionalLinesColor(boolean conditionalLinesColor) {
    this.conditionalLinesColor = conditionalLinesColor;
  }
  
  public boolean isPagination() {
    return pagination;
  }
  
  public void setPagination(boolean pagination) {
    this.pagination = pagination;
  }
  
  public J4CImage getImage() {
    return image;
  }
  
  public void setImage(J4CImage image) {
    this.image = image;
  }
  
  public List<J4CField> getFields() {
    if(fields == null)
      this.fields = new LinkedList<>();
    return fields;
  }

  public void removeAllFields() {
    this.fields = null;
  }
  
  void addField(J4CField field) {
    this.getFields().add(field);
  }
  
  J4CObject removeField(J4CField field) {
    Iterator<J4CField> iterator = this.getFields().iterator();
    while(iterator.hasNext())
      if(iterator.next().getText().getValue().equals(field.getText().getValue()))
        iterator.remove();
    return this;
  }

  public ReportMaker build() {
    return new ReportMaker(this);
  }
  
  public J4CDataset dataset() {
    if(dataset == null)
      this.dataset = new J4CDataset(this);
    return this.dataset;
  }

  public J4CTemplate template() {
    if(template == null)
      this.template = new J4CTemplate(this);
    return this.template;
  }

  /**
   * Sincroniza os fields do relatório de acordo com as colunas escolhidas no editor de queries.
   */
  public void synchronizeFields(Connection connection) {
    J4CDataset dataset = this.dataset();

    LinkedList oldFields = new LinkedList(this.getFields());
    		
    try {
      this.removeAllFields();

      List<J4CColumn> columnRecords = dataset.getDataPreviewWithoutRecords(connection);

      int size = columnRecords.size();
      for(int index = 0; index < size; index++) {
        J4CColumn column = columnRecords.get(index);
        
        J4CField field = new J4CField();
        field.setText(new J4CText(column.getName()));
        field.setType(column.getTypeClass());
        field.setTitle(new J4CText(column.getName()));
        
        if(hasField(field))
          field.setText(new J4CText("COLUMN_".concat(String.valueOf(index + 1))));
                
        //se o campo continua existindo, retorna atributos de text, title, group e summary 
        
		if (oldFields != null && !oldFields.isEmpty()) {
			Iterator<J4CField> iteratorOld = oldFields.iterator();
			while (iteratorOld.hasNext()) {
				J4CField fieldOld = iteratorOld.next();
				
				if (fieldOld.getType().equals(field.getType()) && fieldOld.getText().getValue().equals(field.getText().getValue())) {
															
					if (fieldOld.getGroup() != null ) {
						field.setGroup(fieldOld.getGroup());
					}
					
					if (fieldOld.getSummary() != null ) {
						field.setSummary(fieldOld.getSummary());
					}
				
					J4CText j4cTextTemp = new J4CText();
					j4cTextTemp = fieldOld.getText();
					j4cTextTemp.setValue(field.getText().getValue());
					field.setText(j4cTextTemp);
					
					j4cTextTemp = new J4CText();
					j4cTextTemp = fieldOld.getTitle();
					field.setTitle(j4cTextTemp);					
				}
			}
		}
		 this.addField(field);
          	
      }
              
    }   catch(SQLException e) {
    		throw new RuntimeException(e);
    }
  }


  
  private boolean hasField(J4CField field) {
    return this.getFields().stream().anyMatch(j4CField -> j4CField.getText().getValue().equals(field.getText().getValue()));
  }
  
  public void setDataset(J4CDataset dataset) {
    this.dataset = dataset;
  }
  
  public J4CPage getPage() {
    if(page == null)
      page = new J4CPage();
    return page;
  }
  
  public void setPage(J4CPage page) {
    this.page = page;
  }
  
  public J4CChart getChart() {
    return chart;
  }
  
  public void setChart(J4CChart chart) {
    this.chart = chart;
  }
  
  public Set<J4CParameter> getParameters() {
    if(parameters == null) {
      parameters = new HashSet<>();
      parameters.add(new J4CParameter(J4CConstants.DATA_LIMIT, Integer.class.getName(), 10));
    }
    return parameters;
  }

  public void setParameters(Set<J4CParameter> parameters) {
    this.parameters = parameters;
  }

  public J4CObject addParameter(J4CParameter parameter) {
    String name = parameter.getName();
    J4CParameter j4CParameter = this.hasParameter(name);
    if(j4CParameter != null)
      this.getParameters().removeIf(getPredicateParameterExists(name));
    this.getParameters().add(parameter);
    return this;
  }

  public void removeParameter(String parameterName) {
    this.getParameters().removeIf(getPredicateParameterExists(parameterName));
  }
  
  public Map<String, Object> getParametersValue() {
    if(parametersValue == null)
      parametersValue = new HashMap<>();
    return parametersValue;
  }
  
  public void setParametersValue(Map<String, Object> parametersValue) {
    this.parametersValue = parametersValue;
  }

  public Map<String, Object> getParametersDefaultValue() {
    HashMap<String, Object> parameters = new HashMap<>();
    this.getParameters().forEach(parameter -> parameters.put(parameter.getName(), J4CUtils.defaultValueBy(parameter.getType())));
    return parameters;
  }

  /**
   * Verifica se o relatório contem um determinado parâmetro.
   * 
   * @param parameter
   *          Nome do parâmetro a ser verificado.
   * @return Caso exista, retorna o parametro.
   */
  public J4CParameter hasParameter(String parameter) {
    return this.getParameters().stream().filter(getPredicateParameterExists(parameter)).findFirst().orElse(null);
  }

  private Predicate<J4CParameter> getPredicateParameterExists(String parameter) {
    return p -> p.getName().equals(parameter);
  }

  @Override
  public boolean equals(Object o) {
    if(this == o) return true;
    if(o == null || getClass() != o.getClass()) return false;
    J4CObject object = (J4CObject) o;
    return name != null ? name.equals(object.name) : object.name == null;
  }

  @Override
  public int hashCode() {
    return name != null ? name.hashCode() : 0;
  }

  @Override
  public String toString() {
    return "J4CObject{'" + name + "'}";
  }
  
  @Override
  public J4CObject clone() {
    try {
      J4CObject clone = (J4CObject)super.clone();
      clone.title = this.getTitle().clone();
      clone.subTitle = this.getSubTitle().clone();
      clone.fields = new ArrayList<>(this.getFields());
      clone.page = this.getPage().clone();
      clone.parameters = new HashSet<>(this.getParameters());
      return clone;
    }
    catch(CloneNotSupportedException e) {
      throw new RuntimeException(e);
    }
  }

}
