package cronapp.reports.j4c.dataset;

import static java.util.stream.Collectors.toList;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.net.URLClassLoader;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.springframework.core.io.ContextResource;

import cronapp.reports.commons.AutoObserver;
import cronapp.reports.commons.Functions;
import cronapp.reports.commons.Geleia;
import cronapp.reports.j4c.J4CObject;
import cronapp.reports.j4c.J4CParameter;
import cronapp.reports.j4c.commons.J4CDatasetType;
import cronapp.reports.j4c.commons.J4CUtils;
import cronapp.reports.j4c.dataset.jdbc.JDBC;
import io.zatarox.squiggle.SelectQuery;
import io.zatarox.squiggle.Table;

/**
 * Objeto responsável por armazenar toda e qualquer informação com relação aos dados do relatório a partir de um ODBC.
 * <p>
 * Created by arthemus on 20/06/16.
 */
public class J4CDataset implements Serializable, Cloneable {
  
  private transient J4CObject parent;
  
  private transient Collection<?> collection;
  
  private J4CDatasetType datasetType;
  
  private boolean freeQuery;
  
  private String jndiConnection;
  
  private String sql;

  private J4CEntity entity;
  
  private int recordLimit;
  
  private List<J4CTable> tables;
  
  private List<J4CJoin> joins;
  
  private List<J4CColumn> columns;
  
  private List<J4CWhereCondition> wheres;

  private String persistenceUnitName;

  public J4CDataset() {
    this(new J4CObject());
  }
  
  public J4CDataset(J4CObject parent) {
    this.datasetType = J4CDatasetType.DATASOURCE;
    this.freeQuery = false;
    this.parent = parent;
  }
  
  public J4CObject getParent() {
    return parent;
  }
  
  public void synchronizeParent(J4CObject object) {
    this.parent = object;
  }
  
  public Collection<?> getCollection() {
    return collection;
  }
  
  public J4CDatasetType getDatasetType() {
    return datasetType;
  }

  public void setDatasetType(J4CDatasetType datasetType) {
    this.datasetType = datasetType;
  }

  public boolean isFreeQuery() {
    return freeQuery;
  }
  
  public void setFreeQuery(boolean freeQuery) {
    this.freeQuery = freeQuery;
  }
  
  public String getJndiConnection() {
    return jndiConnection;
  }
  
  public void setJndiConnection(String jndiConnection) {
    this.jndiConnection = jndiConnection;
  }
  
  public void setSql(String sql) {
    this.sql = sql;
  }

  public int getRecordLimit() {
    return recordLimit;
  }

  public void setRecordLimit(int recordLimit) {
    this.recordLimit = recordLimit;
  }
  
  public List<J4CTable> getTables() {
    if(tables == null)
      this.tables = new LinkedList<>();
    return this.tables;
  }
  
  public J4CDataset addTable(J4CTable table) {
    this.getTables().add(table);
    table.getColumns().forEach(this::addColumn);
    return this;
  }
  
  public void removeTable(J4CTable table) {
    Iterator<J4CTable> iterator = this.getTables().iterator();
    while(iterator.hasNext())
      if(iterator.next().getName().equals(table.getName()))
        iterator.remove();
    table.getColumns().forEach(this::removeColumn);
  }
  
  private void removeAllTables() {
    this.tables = null;
  }
  
  public List<J4CJoin> getJoins() {
    if(joins == null)
      this.joins = new LinkedList<>();
    return this.joins;
  }
  
  public void addJoin(J4CJoin join) {
    this.getJoins().add(join);
    join.getLeft().getParent().getColumns().forEach(this::addColumn);
  }
  
  public void removeJoin(J4CJoin join) {
    Iterator<J4CJoin> iterator = this.getJoins().iterator();
    while(iterator.hasNext()) {
      J4CJoin next = iterator.next();
      if(next.getLeft().equals(join.getLeft()) && next.getRight().equals(join.getRight())) {
        next.getLeft().getParent().getColumns().forEach(this::removeColumn);
        iterator.remove();
      }
    }
  }
  
  private void removeAllJoins() {
    this.joins = null;
  }
  
  public List<J4CColumn> getColumns() {
    if(columns == null)
      this.columns = new LinkedList<>();
    return this.columns;
  }
  
  public J4CDataset addColumn(J4CColumn column) {
    this.getColumns().add(column);
    return this;
  }
  
  public J4CDataset removeColumn(J4CColumn column) {
    Iterator<J4CColumn> iterator = this.getColumns().iterator();
    while(iterator.hasNext())
      if(iterator.next().getName().equals(column.getName()))
        iterator.remove();
    return this;
  }
  
  private void removeAllColumns() {
    this.columns = null;
  }
  
  public void addWhere(J4CWhereCondition whereCondition) {
    this.getWheres().add(whereCondition);
  }
  
  public void removeWhere(J4CWhereCondition j4CWhereCondition) {
    this.getWheres().removeIf(condition -> condition.equals(j4CWhereCondition));
  }
  
  private void removeAllWheres() {
    this.wheres = null;
  }
  
  public List<J4CWhereCondition> getWheres() {
    if(wheres == null)
      this.wheres = new LinkedList<>();
    return this.wheres;
  }
  
  /**
   * Define a coleção de dados utilizada pelo dataset.
   * 
   * @param collection
   *          Coleção de objetos.
   */
  public void byCollection(Collection<?> collection) {
    this.datasetType = J4CDatasetType.COLLECTION;
    this.collection = collection;
  }

  /**
   * Obtem uma nova instância do SQL Builder para trabalhar com os metadados de um banco.
   *
   * @return Classe utilitária.
   */
  public J4CSQLBuilder build(Connection connection) throws SQLException {
    return new J4CSQLBuilder(connection);
  }

  /**
   * Obtem uma nova instância do SQL Builder para trabalhar com os metadados de um banco.
   *
   * Prove o log do processamento através da classe {@link AutoObserver}.
   *
   * @param connection
   *          Conexão com o banco de dados.
   * @param autoObserver
   *          Instância da classe AutoObserver para log do processamento interno.
   * @return Nova instância da classe {@link J4CSQLBuilder}
   * @throws SQLException
   */
  public J4CSQLBuilder build(Connection connection, AutoObserver<String> autoObserver) throws SQLException {
    return new J4CSQLBuilder(connection, autoObserver);
  }
  
  private SelectQuery buildSQL(String aspasQuery) {
    SelectQuery selectQuery = new SelectQuery();
    
    // Tabelas e colunas...
    this.getColumns().forEach(j4CColumn -> {
      J4CTable j4CTable = j4CColumn.getParent();
      Table table = new Table(aspasQuery+ j4CTable.getName()+aspasQuery, j4CTable.getAs());
      selectQuery.addColumn(table, j4CColumn.getName());
    });
    
    // Joins...
    this.getJoins().forEach(join -> {
      J4CColumn left = join.getLeft();
      J4CTable leftParent = left.getParent();
      Table leftTable = new Table(aspasQuery+ leftParent.getName()+aspasQuery, leftParent.getAs());
      
      J4CColumn right = join.getRight();
      J4CTable rightParent = right.getParent();
      Table rightTable = new Table(aspasQuery + rightParent.getName()+ aspasQuery, rightParent.getAs());
      
      selectQuery.addJoin(leftTable, left.getName(), join.getOperator().getOperator(), rightTable, right.getName());
    });
    
    // Wheres...
    this.getWheres().forEach(where -> selectQuery.addCriteria(where.newCriteria()));
    
    return selectQuery;
  }
  
  /**
   * @return Uma query SQL com as informações do dataset.
   */
  public String getSql(Connection connection) {
	  
	 

	String aspasQuery = "\"";
    try {
		if("MySQL".equals(connection.getMetaData().getDatabaseProductName()))	   {
				aspasQuery = "`";
		}
	} catch (SQLException e1) {
		throw new RuntimeException(e1);
	}	  
		  
    switch(datasetType) {
      case DATASOURCE: {
        if(isFreeQuery())
          return Geleia.stringNotNull(this.sql);
        try {
          String sql = this.buildSQL(aspasQuery).toString();
          if(sql != null && sql.length() > 0 && !"SELECT".equals(sql))
            return J4CUtils.getPrepareSQLQuery(connection, sql);
        }
        catch(SQLException e) {
          throw new RuntimeException(e);
        }
      }
    }
    return Geleia.stringNotNull(this.sql);
  }

  public J4CEntity getEntity() {
    if(entity == null)
      entity = new J4CEntity();
    return entity;
  }

  public void setEntity(J4CEntity entity) {
    this.entity = entity;
  }

  /**
   * Obtem uma lista de registros com forme a consulta sql realizada.
   * 
   * @return Lista de registros do banco.
   */
  public List<Map<J4CColumn, Object>> getDataPreview(Connection connection) throws SQLException {
    LinkedList<Map<J4CColumn, Object>> list = new LinkedList<>();
    String sql = this.getSql(connection);
    if(Functions.isExists(sql)) {
      PreparedStatement preparedStatement = null;
      try {
        String[] columnNames = columns.stream().map(J4CColumn::getName).collect(Collectors.toList()).toArray(new String[0]);
        preparedStatement = connection.prepareStatement(sql.replace("$P{DATA_LIMIT}", String.valueOf(this.recordLimit)).trim(), columnNames);
        ResultSet resultSet = preparedStatement.executeQuery();
        if(resultSet != null) {
          ResultSetMetaData metaData = resultSet.getMetaData();
          while(resultSet.next()) {
            LinkedHashMap<J4CColumn, Object> record = new LinkedHashMap<>();
            int columnCount = metaData.getColumnCount();
            for(int columnIndex = 1; columnIndex <= columnCount; columnIndex++) {
              J4CTable table = new J4CTable(metaData.getTableName(columnIndex));
              String columnName = metaData.getColumnName(columnIndex);
              String columnLabel = metaData.getColumnLabel(columnIndex);
              int columnType = metaData.getColumnType(columnIndex);
              J4CColumn column = new J4CColumn(table, columnName, columnLabel, JDBC.getJavaType(columnType).getSimpleName());
              Object value = resultSet.getObject(columnIndex);
              record.put(column, value);
            }
            list.add(record);
          }
        }
      }
      finally {
        if(preparedStatement != null)
          try {
            preparedStatement.close();
          }
          catch(SQLException ignored) {
          }
      }
    }
    return list;
  }
  
  public SQLObject getSQLObject(Connection connection) {
    return new SQLObject(this.getSql(connection), this.getParent(), this.recordLimit);
  }
  
  /**
   * Obtem uma referência da consulta afim de visualizar apenas os dados das colunas, sem a necessidade de ter os dados
   * em sí.
   * 
   * @return Lista com os dados das colunas resultantes da consulta.
   * @throws SQLException
   *           Caso ocorra um erro durante a execução da Query no banco de dados.
   */
  public List<J4CColumn> getDataPreviewWithoutRecords(Connection connection) throws SQLException {
    LinkedList<J4CColumn> list = new LinkedList<>();
    String sql = this.getSql(connection);
    if(Functions.isExists(sql)) {
      PreparedStatement preparedStatement = null;
      try {
        SQLObject sqlObject = this.getSQLObject(connection);
        String[] columnNames = getColumns().stream().map(J4CColumn::getName).collect(Collectors.toList()).toArray(new String[0]);
        preparedStatement = connection.prepareStatement(sqlObject.getSQLAnsi(), columnNames);
        if(sqlObject.hasParameters()) {
          Iterator<Map.Entry<String, Object>> iterator = sqlObject.getParameters().entrySet().iterator();
          int count = 0;
          while(iterator.hasNext()) {
            Map.Entry<String, Object> entry = iterator.next();
            preparedStatement.setObject(++count, entry.getValue());
          }
        }
        ResultSet resultSet = preparedStatement.executeQuery();
        if(resultSet != null) {
          ResultSetMetaData metaData = resultSet.getMetaData();
          int columnCount = metaData.getColumnCount();
          for(int columnIndex = 1; columnIndex <= columnCount; columnIndex++) {
            String columnName = metaData.getColumnName(columnIndex);
            String columnLabel = metaData.getColumnLabel(columnIndex);
            int columnType = metaData.getColumnType(columnIndex);
            list.add(new J4CColumn(columnName, columnLabel, JDBC.getJavaType(columnType).getName()));
          }
        }
      }
      finally {
        if(preparedStatement != null)
          try {
            preparedStatement.close();
          }
          catch(SQLException ignored) {
          }
      }
    }
    return list;
  }

  public List<J4CColumn> getDataPreviewWithoutRecords(URLClassLoader classLoader) throws Exception {
    LinkedList<J4CColumn> list = new LinkedList<>();
    if(entity != null) {
      String entity = this.entity.getEntity();
      if(Functions.isExists(entity)) {
        Class<?> classReference = classLoader.loadClass(entity);
        Field[] fields = classReference.getDeclaredFields();
        for(Field field : fields) {
          String fieldName = field.getName();
          if(fieldName.equalsIgnoreCase("serialVersionUID"))
            continue;
          list.add(new J4CColumn(fieldName, fieldName, field.getType().getName()));
        }
      }
    }
    return list;
  }

  @Override
  public boolean equals(Object o) {
    if(this == o) return true;
    if(o == null || getClass() != o.getClass()) return false;
    J4CDataset that = (J4CDataset) o;
    if(parent != null ? !parent.equals(that.parent) : that.parent != null) return false;
    if(jndiConnection != null ? !jndiConnection.equals(that.jndiConnection) : that.jndiConnection != null) return false;
    if(sql != null ? !sql.equals(that.sql) : that.sql != null) return false;
    if(tables != null ? !tables.equals(that.tables) : that.tables != null) return false;
    if(joins != null ? !joins.equals(that.joins) : that.joins != null) return false;
    if(columns != null ? !columns.equals(that.columns) : that.columns != null) return false;
    return wheres != null ? wheres.equals(that.wheres) : that.wheres == null;
  }

  @Override
  public int hashCode() {
    int result = parent != null ? parent.hashCode() : 0;
    result = 31 * result + (jndiConnection != null ? jndiConnection.hashCode() : 0);
    result = 31 * result + (sql != null ? sql.hashCode() : 0);
    result = 31 * result + (tables != null ? tables.hashCode() : 0);
    result = 31 * result + (joins != null ? joins.hashCode() : 0);
    result = 31 * result + (columns != null ? columns.hashCode() : 0);
    result = 31 * result + (wheres != null ? wheres.hashCode() : 0);
    return result;
  }

  @Override
  public String toString() {
    return "J4CDataset{" + "jndiConnection='" + jndiConnection + '\'' + '}';
  }
  
  @Override
  public J4CDataset clone() {
    try {
      J4CDataset clone = (J4CDataset)super.clone();
      clone.parent = this.getParent().clone();
      clone.tables = new ArrayList<>(this.getTables());
      clone.columns = new ArrayList<>(this.getColumns());
      clone.joins = new ArrayList<>(this.getJoins());
      clone.wheres = new ArrayList<>(this.getWheres());
      return clone;
    }
    catch(CloneNotSupportedException e) {
      throw new RuntimeException(e);
    }
  }
  
  /**
   * Verifica se o dataset já tem uma tabela adicionada.
   * 
   * @return true caso o dataset já tenha tabela adicionada.
   */
  public boolean hasParentTables() {
    return this.getTables().size() > 0;
  }
  
  /**
   * Retorna as tabelas que compoem as clausulas from e join do dataset.
   * 
   * @return Lista com as tabelas já existentes no dataset.
   */
  public List<J4CTable> getParentTables() {
    HashSet<J4CTable> parentTables = new HashSet<>();
    
    List<J4CTable> tables = new ArrayList<>(this.getTables());
    tables.forEach(parentTables::add);
    
    List<J4CTable> joins = this.getJoins().stream().map(join -> join.getLeft().getParent()).collect(toList());
    joins.forEach(parentTables::add);
    
    return new ArrayList<>(parentTables);
  }
  
  /**
   * Verifica se uma determinada coluna existe dentro do dataset.
   * 
   * @param column
   *          Coluna a ser analizada.
   * @return true ou false
   */
  public boolean hasColumn(J4CColumn column) {
    return this.hasColumn(this.getColumns(), column);
  }
  
  public boolean hasColumn(Collection<J4CColumn> columnsSource, J4CColumn column) {
    return columnsSource.stream().anyMatch(j4CColumn -> j4CColumn.getName().equals(column.getName()));
  }
  
  public void onRefreshTransientObjects(Connection connection) {
    LinkedList<J4CTable> tablesUpdated = new LinkedList<>();
    
    try {
      J4CSQLBuilder builder = this.build(connection);

      List<J4CTable> tablesFromDB = builder.listTables();
      this.getTables().forEach(table -> {
        Optional<J4CTable> optional = tablesFromDB.stream()
                .filter(j4CTable -> j4CTable.getName().equals(table.getName()))
                .findFirst();
        
        if(optional.isPresent()) {
          J4CTable j4CTable = optional.get();
          try {
            builder.populateTable(j4CTable);
            tablesUpdated.add(j4CTable.clone());
          }
          catch(SQLException e) {
            throw new RuntimeException("Erro ao tentar ler as colunas do banco de dados.", e);
          }
        }
        else {
          throw new RuntimeException(String.format("Table %s not found.", table.getName()));
        }
      });
      
      // Salvando colunas previamente definidas...
      LinkedList<J4CColumn> columnsBackup = new LinkedList<>();
      columnsBackup.addAll(this.getColumns());
      
      // Salvando Joins...
      LinkedList<J4CJoin> joinsBackup = new LinkedList<>();
      joinsBackup.addAll(this.getJoins());
      
      this.removeAllTables();
      this.removeAllColumns();
      this.removeAllJoins();
      
      // Atualizando tabelas do objeto Dataset já com as respectivas colunas agregadas...
      tablesUpdated.forEach(this::addTable);
      
      // Atualizando lista de colunas de acordo com os Joins...
      joinsBackup.forEach(join -> {
        
        J4CJoin newJoin = join.clone();
        
        // Left
        J4CTable parentLeft = join.getLeft().getParent();
        Optional<J4CTable> optionalLeft = tablesFromDB.stream()
                .filter(j4CTable -> j4CTable.getName().equals(parentLeft.getName()))
                .findFirst();

        if(optionalLeft.isPresent()) {
          J4CTable table = optionalLeft.get();
          try {
            builder.populateTable(table);
            J4CColumn clone = join.getLeft().clone();
            clone.setParent(table);
            newJoin.setLeft(clone);
          }
          catch(SQLException e) {
            throw new RuntimeException("Erro ao tentar ler as colunas do banco de dados.", e);
          }
        }
        
        // Right
        J4CTable parentRight = join.getRight().getParent();
        Optional<J4CTable> optionalRight = tablesFromDB.stream()
                .filter(j4CTable -> j4CTable.getName().equals(parentRight.getName()))
                .findFirst();
        
        if(optionalRight.isPresent()) {
          J4CTable table = optionalRight.get();
          try {
            builder.populateTable(table);
            J4CColumn clone = join.getRight().clone();
            clone.setParent(table);
            newJoin.setRight(clone);
          }
          catch(SQLException e) {
            throw new RuntimeException("Erro ao tentar ler as colunas do banco de dados.", e);
          }
        }
        
        this.addJoin(newJoin);
      });
      
      // Atualiza as colunas realmente selecionadas...
      Iterator<J4CColumn> iterator = this.getColumns().iterator();
      while(iterator.hasNext()) {

        J4CColumn column = iterator.next();
        Optional<J4CColumn> optional = columnsBackup.stream()
                .filter(j4CColumn -> j4CColumn.getName().equals(column.getName()))
                .findFirst();

        if(!optional.isPresent())
          iterator.remove();
      }

      // Ordena as colunas com base no backup, com a ordem original...
      this.getColumns().sort(Comparator.comparing(columnsBackup::indexOf));
    }
    catch(SQLException e) {
      throw new RuntimeException(e);
    }
  }
  
  /**
   * Caso o dataset utilize a edição de query livre ou tenha seu datasource alterado,
   * remove-se todas as tabelas, colunas, joins e cláusulas where.
   */
  public void reset() {
    this.removeAllTables();
    this.removeAllColumns();
    this.removeAllJoins();
    this.removeAllWheres();
  }

  public String getPersistenceUnitName() {
    return persistenceUnitName;
  }

  public void setPersistenceUnitName(String persistenceUnitName) {
    this.persistenceUnitName = persistenceUnitName;
  }

  /**
   * Limpa todas as colunas do dataset e adiciona novas colunas.
   * 
   * @param columns
   *          Colunas a serem adicionadas.
   */
  public void replaceColumns(List<J4CColumn> columns) {
    this.columns.clear();
    this.columns.addAll(columns);
  }

  /**
   * Para manipulação da query antes de ser executada.
   */
  public class SQLObject {
    
    private final String sql;
    private final LinkedHashMap<String, Object> parameters;
    
    SQLObject(String sql, J4CObject object, int dataLimit) {
      
      // Ajustando o limite de dados...
      sql = sql.replace("$P{DATA_LIMIT}", String.valueOf(dataLimit)).trim();
      
      this.parameters = new LinkedHashMap<>();
      
      String delims = " \n\r\t.(),+";
      String quots = "\'";
      
      StringBuilder token = new StringBuilder();
      boolean isQuoted = false;
      
      LinkedList<String> tokens = new LinkedList<>();
      
      for(int i = 0; i < sql.length(); i++) {
        if(quots.indexOf(sql.charAt(i)) != -1) {
          isQuoted = token.length() == 0;
        }
        if(delims.indexOf(sql.charAt(i)) == -1 || isQuoted) {
          token.append(sql.charAt(i));
        }
        else {
          if(token.length() > 0) {
            tokens.add(token.toString());
            token = new StringBuilder();
            isQuoted = false;
          }
        }
      }
      
      if(token.length() > 0)
        tokens.add(token.toString());
      
      List<String> parameters = tokens.stream().filter(s -> s.startsWith("$P{")).collect(Collectors.toList());
      
      String[] newSQL = { sql };
      parameters.forEach(p -> {
        String parameterClean = p.replace("$P{", "").replace("}", "");
        J4CParameter parameter = object.hasParameter(parameterClean);
        if(parameter != null) {
          this.parameters.put(parameter.getName(), J4CUtils.defaultValueBy(parameter.getType()));
          newSQL[0] = newSQL[0].replace(p, "?");
        }
      });
      
      this.sql = newSQL[0];
    }
    
    String getSQLAnsi() {
      return this.sql;
    }
    
    public boolean hasParameters() {
      return !this.parameters.isEmpty();
    }
    
    public Map<String, Object> getParameters() {
      return this.parameters;
    }
    
  }
  
}
