package cronapp.reports.j4c.dataset.jdbc;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Map;

import javax.sql.DataSource;

public class Schema {
  
  private final Map<String, Relation> relations;
  private final Map<String, PrimaryKey> primaryKeys;
  private final Map<String, ForeignKey> foreignKeys;
  private final Map<String, Index> indexes;
  
  public Schema(Map<String, Relation> relations, Map<String, PrimaryKey> primaryKeys,
          Map<String, ForeignKey> foreignKeys, Map<String, Index> indexes) {
    this.relations = relations;
    this.primaryKeys = primaryKeys;
    this.foreignKeys = foreignKeys;
    this.indexes = indexes;
  }
  
  public Map<String, Relation> getRelations() {
    return relations;
  }
  
  public Map<String, PrimaryKey> getPrimaryKeys() {
    return primaryKeys;
  }
  
  public Map<String, ForeignKey> getForeignKeys() {
    return foreignKeys;
  }
  
  public Map<String, Index> getIndexes() {
    return indexes;
  }
  
  public static Schema create(Connection conn, String table) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return create(meta, catalog, table);
  }
  
  public static Schema create(DatabaseMetaData meta, String catalog, String table) throws SQLException {
    Map<String, Relation> relations = JDBC.fetchColumns(meta, catalog, table);
    Map<String, PrimaryKey> primaryKeys = JDBC.fetchPrimaryKeys(meta, catalog, table);
    Map<String, ForeignKey> foreignKeys = JDBC.fetchForeignKeys(meta, catalog, table);
    
    final HashMap<String, Index> indexes = new HashMap<>();
    for(String tableName : relations.keySet())
      indexes.putAll(JDBC.fetchIndexes(meta, catalog, null, tableName, true));
    
    return new Schema(relations, primaryKeys, foreignKeys, indexes);
  }
  
  public static Schema createSchema(DataSource dataSource, String table) throws SQLException {
    Schema result;
    Connection conn = dataSource.getConnection();
    try {
      result = Schema.create(conn, table);
    }
    finally {
      if(conn != null)
        conn.close();
    }
    return result;
  }
  
  @Override
  public boolean equals(Object o) {
    if(this == o)
      return true;
    if(o == null || getClass() != o.getClass())
      return false;
    Schema schema = (Schema)o;
    if(relations != null ? !relations.equals(schema.relations) : schema.relations != null)
      return false;
    if(primaryKeys != null ? !primaryKeys.equals(schema.primaryKeys) : schema.primaryKeys != null)
      return false;
    if(foreignKeys != null ? !foreignKeys.equals(schema.foreignKeys) : schema.foreignKeys != null)
      return false;
    return indexes != null ? indexes.equals(schema.indexes) : schema.indexes == null;
  }
  
  @Override
  public int hashCode() {
    int result = relations != null ? relations.hashCode() : 0;
    result = 31 * result + (primaryKeys != null ? primaryKeys.hashCode() : 0);
    result = 31 * result + (foreignKeys != null ? foreignKeys.hashCode() : 0);
    result = 31 * result + (indexes != null ? indexes.hashCode() : 0);
    return result;
  }
  
  @Override
  public String toString() {
    return "Schema{" + "relations=" + relations + ", primaryKeys=" + primaryKeys + ", foreignKeys=" + foreignKeys +
            ", indexes=" + indexes + '}';
  }
}
