package cronapp.reports.j4c.dataset;

import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import cronapp.reports.commons.AutoObserver;
import cronapp.reports.commons.Functions;
import cronapp.reports.j4c.dataset.jdbc.Column;
import cronapp.reports.j4c.dataset.jdbc.ForeignKey;
import cronapp.reports.j4c.dataset.jdbc.JDBC;
import cronapp.reports.j4c.dataset.jdbc.PrimaryKey;
import cronapp.reports.j4c.dataset.jdbc.Relation;

/**
 * Utilitário para trabalhar com os metadados de um determinado banco de dados.
 *
 * Created by arthemus on 19/07/16.
 */
public class J4CSQLBuilder {
  
  private final Connection connection;
  
  private AutoObserver<String> autoObserver;
  
  J4CSQLBuilder(Connection connection) {
    this.connection = connection;
  }
  
  J4CSQLBuilder(Connection connection, AutoObserver<String> autoObserver) {
    this.connection = connection;
    this.autoObserver = autoObserver;
  }
  
  public List<J4CTable> listTables() {
    return listTables(null);
  }

  public List<J4CTable> listTables(String schema) {
    ArrayList<J4CTable> tables = new ArrayList<>();
    try {
      String[] types = { "TABLE" };
      DatabaseMetaData metaData = connection.getMetaData();
      ResultSet rs = metaData.getTables(null, schema, null, types);
      while(rs.next()) {
        String tableName = rs.getString(3);
        if(autoObserver != null)
          autoObserver.setValue(tableName);
        tables.add(new J4CTable(tableName));
      }
    }
    catch(SQLException e) {
      throw new RuntimeException(e);
    }
    return tables.stream().sorted().collect(Collectors.toList());
  }

  /**
   * Insere em uma determinada tabela suas respectivas colunas.
   *
   * @param table
   *          Tabela a ser editada.
   * @return A mesma instância da tabela mas com as informações das colunas definidas.
   * @throws SQLException
   *           Caso ocorra algum problema com a leitura dos metadados.
   */
  public J4CTable populateTable(final J4CTable table) throws SQLException {
    String tableName = table.getName();
    if(Functions.isExists(tableName)) {
      Map<String, Relation> fetchColumns = JDBC.fetchColumns(connection, null, tableName);
      fetchColumns.forEach((key, relation) -> {
        Collection<Column> columns = relation.getColumns().values();
        List<J4CColumn> j4CColumns = columns.stream()
                .sorted((o1, o2) -> Integer.compare(o2.getOrdinalPosition(), o1.getOrdinalPosition()))
                .map(column -> new J4CColumn(table, column.getName(), column.getType()))
                .collect(Collectors.toList());
        table.setColumns(j4CColumns);
      });
    }
    return table;
  }

  public J4CColumn getForeignkey(final J4CTable from, final J4CTable join) {
    final J4CColumn[] j4CColumn = { null };
    try {
      Map<String, ForeignKey> foreignKeys = JDBC.fetchForeignKeys(this.connection, from.getName());
      Map<String, PrimaryKey> primaryKeys = JDBC.fetchPrimaryKeys(this.connection, join.getName());

      foreignKeys.entrySet().stream()
              .filter(foreignKeyEntry -> foreignKeyEntry.getKey().equals(from.getName()))
              .findAny()
              .ifPresent(foreignKey -> primaryKeys.entrySet().stream()
                      .filter(primaryKey -> primaryKey.getKey().equals(join.getName()))
                      .findAny()
                      .ifPresent(primaryKey -> j4CColumn[0] = new J4CColumn(from, foreignKey.getValue().getSource().getColumnName())));

    }
    catch(SQLException e) {
      throw new RuntimeException(e);
    }
    return j4CColumn[0];
  }

  public J4CColumn getPrimarykey(final J4CTable table) {
    final J4CColumn[] j4CColumn = { null };
    try {
      Map<String, PrimaryKey> primaryKeys = JDBC.fetchPrimaryKeys(this.connection, table.getName());
      primaryKeys.entrySet().stream()
              .filter(primaryKey -> primaryKey.getKey().equals(table.getName()))
              .findAny()
              .ifPresent(primaryKey -> j4CColumn[0] = new J4CColumn(table, primaryKey.getValue().getSource().getColumnName()));
    }
    catch(SQLException e) {
      throw new RuntimeException(e);
    }
    return j4CColumn[0];
  }

}
