package cronapp.reports;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import cronapp.reports.commons.Functions;
import cronapp.reports.commons.GsonSingleton;
import cronapp.reports.j4c.dataset.J4CDataset;

/**
 * Classe auxiliar responsavel por guardar as informações de acesso a dados definidas dentro de um relatório.
 *
 * @author arthemus
 * @since 10/07/17
 */
public class ReportDataset {
  
  private final String name;
  
  private J4CDataset j4CDataset;
  
  private boolean main;
  
  public ReportDataset() {
    this.name = "main-dataset";
    this.j4CDataset = new J4CDataset();
    this.main = true;
  }
  
  public ReportDataset(String name) {
    this.name = name;
    this.j4CDataset = new J4CDataset();
    this.main = true;
  }
  
  public ReportDataset(String name, J4CDataset j4CDataset, boolean main) {
    this.name = name;
    this.j4CDataset = j4CDataset;
    this.main = main;
  }
  
  public String getName() {
    return name;
  }
  
  public J4CDataset getJ4CDataset() {
    return j4CDataset;
  }
  
  public void setJ4CDataset(J4CDataset j4CDataset) {
    this.j4CDataset = j4CDataset;
  }
  
  public void setMain(boolean main) {
    this.main = main;
  }
  
  public boolean isMain() {
    return main;
  }
  
  /**
   * Faz o parse de um json (possivelmente salvo nas propriedades do relatório) para uma lista de objetos CJRDataset.
   * 
   * Se o json não existir ou estiver vazion, uma lista vazia será retornada.
   *
   * @param json
   *          Json a ser processado.
   * @return Lista de objetos.
   */
  public static List<ReportDataset> listFromJson(String json) {
    if(!Functions.isExists(json))
      return new ArrayList<>();
    Gson gson = GsonSingleton.INSTANCE.get();
    Type type = new TypeToken<Collection<ReportDataset>>() {}.getType();
    return gson.fromJson(json, type);
  }
  
  /**
   * Serializa uma lista de datasets para uma String json.
   *
   * @param datasets
   *          Lista a ser serializada.
   * @return Novo json com as informações da lista.
   */
  public static String toJson(List<ReportDataset> datasets) {
    if(datasets == null || datasets.isEmpty())
      return "";
    Gson gson = GsonSingleton.INSTANCE.get();
    return gson.toJson(datasets);
  }
  
  /**
   * Obtem o dataset principal dentro de uma lista de datasets.
   *
   * @param datasets
   *          Lista a ser verificada.
   * @return Dataset principal.
   */
  public static ReportDataset getMain(List<ReportDataset> datasets) {
    if(datasets.isEmpty())
      return new ReportDataset();
    return datasets.stream().filter(ReportDataset::isMain).findFirst().orElse(null);
  }
  
}
