package cronapp.reports.j4c.dataset.jdbc;

import java.lang.reflect.Field;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.Date;
import java.sql.JDBCType;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

public final class JDBC {
  
  public static Map<String, ForeignKey> fetchForeignKeys(Connection conn, String table) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return fetchForeignKeys(meta, catalog, table);
  }
  
  static Map<String, ForeignKey> fetchForeignKeys(DatabaseMetaData meta, String catalog, String table) throws SQLException {
    final HashMap<String, ForeignKey> result = new HashMap<>();
    
    try (ResultSet rs = meta.getImportedKeys(catalog, null, table)) {
      Map<String, ForeignKey> fkNameMap = new HashMap<>();
      while(rs.next()) {
        String fkName = rs.getString("FK_NAME");
        String fkTableName = rs.getString("FKTABLE_NAME");
        String fkColumnName = rs.getString("FKCOLUMN_NAME");
        
        String pkTableName = rs.getString("PKTABLE_NAME");
        String pkColumnName = rs.getString("PKCOLUMN_NAME");
        
        ForeignKey current = fkNameMap.get(fkName);
        if(current == null) {
          current = new ForeignKey(fkName, new ColumnsReference(fkTableName, fkColumnName),
                  new ColumnsReference(pkTableName, pkColumnName));
          fkNameMap.put(fkName, current);
        }
      }
      
      for(ForeignKey fk : fkNameMap.values())
        result.put(fk.getSource().getTableName(), fk);
    }
    
    return result;
  }
  
  public static Map<String, PrimaryKey> fetchPrimaryKeys(Connection conn, String table) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return fetchPrimaryKeys(meta, catalog, table);
  }
  
  static Map<String, PrimaryKey> fetchPrimaryKeys(DatabaseMetaData meta, String catalog, String table) throws SQLException {
    final HashMap<String, PrimaryKey> result = new HashMap<>();
    
    try (ResultSet rs = meta.getPrimaryKeys(catalog, null, table)) {
      PrimaryKey current = null;
      while(rs.next()) {
        String tableName = rs.getString("TABLE_NAME");
        String columnName = rs.getString("COLUMN_NAME");
        String pkName = rs.getString("PK_NAME");
        if(current == null || !tableName.equals(current.getSource().getTableName())) {
          current = new PrimaryKey(pkName, new ColumnsReference(tableName, columnName));
          result.put(tableName, current);
        }
      }
    }
    
    return result;
  }
  
  public static Set<String> fetchRelationNames(Connection conn) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return fetchRelationNames(meta, catalog);
  }
  
  private static Set<String> fetchRelationNames(DatabaseMetaData meta, String catalog) throws SQLException {
    final TreeSet<String> result = new TreeSet<>();
    String[] types = { "TABLE", "VIEW" };
    try (ResultSet rs = meta.getTables(catalog, null, null, types)) {
      while(rs.next())
        result.add(rs.getString("TABLE_NAME"));
    }
    return result;
  }
  
  public static Map<String, Relation> fetchColumns(Connection conn) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return fetchColumns(meta, catalog, null, null);
  }

  public static Map<String, Relation> fetchColumns(Connection conn, String schema) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return fetchColumns(meta, catalog, schema, null);
  }

  public static Map<String, Relation> fetchColumns(Connection conn, String schema, String table) throws SQLException {
    DatabaseMetaData meta = conn.getMetaData();
    String catalog = conn.getCatalog();
    return fetchColumns(meta, catalog, schema, table);
  }
  
  static Map<String, Relation> fetchColumns(DatabaseMetaData meta, String catalog, String table) throws SQLException {
    return fetchColumns(meta, catalog, null, table);
  }
  
  private static Map<String, Relation> fetchColumns(DatabaseMetaData meta, String catalog, String schema, String table) throws SQLException {
    final HashMap<String, Relation> result = new HashMap<>();
    
    try (ResultSet rs = meta.getColumns(catalog, schema, table, null)) {
      Relation current = null;
      while(rs.next()) {
        String tableName = rs.getString("TABLE_NAME");
        String columnName = rs.getString("COLUMN_NAME");
        int dataType = rs.getInt("DATA_TYPE");
        String rawIsNullable = rs.getString("IS_NULLABLE");
        int ordinalPosition = rs.getInt("ORDINAL_POSITION");
        
        Boolean isNullable = null;
        if("YES".equalsIgnoreCase(rawIsNullable)) {
          isNullable = true;
        }
        else if("NO".equalsIgnoreCase(rawIsNullable)) {
          isNullable = false;
        }
        
        if(current == null || !tableName.equals(current.getName())) {
          current = new Relation(tableName);
          result.put(tableName, current);
        }
        
        Column column = new Column(ordinalPosition, columnName, JDBCType.valueOf(dataType).getName(), isNullable);
        current.getColumns().put(columnName, column);
      }
    }
    
    return result;
  }
  
  public static Map<String, Index> fetchIndexes(DatabaseMetaData meta, String catalog, String schemaName,
                                                String tableName, boolean unique) throws SQLException {
    final HashMap<String, Index> result = new HashMap<>();
    
    ResultSet rs = meta.getIndexInfo(catalog, schemaName, tableName, unique, false);
    
    Index current = null;
    
    while(rs.next()) {
      
      String indexName = rs.getString("INDEX_NAME");
      String table = rs.getString("TABLE_NAME");
      String columnName = rs.getString("COLUMN_NAME");
      boolean isNonUnique = rs.getBoolean("NON_UNIQUE");
      
      if(indexName == null) {
        continue;
      }
      
      if(current == null || !indexName.equals(current.getName())) {
        current = new Index(tableName, table, columnName, !isNonUnique);
        result.put(tableName, current);
      }
    }
    
    return result;
  }
  
  public static String getSqlTypeName(int sqlType) {
    try {
      Integer val = sqlType;
      for(Field field : Types.class.getFields())
        if(val.equals(field.get(null)))
          return field.getName();
    }
    catch(IllegalAccessException e) {
      throw new RuntimeException("Could not get sqlTypeName ", e);
    }
    throw new RuntimeException("Unknown sqlType " + sqlType);
  }
  
  public static Class<?> getJavaType(int sqlType) {
    switch(sqlType) {
      case Types.CHAR:
      case Types.VARCHAR:
      case Types.LONGVARCHAR:
        return String.class;
      case Types.BINARY:
      case Types.VARBINARY:
      case Types.LONGVARBINARY:
        return byte[].class;
      case Types.BIT:
        return Boolean.class;
      case Types.TINYINT:
      case Types.SMALLINT:
        return Short.class;
      case Types.INTEGER:
        return Integer.class;
      case Types.NUMERIC:
        return Long.class;
      case Types.BIGINT:
        return Long.class;
      case Types.REAL:
        return Float.class;
      case Types.DOUBLE:
      case Types.DECIMAL:
        return Double.class;
      case Types.FLOAT:
        return Float.class;
      case Types.DATE:
        return Date.class;
      case Types.TIME:
        return Time.class;
      case Types.TIMESTAMP:
        return Timestamp.class;
      default:
        return Object.class;
    }
  }
  
  public static String getJavaType(String type) {
    return getJavaType(JDBCType.valueOf(type).getVendorTypeNumber()).getSimpleName();
  }
  
}
