package cronapp.reports.j4c.dataset;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Set;

import org.apache.commons.lang3.ArrayUtils;
import org.eclipse.swt.widgets.Control;

import cronapp.reports.j4c.commons.J4CMatchOperator;
import io.zatarox.squiggle.Column;
import io.zatarox.squiggle.Criteria;
import io.zatarox.squiggle.Table;
import io.zatarox.squiggle.criteria.BetweenCriteria;
import io.zatarox.squiggle.criteria.MatchCriteria;

/**
 * Classe de representação de uma condição do relatório.
 * 
 * Created by arthemus on 05/05/16.
 */
public class J4CWhereCondition implements Serializable {
  
  private transient final LinkedHashMap<Control, Object> betweenComponents;
  
  private transient J4CDataset parent;
  
  private J4CColumn column;
  
  private String operator;
  
  private Set<Object> values;
  
  private String aspasQuery = "\"";
  	
		  

  
  private J4CWhereCondition() {
    this.betweenComponents = new LinkedHashMap<>();
  }
  
  public J4CWhereCondition(J4CDataset parent, J4CColumn column, String operator, Object ... values) {
    this();
    this.parent = parent;
    this.column = column;
    this.operator = operator;
    this.values = new LinkedHashSet<>();
    this.values.addAll(Arrays.asList(ArrayUtils.nullToEmpty(values)));
  }
  
  public J4CWhereCondition(J4CDataset parent, J4CColumn column, J4CMatchOperator operator, Object ... values) {
    this();
    this.parent = parent;
    this.column = column;
    this.operator = operator.getOperator();
    this.values = new LinkedHashSet<>();
    this.values.addAll(Arrays.asList(ArrayUtils.nullToEmpty(values)));
  }
  
  public J4CDataset getParent() {
    return parent;
  }
  
  public J4CColumn getColumn() {
    return column;
  }
  
  public void setColumn(J4CColumn column) {
    this.column = column;
  }
  
  public String getOperator() {
    return operator;
  }
  
  public void setOperator(String operator) {
    this.operator = operator;
  }
  
  public Collection<Object> getValues() {
    if(isBetween())
      return this.betweenComponents.values();
    return this.values;
  }
  
  public Object[] getValuesArray() {
    if(isBetween())
      return this.betweenComponents.values().toArray();
    return ArrayUtils.nullToEmpty(this.values.toArray());
  }
  
  private Object getUniqueValue() {
    Object[] array = this.getValuesArray();
    if(array != null && array.length > 0)
      return array[0];
    return null;
  }
  
  public void addValues(Object ... values) {
    if(isCommonComparation()) {
      this.values.clear();
      this.values.addAll(Arrays.asList(ArrayUtils.nullToEmpty(values)));
    }
  }
  
  public boolean isBetween() {
    return "BETWEEN".equals(this.operator);
  }
  
  public String getAspasQuery() {
	    return aspasQuery;
  }
	  
  public void setAspasQuery(String aspasQuery) {
	    this.aspasQuery = aspasQuery;
  }
  
  public boolean isCommonComparation() {
    return Arrays.asList("=", "<>", ">", ">=", "<", "<=", "LIKE").contains(this.operator);
  }
  
  @Override
  public boolean equals(Object o) {
    if(this == o)
      return true;
    if(o == null || getClass() != o.getClass())
      return false;
    J4CWhereCondition that = (J4CWhereCondition)o;
    return column != null ? column.equals(that.column)
            : that.column == null &&
                    (operator != null ? operator.equals(that.operator)
                            : that.operator == null && (values != null ? values.equals(that.values)
                                    : that.values == null && (betweenComponents != null
                                            ? betweenComponents.equals(that.betweenComponents)
                                            : that.betweenComponents == null)));
  }
  
  @Override
  public int hashCode() {
    int result = column != null ? column.hashCode() : 0;
    result = 31 * result + (operator != null ? operator.hashCode() : 0);
    result = 31 * result + (values != null ? values.hashCode() : 0);
    result = 31 * result + (betweenComponents != null ? betweenComponents.hashCode() : 0);
    return result;
  }
  
  @Override
  public String toString() {
	  
    String whereText = "";
    String columnName = aspasQuery  + this.column.getParent().getName() + aspasQuery + "." + this.column.getName();
    if(isCommonComparation()) {
      whereText = columnName + " [" + this.operator + "]";
      Object[] values = this.getValuesArray();
      if(values.length > 0)
        whereText = whereText + " [" + values[0] + "]";
    }
    else if(isBetween()) {
      whereText = columnName + " [" + this.operator + "]";
      Object[] values = this.betweenComponents.values().toArray();
      if(values.length > 1) {
        whereText = whereText + " [" + values[0] + "]";
        whereText = whereText + " AND [" + values[1] + "]";
      }
    }
    return whereText;
  }
  
  public void addBetweenComponent(Control component) {
    this.betweenComponents.put(component, null);
  }
  
  public void addBetweenValue(Control component, Object value) {
    this.betweenComponents.put(component, value);
  }
  
  Criteria newCriteria() {
    J4CTable parent = this.column.getParent();
    Table table = new Table(aspasQuery + parent.getName() + aspasQuery, parent.getAs());
    Column column = table.getColumn(  this.column.getName());
    if(isBetween()) {
      Object[] objects = this.betweenComponents.values().toArray();
      return new BetweenCriteria(column, CriteriaUtil.newMatch(objects[0]), CriteriaUtil.newMatch(objects[1]));
    }
    return new MatchCriteria(column, this.operator, CriteriaUtil.newMatch(this.getUniqueValue()));
  }
  
}
