package cronapp.reports.j4c.export;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Paths;
import java.sql.Connection;
import java.util.Base64;
import java.util.Map;

import cronapp.reports.j4c.J4CObject;
import cronapp.reports.j4c.commons.J4CConstants;
import cronapp.reports.j4c.commons.J4CDatasetType;
import cronapp.reports.j4c.dataset.J4CDataset;
import net.sf.jasperreports.engine.JRDataSource;
import net.sf.jasperreports.engine.JasperCompileManager;
import net.sf.jasperreports.engine.JasperExportManager;
import net.sf.jasperreports.engine.JasperFillManager;
import net.sf.jasperreports.engine.JasperPrint;
import net.sf.jasperreports.engine.JasperReport;
import net.sf.jasperreports.engine.data.JRBeanCollectionDataSource;
import net.sf.jasperreports.engine.design.JasperDesign;
import net.sf.jasperreports.engine.xml.JRXmlLoader;

/**
 * Exportação do relatório no formato Base64.
 *
 * Created by arthemus on 21/06/16.
 */
public class Base64Export implements J4CExport<String> {
  
  private final J4CObject j4CObject;
  private final Connection connection;
  
  Base64Export(J4CObject j4CObject, Connection connection) {
    this.j4CObject = j4CObject;
    this.connection = connection;
  }
  
  @Override
  @SuppressWarnings("ResultOfMethodCallIgnored")
  public String get() throws Exception {
    File temporaryFile = null;
    try {
      try {
        temporaryFile = File.createTempFile(J4CConstants.TEMPORARY_PDF_NAME, ".pdf");
      }
      catch(IOException e) {
        throw new RuntimeException(e);
      }
      
      FileOutputStream outputStream = new FileOutputStream(temporaryFile);
      
      String xml = j4CObject.build().toJrxml(connection).get();
      InputStream contentStream = new ByteArrayInputStream(xml.getBytes(StandardCharsets.UTF_8));
      
      JasperDesign jasperDesign = JRXmlLoader.load(contentStream);
      JasperReport jasperReport = JasperCompileManager.compileReport(jasperDesign);
      
      J4CDataset dataset = this.j4CObject.dataset();
      J4CDatasetType datasetType = dataset.getDatasetType();
      
      Map<String, Object> parameterValues = this.j4CObject.getParametersValue();
      
      if(J4CDatasetType.COLLECTION.equals(datasetType)) {
        JRDataSource dataSource = new JRBeanCollectionDataSource(dataset.getCollection());
        JasperPrint jasperPrint = JasperFillManager.fillReport(jasperReport, parameterValues, dataSource);
        JasperExportManager.exportReportToPdfStream(jasperPrint, outputStream);
      }
      else if(J4CDatasetType.DATASOURCE.equals(datasetType)) {
        JasperPrint jasperPrint;
        if(connection == null) {
          jasperPrint = JasperFillManager.fillReport(jasperReport, parameterValues);
        }
        else {
          jasperPrint = JasperFillManager.fillReport(jasperReport, parameterValues, connection);
        }
        JasperExportManager.exportReportToPdfStream(jasperPrint, outputStream);
      }
      
      java.nio.file.Path path = Paths.get(temporaryFile.getAbsolutePath());
      
      return Base64.getEncoder().encodeToString(java.nio.file.Files.readAllBytes(path));
    }
    finally {
      if(temporaryFile != null && temporaryFile.exists())
        temporaryFile.delete();
    }
  }
  
}
