package cronapp.framework.authentication.sso;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import com.google.gson.JsonObject;
import cronapi.ErrorResponse;
import cronapi.RestClient;
import cronapp.framework.authentication.token.AuthenticationController;
import cronapp.framework.authentication.token.AuthenticationResponse;
import cronapp.framework.i18n.Messages;
import cronapp.framework.tenant.TenantComponent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.security.oauth2.resource.UserInfoTokenServices;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.lang.Nullable;
import org.springframework.mobile.device.DeviceResolver;
import org.springframework.mobile.device.LiteDeviceResolver;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.AuthenticationException;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import java.util.Map;

@RestController
@RequestMapping("auth")
public class SSORESTController {

  private static final Logger log = LoggerFactory.getLogger(SSORESTController.class);

  private AuthenticationController authenticationController;

  @Autowired
  private ApplicationContext applicationContext;

  @Autowired
  private UserInfoTokenServices tokenServices;

  @Autowired
  private Environment env;

  public SSORESTController(@Nullable TenantComponent tenantComponent) {
    authenticationController = new AuthenticationController(tenantComponent);
  }

  @ExceptionHandler(Throwable.class)
  @ResponseBody
  ResponseEntity<ErrorResponse> handleControllerException(HttpServletRequest req, Throwable ex) {
    log.error(ex.getMessage(), ex);
    ErrorResponse errorResponse = new ErrorResponse(HttpStatus.INTERNAL_SERVER_ERROR.value(), ex, req.getMethod());
    return new ResponseEntity<>(errorResponse, HttpStatus.INTERNAL_SERVER_ERROR);
  }

  @RequestMapping(value = "sso", method = RequestMethod.POST)
  public ResponseEntity<AuthenticationResponse> authenticationSSORequest(@RequestParam(name = "client_id") String client_id, @RequestParam(name = "client_secret") String client_secret, @RequestParam(name = "access_token") String access_token, final HttpServletRequest request) throws AuthenticationException {

    if (!env.getProperty("security.oauth2.client.clientId").equals(client_id) || !env.getProperty("security.oauth2.client.clientSecret").equals(client_secret)) {
      throw new AuthenticationServiceException(Messages.getString("AuthError", "Not Authorized"));
    }

    var authentication = tokenServices.loadAuthentication(access_token);

    var userAuthentication = (UsernamePasswordAuthenticationToken) authentication.getUserAuthentication();

    var externalUserDetails = SsoSecurityConfigurer.getCronappUserDetails(userAuthentication);

    var gson = new Gson();
    var json = (JsonObject) gson.toJsonTree(externalUserDetails);

    var mapper = new ObjectMapper();
    var details = (Map) mapper.convertValue(externalUserDetails, Map.class);

    var deviceResolver = new LiteDeviceResolver();

    var authenticationController = new AuthenticationController(null);

    RestClient.getRestClient().getRequest().setAttribute("CronappToken:SSOAccessToken", access_token);
    RestClient.getRestClient().getRequest().setAttribute("CronappToken:SSOUserDetails", details);

    return authenticationController.auth(
        externalUserDetails.getEmail(),
        "cronapp",
        deviceResolver.resolveDevice(request),
        "SSO",
        null,
        json,
        request
    );
  }

  @RequestMapping(method = RequestMethod.POST)
  public ResponseEntity<AuthenticationResponse> authenticationRequest(@RequestParam String username, String password, @RequestHeader(name = "X-AUTH-TOKEN", required = false) String token, final HttpServletRequest request) throws AuthenticationException {
    DeviceResolver deviceResolver = new LiteDeviceResolver();
    return authenticationController.auth(username, password, deviceResolver.resolveDevice(request), "local", token, null, request);
  }

  @RequestMapping(value = "refresh", method = RequestMethod.GET)
  public ResponseEntity<?> authenticationRequest(HttpServletRequest request) {
    return authenticationController.authenticationRequest(request);
  }

  @RequestMapping(value = "/signin/sso", method = RequestMethod.GET)
  @ResponseBody
  public String forwardLoginSSO(@RequestParam(name = "_ctk") String _ctk, HttpServletRequest request) {

    String forwardPage = "<!DOCTYPE html>" +
        "<html>" +
        "<head>" +
        "    <meta charset='UTF-8'>" +
        "    <meta http-equiv='Content-Security-Policy' content=\"default-src * cronapp://*; connect-src * cronapp:;script-src 'unsafe-inline';\">" +
        "</head>" +
        "<body>" +
        "    <script>" +
        "       try { " +
        "         let data = {'type': 'sso_user', '_ctk': '" + _ctk + "' }; " +
        "         let parentWindow = window.opener; " +
        "         parentWindow.postMessage(data, location.href); " +
        "       } catch(error) { " +
        "         console.error(error); " +
        "       } " +
        "       window.location.assign('" + request.getContextPath() + "/#/connected?_ctk=" + _ctk + "');  " +
        "    </script>" +
        "" +
        "</body>" +
        "</html>";
    return forwardPage;
  }
}



