package cronapp.framework.authentication.saml;

import cronapp.framework.SessionListener;
import cronapp.framework.api.EventsManager;
import cronapp.framework.authentication.token.CorsFilter;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.contrib.ssl.EasySSLProtocolSocketFactory;
import org.apache.commons.httpclient.protocol.Protocol;
import org.apache.commons.httpclient.protocol.ProtocolSocketFactory;
import org.apache.velocity.app.VelocityEngine;
import org.opensaml.saml2.metadata.provider.HTTPMetadataProvider;
import org.opensaml.saml2.metadata.provider.MetadataProvider;
import org.opensaml.xml.parse.ParserPool;
import org.opensaml.xml.parse.StaticBasicParserPool;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.boot.web.servlet.ServletListenerRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.io.ClassPathResource;
import org.springframework.mobile.device.DeviceHandlerMethodArgumentResolver;
import org.springframework.mobile.device.DeviceResolverHandlerInterceptor;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.saml.SAMLAuthenticationProvider;
import org.springframework.security.saml.SAMLBootstrap;
import org.springframework.security.saml.context.SAMLContextProvider;
import org.springframework.security.saml.context.SAMLContextProviderImpl;
import org.springframework.security.saml.key.JKSKeyManager;
import org.springframework.security.saml.log.SAMLDefaultLogger;
import org.springframework.security.saml.metadata.CachingMetadataManager;
import org.springframework.security.saml.metadata.ExtendedMetadata;
import org.springframework.security.saml.metadata.MetadataGenerator;
import org.springframework.security.saml.processor.*;
import org.springframework.security.saml.storage.EmptyStorageFactory;
import org.springframework.security.saml.util.VelocityFactory;
import org.springframework.security.saml.websso.*;
import org.springframework.web.method.support.HandlerMethodArgumentResolver;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.servlet.http.HttpSessionListener;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.*;

@Configuration
public class SamlConfigurer implements WebMvcConfigurer {
  private Timer backgroundTaskTimer;
  private MultiThreadedHttpConnectionManager multiThreadedHttpConnectionManager;

  @Value("${security.oauth2.saml.maxAuthenticationAge:#{null}}")
  private Long maxAuthenticationAge;

  @PostConstruct
  public void startup() throws GeneralSecurityException, IOException {
    this.backgroundTaskTimer = new Timer(true);
    this.multiThreadedHttpConnectionManager = new MultiThreadedHttpConnectionManager();

    System.setProperty("org.opensaml.httpclient.https.disableHostnameVerification", "true");
    Protocol.registerProtocol("https", new Protocol("https", (ProtocolSocketFactory) new EasySSLProtocolSocketFactory(), 443));
  }

  @PreDestroy
  public void shutdown() {
    this.backgroundTaskTimer.purge();
    this.backgroundTaskTimer.cancel();
    this.multiThreadedHttpConnectionManager.shutdown();
  }

  @Bean
  public FilterRegistrationBean<CorsFilter> configureFilters() {
    FilterRegistrationBean<CorsFilter> registrationBean = new FilterRegistrationBean<>();
    registrationBean.setFilter(new CorsFilter());
    registrationBean.setOrder(-1 * (Integer.MAX_VALUE - 1));
    return registrationBean;
  }

  @EventListener(ApplicationReadyEvent.class)
  public void doSomethingAfterStartup() {
    if (EventsManager.hasEvent("onSystemStarts")) {
      EventsManager.executeEventOnTransaction("onSystemStarts");
    }
  }

  @Bean
  public ServletListenerRegistrationBean<HttpSessionListener> sessionListener() {
    return new ServletListenerRegistrationBean<>(new SessionListener());
  }

  @Bean
  public DeviceResolverHandlerInterceptor deviceResolverHandlerInterceptor() {
    return new DeviceResolverHandlerInterceptor();
  }

  @Bean
  public DeviceHandlerMethodArgumentResolver deviceHandlerMethodArgumentResolver() {
    return new DeviceHandlerMethodArgumentResolver();
  }

  @Override
  public void addInterceptors(InterceptorRegistry registry) {
    registry.addInterceptor(deviceResolverHandlerInterceptor());
  }

  @Override
  public void addArgumentResolvers(List<HandlerMethodArgumentResolver> argumentResolvers) {
    argumentResolvers.add(deviceHandlerMethodArgumentResolver());
  }

  @Bean
  public MetadataGenerator metadataGenerator(SamlProperties samlProperties) {
    MetadataGenerator metadataGenerator = new MetadataGenerator();
    ExtendedMetadata extendedMetadata = new ExtendedMetadata();
    extendedMetadata.setIdpDiscoveryEnabled(true);
    metadataGenerator.setExtendedMetadata(extendedMetadata);
    metadataGenerator.setEntityId(samlProperties.getEntityId());
    return metadataGenerator;
  }

  @Bean
  @ConfigurationProperties(prefix = "security.oauth2.saml")
  SamlProperties samlProperties() {
    return new SamlProperties();
  }

  @Bean
  public SAMLContextProvider contextProvider() {
    SAMLContextProviderImpl samlContextProvider = new SAMLContextProviderImpl();
    samlContextProvider.setStorageFactory(new EmptyStorageFactory());
    return samlContextProvider;
  }

  @Bean
  public JKSKeyManager keyManager(SamlProperties samlProperties) {
    Map<String, String> passwords = new HashMap<>();
    passwords.put(samlProperties.getPrivateKey(), samlProperties.getPrivateKeyPass());
    return new JKSKeyManager(new ClassPathResource(samlProperties.getStoreFile()), samlProperties.getStorePass(), passwords, samlProperties.getPrivateKey());
  }

  @Bean
  public HTTPMetadataProvider httpMetadataProvider(ParserPool parserPool, HttpClient httpClient, SamlProperties samlProperties) throws Exception {
    HTTPMetadataProvider metadataProvider = new HTTPMetadataProvider(backgroundTaskTimer, httpClient, samlProperties.getMetadataUrl());
    metadataProvider.setParserPool(parserPool);
    return metadataProvider;
  }

  @Bean
  public CachingMetadataManager metadata(List<MetadataProvider> providers) throws Exception {
    return new CachingMetadataManager(providers);
  }

  @Bean
  public SAMLDefaultLogger samlLogger() {
    return new SAMLDefaultLogger();
  }

  @Bean
  public static SAMLBootstrap samlBootstrap() {
    return new SAMLBootstrap();
  }

  @Bean
  public WebSSOProfile webSSOprofile() {
    return new WebSSOProfileImpl();
  }

  @Bean
  public SAMLProcessorImpl processor() {
    Collection<SAMLBinding> bindings = new ArrayList<>();
    bindings.add(httpRedirectDeflateBinding());
    bindings.add(httpPostBinding());
    bindings.add(artifactBinding(parserPool(), velocityEngine()));
    bindings.add(httpSOAP11Binding());
    bindings.add(httpPAOS11Binding());
    return new SAMLProcessorImpl(bindings);
  }

  private ArtifactResolutionProfile artifactResolutionProfile() {
    final ArtifactResolutionProfileImpl artifactResolutionProfile =
        new ArtifactResolutionProfileImpl(httpClient());
    artifactResolutionProfile.setProcessor(new SAMLProcessorImpl(soapBinding()));
    return artifactResolutionProfile;
  }

  // Initialization of the velocity engine
  @Bean
  public VelocityEngine velocityEngine() {
    return VelocityFactory.getEngine();
  }

  // XML parser pool needed for OpenSAML parsing
  @Bean(initMethod = "initialize")
  public StaticBasicParserPool parserPool() {
    return new StaticBasicParserPool();
  }

  @Bean
  public HttpClient httpClient() {
    return new HttpClient(multiThreadedHttpConnectionManager);
  }

  @Bean
  public HTTPArtifactBinding artifactBinding(ParserPool parserPool, VelocityEngine velocityEngine) {
    return new HTTPArtifactBinding(parserPool, velocityEngine, artifactResolutionProfile());
  }

  @Bean
  public HTTPSOAP11Binding soapBinding() {
    return new HTTPSOAP11Binding(parserPool());
  }

  @Bean
  public HTTPPostBinding httpPostBinding() {
    return new HTTPPostBinding(parserPool(), velocityEngine());
  }

  @Bean
  public HTTPRedirectDeflateBinding httpRedirectDeflateBinding() {
    return new HTTPRedirectDeflateBinding(parserPool());
  }

  @Bean
  public HTTPSOAP11Binding httpSOAP11Binding() {
    return new HTTPSOAP11Binding(parserPool());
  }

  @Bean
  public HTTPPAOS11Binding httpPAOS11Binding() {
    return new HTTPPAOS11Binding(parserPool());
  }

  @Bean
  public SingleLogoutProfile singleLogoutProfile() {
    return new SingleLogoutProfileImpl();
  }

  @Bean
  public AuthenticationProvider samlAuthenticationProvider() {
    SAMLAuthenticationProvider samlAuthenticationProvider = new SAMLAuthenticationProvider();
    samlAuthenticationProvider.setUserDetails(new SamlUserDetailsService());
    samlAuthenticationProvider.setForcePrincipalAsString(false);
    return samlAuthenticationProvider;
  }

  @Bean
  WebSSOProfileConsumerImpl webSSOprofileConsumer() {
    var webSSOProfileConsumer = new WebSSOProfileConsumerImpl();
    if (maxAuthenticationAge != null) {
      webSSOProfileConsumer.setMaxAuthenticationAge(maxAuthenticationAge);
    }
    return webSSOProfileConsumer;
  }

  @Bean
  public WebSSOProfileConsumerHoKImpl hokWebSSOprofileConsumer() {
    return new WebSSOProfileConsumerHoKImpl();
  }

  // SAML 2.0 Holder-of-Key Web SSO profile
  @Bean
  public WebSSOProfileConsumerHoKImpl hokWebSSOProfile() {
    return new WebSSOProfileConsumerHoKImpl();
  }

  // SAML 2.0 ECP profile
  @Bean
  public WebSSOProfileECPImpl ecpprofile() {
    return new WebSSOProfileECPImpl();
  }
}
