package cronapp.framework.authentication.sso;

import com.google.gson.Gson;
import cronapp.framework.api.ApiManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.security.oauth2.resource.AuthoritiesExtractor;
import org.springframework.boot.autoconfigure.security.oauth2.resource.PrincipalExtractor;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

public class CustomAuthoritiesExtractor implements AuthoritiesExtractor {

    private static final Logger logger = LoggerFactory.getLogger(CustomAuthoritiesExtractor.class);

    private static final String CLAIM_AUTHORITIES = "https://www.cronapp.io/authorities";

    private static final String[] AUTHORITY_KEYS = { "authority", "role", "value" };

    private PrincipalExtractor principalExtractor;

    public static void main(String[] args) {
      Map<String, Object> rawAuthorities = new HashMap<>();
      rawAuthorities.put("sub", "9202d3da-0403-44c5-b947-aed909b0b41c");
      rawAuthorities.put("email_verified", false);
      rawAuthorities.put("name", "Teste Teste");
      rawAuthorities.put("preferred_username", "teste");
      rawAuthorities.put("given_name", "Teste");
      rawAuthorities.put("family_name", "Teste");
      new CustomAuthoritiesExtractor(new CustomPrincipalExtractor()).extractAuthorities(rawAuthorities);
    }

    CustomAuthoritiesExtractor(final PrincipalExtractor principalExtractor) {
        this.principalExtractor = principalExtractor;
    }

    @Override
    public List<GrantedAuthority> extractAuthorities(Map<String, Object> map) {
        logger.info("Extracting authorities from values: " + new Gson().toJson(map));

        String authorities = "ROLE_USER";
        if (map.containsKey(CLAIM_AUTHORITIES)) {
            authorities = asAuthorities(map.get(CLAIM_AUTHORITIES));
        }
        List<GrantedAuthority> grantedAuthorities = AuthorityUtils.commaSeparatedStringToAuthorityList(authorities);
        String userName = (String) principalExtractor.extractPrincipal(map);
        return getMappedAuthorities(userName, grantedAuthorities);
    }

    private String asAuthorities(Object object) {
        List<Object> authorities = new ArrayList<>();
        if (object instanceof Collection) {
            Collection<?> collection = (Collection<?>) object;
            object = collection.toArray(new Object[0]);
        }
        if (ObjectUtils.isArray(object)) {
            Object[] array = (Object[]) object;
            for (Object value : array) {
                if (value instanceof String) {
                    authorities.add(value);
                }
                else if (value instanceof Map) {
                    authorities.add(asAuthority((Map<?, ?>) value));
                }
                else {
                    authorities.add(value);
                }
            }
            return StringUtils.collectionToCommaDelimitedString(authorities);
        }
        return object.toString();
    }

    private Object asAuthority(Map<?, ?> map) {
        if (map.size() == 1) {
            return map.values().iterator().next();
        }
        for (String key : AUTHORITY_KEYS) {
            if (map.containsKey(key)) {
                return map.get(key);
            }
        }
        return map;
    }

    private List<GrantedAuthority> getMappedAuthorities(String userName, List<GrantedAuthority> grantedAuthorities) {
        Set<String> securables = grantedAuthorities.stream()
                .flatMap(authority -> ApiManager.getRoleSecurables(authority.getAuthority()).stream())
                .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
                .collect(Collectors.toSet());

        securables.addAll(ApiManager.getUserSecurables(userName).stream()
                .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
                .collect(Collectors.toList()));

        securables.addAll(ApiManager.getAuthenticatedSecurables().stream()
                .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
                .collect(Collectors.toList()));

        securables.addAll(ApiManager.getPublicSecurables().stream()
                .map(securable -> securable.getStringField(ApiManager.SECURABLE_ATTRIBUTE_NAME))
                .collect(Collectors.toList()));

        return securables.stream().distinct()
                .sorted()
                .map(SimpleGrantedAuthority::new)
                .collect(Collectors.toList());
    }
}