package cronapp.framework.authentication;

import cronapi.Var;
import cronapi.database.DatabaseQueryManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import java.util.HashSet;
import java.util.LinkedList;

@Component
public class AuthenticationConfigurer implements AuthenticationProvider {

    private static final Logger log = LoggerFactory.getLogger(AuthenticationConfigurer.class);

    private final PasswordEncoder passwordEncoder = new BCryptPasswordEncoder();

    private final HttpServletRequest request;

    @Autowired
    public AuthenticationConfigurer(HttpServletRequest request) {
        this.request = request;
    }

    @Override
    @SuppressWarnings("unchecked")
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        String name = authentication.getName();
        String rawPassword = authentication.getCredentials().toString();
        try {
            DatabaseQueryManager userManager = new DatabaseQueryManager("auth");
            Var userResult = userManager.get(name);
            if (userResult.size() > 0) {
                Var resultObject = userResult.get(0);
                Var password = resultObject.getField("password");
                String encodedPassword = password.getObjectAsString();
                if (passwordEncoder.matches(rawPassword, encodedPassword)) {
                    DatabaseQueryManager roleManager = new DatabaseQueryManager("roles");
                    Var roleResult = roleManager.get(name);
                    if (roleResult.size() > 0) {
                        Var roleObject = roleResult.get(0);
                        LinkedList<Var> roles = roleObject.getObjectAsList();

                        HashSet<GrantedAuthority> authorities = new HashSet<>();
                        for (Var entityVar : roles) {
                            Var roleVar = entityVar.getField("role");
                            Var role = roleVar.getField("name");
                            String roleString = role.getObjectAsString();

                            GrantedAuthority grantedAuthority = new SimpleGrantedAuthority(roleString);
                            authorities.add(grantedAuthority);
                        }

                        User userDetails = new User(name, encodedPassword, false, false, false, false, authorities);
                        UsernamePasswordAuthenticationToken userToken = new UsernamePasswordAuthenticationToken(userDetails, encodedPassword, authorities);
                        userToken.setDetails(userDetails);

                        Var theme = resultObject.getField("theme");
                        HttpSession session = request.getSession();
                        session.setAttribute("theme", theme.getObject() == null ? "" : theme.getObject());

                        return userToken;
                    }
                }
            } else {
                log.error("Usuário não encontrado!");
                throw new UsernameNotFoundException("Usuário não encontrado!");
            }
        } catch (Exception e) {
            log.error(e.getMessage());
            throw new AuthenticationServiceException("Erro durante o processo de autenticação:\n".concat(e.getMessage()));
        }

        log.error("Usuário ou senha incorretos!");
        throw new BadCredentialsException("Usuário ou senha incorreta!");
    }

    @Override
    public boolean supports(Class<?> authentication) {
        return authentication.equals(UsernamePasswordAuthenticationToken.class);
    }

}
