package cronapi.workflow.support;

import org.drools.core.impl.EnvironmentFactory;
import org.drools.core.marshalling.impl.ClassObjectMarshallingStrategyAcceptor;
import org.drools.core.marshalling.impl.SerializablePlaceholderResolverStrategy;
import org.drools.persistence.api.TransactionManager;
import org.drools.persistence.jpa.marshaller.JPAPlaceholderResolverStrategy;
import org.kie.api.KieServices;
import org.kie.api.io.KieResources;
import org.kie.api.io.Resource;
import org.kie.api.io.ResourceType;
import org.kie.api.marshalling.ObjectMarshallingStrategy;
import org.kie.api.runtime.Environment;
import org.kie.api.runtime.EnvironmentName;
import org.kie.api.runtime.manager.*;
import org.kie.internal.runtime.manager.context.EmptyContext;
import org.kie.spring.persistence.KieSpringJpaManager;
import org.kie.spring.persistence.KieSpringTaskJpaManager;
import org.kie.spring.persistence.KieSpringTransactionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.orm.jpa.JpaTransactionManager;

import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

public final class Utils {

  private static final Logger log = LoggerFactory.getLogger(Utils.class);

  private static ThreadLocal<EntityManagerFactory> ENTITY_MANAGER_FACTORY = new ThreadLocal<>();

  public static RuntimeManager getRuntimeManager() {
    return RuntimeManagerFactory.Factory.get().newSingletonRuntimeManager(getRuntimeEnvironment());
  }

  public static RuntimeEngine getRuntimeEngine(RuntimeManager manager) {
    return manager.getRuntimeEngine(EmptyContext.get());
  }

  private static RuntimeEnvironment getRuntimeEnvironment() {
    EntityManagerFactory entityManagerFactory = Utils.getEntityManagerFactory();
    KieSpringTransactionManager transactionManager = new KieSpringTransactionManager(new JpaTransactionManager(entityManagerFactory));

    Environment env = getEnvironment(entityManagerFactory, transactionManager);

    KieSpringJpaManager jpaManager = new KieSpringJpaManager(env);
    KieSpringTaskJpaManager taskJpaManager = new KieSpringTaskJpaManager(env);

    RuntimeEnvironmentBuilder environmentBuilder = RuntimeEnvironmentBuilder.Factory.get().newEmptyBuilder()
        .entityManagerFactory(entityManagerFactory)
        .addEnvironmentEntry(EnvironmentName.TRANSACTION_MANAGER, transactionManager)
        .addEnvironmentEntry(EnvironmentName.PERSISTENCE_CONTEXT_MANAGER, jpaManager)
        .addEnvironmentEntry(EnvironmentName.TASK_PERSISTENCE_CONTEXT_MANAGER, taskJpaManager);

    Utils.getAllBpmnFilesAsResource().forEach(asset -> environmentBuilder.addAsset(asset, ResourceType.BPMN2));

    return environmentBuilder.get();
  }

  private static Environment getEnvironment(EntityManagerFactory entityManagerFactory, TransactionManager transactionManager) {
    Environment environment = EnvironmentFactory.newEnvironment();
    environment.set(EnvironmentName.TRANSACTION_MANAGER, transactionManager);
    environment.set(EnvironmentName.ENTITY_MANAGER_FACTORY, entityManagerFactory);
    environment.set(Constants.IS_JTA_TRANSACTION, false);
    environment.set(EnvironmentName.OBJECT_MARSHALLING_STRATEGIES,
        new ObjectMarshallingStrategy[]{
            new JPAPlaceholderResolverStrategy(entityManagerFactory),
            new SerializablePlaceholderResolverStrategy(ClassObjectMarshallingStrategyAcceptor.DEFAULT)
        }
    );

    return environment;
  }

  private static List<Resource> getAllBpmnFilesAsResource() {
    Path resources = Paths.get("src/main/resources");
    List<Resource> bpmnResources = new ArrayList<>();

    try {
      Files.walkFileTree(resources, new BPMNFileAsResourceFilesVisitor(bpmnResources));
    } catch (IOException e) {
      log.error("Error obtaining bpmn files", e);
    }
    return bpmnResources;
  }

  private static EntityManagerFactory getEntityManagerFactory() {
    EntityManagerFactory factory = ENTITY_MANAGER_FACTORY.get();

    if (factory == null) {
      factory = Persistence.createEntityManagerFactory(Constants.PERSISTENCE_UNIT_NAME);
      ENTITY_MANAGER_FACTORY.set(factory);
    }

    return factory;
  }

  private static class BPMNFileAsResourceFilesVisitor extends SimpleFileVisitor<Path> {

    private KieResources resources;
    private final List<Resource> bpmnResources;

    public BPMNFileAsResourceFilesVisitor(final List<Resource> bpmnResources) {
      Objects.requireNonNull(bpmnResources);
      this.bpmnResources = bpmnResources;
      this.resources = KieServices.Factory.get().getResources();
    }

    @Override
    public FileVisitResult visitFile(Path path, BasicFileAttributes attrs) {
      File visitedFile = path.toFile();

      if (visitedFile.isFile() && ResourceType.BPMN2.matchesExtension(visitedFile.getName())) {
        bpmnResources.add(resources.newFileSystemResource(visitedFile));
      }

      return FileVisitResult.CONTINUE;
    }
  }
}