package cronapi.workflow;

import cronapi.CronapiMetaData;
import org.camunda.bpm.engine.AuthorizationException;
import org.camunda.bpm.engine.BadUserRequestException;
import org.camunda.bpm.engine.FormService;
import org.camunda.bpm.engine.authorization.Permissions;
import org.camunda.bpm.engine.authorization.Resources;
import org.camunda.bpm.engine.exception.NotFoundException;
import org.camunda.bpm.engine.form.StartFormData;
import org.camunda.bpm.engine.form.TaskFormData;
import org.camunda.bpm.engine.impl.context.Context;
import org.camunda.bpm.engine.runtime.ProcessInstance;
import org.camunda.bpm.engine.runtime.ProcessInstanceQuery;
import org.camunda.bpm.engine.task.Task;
import org.camunda.bpm.engine.task.TaskQuery;
import org.camunda.bpm.engine.variable.VariableMap;

import java.io.InputStream;
import java.util.Collection;
import java.util.Map;

@CronapiMetaData(categoryName = "Workflow Form Operations",
        categoryTags = {"Workflow", "Form", "Operations"})
public class WorkflowFormOperations {

    private static FormService getFormService() {
        return Context.getProcessEngineConfiguration().getFormService();
    }

    /**
     * Retrieves all data necessary for rendering a form to start a new process instance.
     * This can be used to perform rendering of the forms outside of the process engine.
     *
     * @param processDefinitionId
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getStartFormData}}",
            description = "{{getStartFormDataDescription}}"
    )
    public static StartFormData getStartFormData(String processDefinitionId) {
        return getFormService().getStartFormData(processDefinitionId);
    }

    /**
     * Rendered form generated by the default build-in form engine for starting a new process instance.
     *
     * @param processDefinitionId
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getRenderedStartForm}}",
            description = "{{getRenderedStartFormDescription}}"
    )
    public static Object getRenderedStartForm(String processDefinitionId) {
        return getFormService().getRenderedStartForm(processDefinitionId);
    }

    /**
     * Rendered form generated by the given build-in form engine for starting a new process instance.
     *
     * @param processDefinitionId
     * @param formEngineName
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getRenderedStartForm}}",
            description = "{{getRenderedStartFormDescription}}"
    )
    public static Object getRenderedStartForm(String processDefinitionId, String formEngineName) {
        return getFormService().getRenderedStartForm(processDefinitionId, formEngineName);
    }

    /**
     * @param processDefinitionId
     * @param properties
     * @deprecated use {@link #submitStartForm(String, Map)}
     */
    @CronapiMetaData(
            name = "{{submitStartFormData}}",
            description = "{{submitStartFormDataDescription}}"
    )
    public static ProcessInstance submitStartFormData(String processDefinitionId, Map<String, String> properties) {
        return getFormService().submitStartFormData(processDefinitionId, properties);
    }

    /**
     * Start a new process instance with the user data that was entered as properties in a start form.
     *
     * @param processDefinitionId
     * @param properties
     * @throws AuthorizationException If the user has no {@link Permissions#CREATE} permission on {@link Resources#PROCESS_INSTANCE}
     *                                and no {@link Permissions#CREATE_INSTANCE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{submitStartForm}}",
            description = "{{submitStartFormDescription}}"
    )
    public static ProcessInstance submitStartForm(String processDefinitionId, Map<String, Object> properties) {
        return getFormService().submitStartForm(processDefinitionId, properties);
    }

    /**
     * @param processDefinitionId
     * @param businessKey
     * @param properties
     * @deprecated use {@link #submitStartForm(String, String, Map)}
     */
    @CronapiMetaData(
            name = "{{submitStartFormData}}",
            description = "{{submitStartFormDataDescription}}"
    )
    public static ProcessInstance submitStartFormData(String processDefinitionId, String businessKey, Map<String, String> properties) {
        return getFormService().submitStartFormData(processDefinitionId, businessKey, properties);
    }

    /**
     * Start a new process instance with the user data that was entered as properties in a start form.
     * <p>
     * A business key can be provided to associate the process instance with a
     * certain identifier that has a clear business meaning. For example in an
     * order process, the business key could be an order id. This business key can
     * then be used to easily look up that process instance , see
     * {@link ProcessInstanceQuery#processInstanceBusinessKey(String)}. Providing such a business
     * key is definitely a best practice.
     * <p>
     * Note that a business key MUST be unique for the given process definition.
     * Process instance from different process definition are allowed to have the
     * same business key.
     *
     * @param processDefinitionId the id of the process definition, cannot be null.
     * @param businessKey         a key that uniquely identifies the process instance in the context or the
     *                            given process definition.
     * @param properties          the properties to pass, can be null.
     * @throws AuthorizationException If the user has no {@link Permissions#CREATE} permission on {@link Resources#PROCESS_INSTANCE}
     *                                and no {@link Permissions#CREATE_INSTANCE} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{submitStartForm}}",
            description = "{{submitStartFormDescription}}"
    )
    public static ProcessInstance submitStartForm(String processDefinitionId, String businessKey, Map<String, Object> properties) {
        return getFormService().submitStartForm(processDefinitionId, businessKey, properties);
    }

    /**
     * Retrieves all data necessary for rendering a form to complete a task.
     * This can be used to perform rendering of the forms outside of the process engine.
     *
     * @param taskId
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getTaskFormData}}",
            description = "{{getTaskFormDataDescription}}"
    )
    public static TaskFormData getTaskFormData(String taskId) {
        return getFormService().getTaskFormData(taskId);
    }

    /**
     * Rendered form generated by the default build-in form engine for completing a task.
     *
     * @param taskId
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getRenderedTaskForm}}",
            description = "{{getRenderedTaskFormDescription}}"
    )
    public static Object getRenderedTaskForm(String taskId) {
        return getFormService().getRenderedTaskForm(taskId);
    }

    /**
     * Rendered form generated by the given build-in form engine for completing a task.
     *
     * @param taskId
     * @param formEngineName
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getRenderedTaskForm}}",
            description = "{{getRenderedTaskFormDescription}}"
    )
    public static Object getRenderedTaskForm(String taskId, String formEngineName) {
        return getFormService().getRenderedTaskForm(taskId, formEngineName);
    }

    /**
     * @param taskId
     * @param properties
     * @deprecated use {@link #submitTaskForm(String, Map)}
     */
    @CronapiMetaData(
            name = "{{submitTaskFormData}}",
            description = "{{submitTaskFormDataDescription}}"
    )
    public static void submitTaskFormData(String taskId, Map<String, String> properties) {
        getFormService().submitTaskFormData(taskId, properties);
    }

    /**
     * Completes a task with the user data that was entered as properties in a task form.
     *
     * @param taskId
     * @param properties
     * @throws AuthorizationException If the user has no {@link Permissions#UPDATE} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#UPDATE_TASK} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{submitTaskForm}}",
            description = "{{submitTaskFormDescription}}"
    )
    public static void submitTaskForm(String taskId, Map<String, Object> properties) {
        getFormService().submitTaskForm(taskId, properties);
    }

    /**
     * Retrieves a list of all variables for rendering a start from. The method takes into account
     * FormData specified for the start event. This allows defining default values for form fields.
     *
     * @param processDefinitionId the id of the process definition for which the start form should be retrieved.
     * @return a map of VariableInstances.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getStartFormVariables}}",
            description = "{{getStartFormVariablesDescription}}"
    )
    public static VariableMap getStartFormVariables(String processDefinitionId) {
        return getFormService().getStartFormVariables(processDefinitionId);
    }

    /**
     * Retrieves a list of requested variables for rendering a start from. The method takes into account
     * FormData specified for the start event. This allows defining default values for form fields.
     *
     * @param processDefinitionId     the id of the process definition for which the start form should be retrieved.
     * @param formVariables           a Collection of the names of the variables to retrieve. Allows restricting the set of retrieved variables.
     * @param deserializeObjectValues if false object values are not deserialized
     * @return a map of VariableInstances.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getStartFormVariables}}",
            description = "{{getStartFormVariablesDescription}}"
    )
    public static VariableMap getStartFormVariables(String processDefinitionId, Collection<String> formVariables, boolean deserializeObjectValues) {
        return getFormService().getStartFormVariables(processDefinitionId, formVariables, deserializeObjectValues);
    }

    /**
     * <p>Retrieves a list of all variables for rendering a task form. In addition to the task variables and process variables,
     * the method takes into account FormData specified for the task. This allows defining default values for form fields.</p>
     *
     * <p>A variable is resolved in the following order:
     * <ul>
     * <li>First, the method collects all form fields and creates variable instances for the form fields.</li>
     * <li>Next, the task variables are collected.</li>
     * <li>Next, process variables from the parent scopes of the task are collected, until the process instance scope is reached.</li>
     * </ul>
     * </p>
     *
     * @param taskId the id of the task for which the variables should be retrieved.
     * @return a map of VariableInstances.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getTaskFormVariables}}",
            description = "{{getTaskFormVariablesDescription}}"
    )
    public static VariableMap getTaskFormVariables(String taskId) {
        return getFormService().getTaskFormVariables(taskId);
    }

    /**
     * <p>Retrieves a list of requested variables for rendering a task form. In addition to the task variables and process variables,
     * the method takes into account FormData specified for the task. This allows defining default values for form fields.</p>
     *
     * <p>A variable is resolved in the following order:
     * <ul>
     * <li>First, the method collects all form fields and creates variable instances for the form fields.</li>
     * <li>Next, the task variables are collected.</li>
     * <li>Next, process variables from the parent scopes of the task are collected, until the process instance scope is reached.</li>
     * </ul>
     * </p>
     *
     * @param taskId                  the id of the task for which the variables should be retrieved.
     * @param formVariables           a Collection of the names of the variables to retrieve. Allows restricting the set of retrieved variables.
     * @param deserializeObjectValues if false object values are not deserialized
     * @return a map of VariableInstances.
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}
     *                                or no {@link Permissions#READ_TASK} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getTaskFormVariables}}",
            description = "{{getTaskFormVariablesDescription}}"
    )
    public static VariableMap getTaskFormVariables(String taskId, Collection<String> formVariables, boolean deserializeObjectValues) {
        return getFormService().getTaskFormVariables(taskId, formVariables, deserializeObjectValues);
    }

    /**
     * Retrieves a user defined reference to a start form.
     * <p>
     * In the Explorer app, it is assumed that the form key specifies a resource
     * in the deployment, which is the template for the form.  But users are free
     * to use this property differently.
     *
     * @param processDefinitionId
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getStartFormKey}}",
            description = "{{getStartFormKeyDescription}}"
    )
    public static String getStartFormKey(String processDefinitionId) {
        return getFormService().getStartFormKey(processDefinitionId);
    }

    /**
     * Retrieves a user defined reference to a task form.
     * <p>
     * In the Explorer app, it is assumed that the form key specifies a resource
     * in the deployment, which is the template for the form.  But users are free
     * to use this property differently.
     * <p>
     * Both arguments can be obtained from {@link Task} instances returned by any
     * {@link TaskQuery}.
     *
     * @param processDefinitionId
     * @param taskDefinitionKey
     * @throws AuthorizationException If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{getTaskFormKey}}",
            description = "{{getTaskFormKeyDescription}}"
    )
    public static String getTaskFormKey(String processDefinitionId, String taskDefinitionKey) {
        return getFormService().getTaskFormKey(processDefinitionId, taskDefinitionKey);
    }

    /**
     * Retrieves a deployed start form for a process definition with a given id.
     *
     * @param processDefinitionId
     * @throws AuthorizationException  If the user has no {@link Permissions#READ} permission on {@link Resources#PROCESS_DEFINITION}.
     * @throws NotFoundException       If the start form cannot be found.
     * @throws BadUserRequestException If the start form key has wrong format ("embedded:deployment:<path>" or "deployment:<path>" required).
     */
    @CronapiMetaData(
            name = "{{getDeployedStartForm}}",
            description = "{{getDeployedStartFormDescription}}"
    )
    public static InputStream getDeployedStartForm(String processDefinitionId) {
        return getFormService().getDeployedStartForm(processDefinitionId);
    }

    /**
     * Retrieves a deployed task form for a task with a given id.
     *
     * @param taskId
     * @throws AuthorizationException  If the user has no {@link Permissions#READ} permission on {@link Resources#TASK}.
     * @throws NotFoundException       If the task form cannot be found.
     * @throws BadUserRequestException If the task form key has wrong format ("embedded:deployment:<path>" or "deployment:<path>" required).
     */
    @CronapiMetaData(
            name = "{{getDeployedTaskForm}}",
            description = "{{getDeployedTaskFormDescription}}"
    )
    public static InputStream getDeployedTaskForm(String taskId) {
        return getFormService().getDeployedTaskForm(taskId);
    }
}
