package cronapi;

import cronapi.plugin.CronAuthorizationConfiguration;
import cronapi.plugin.FilterPlugin;
import cronapi.support.Constants;
import io.cronapp.bpm.identity.plugin.CronIdentityProviderPlugin;
import io.cronapp.bpm.identity.plugin.filter.CronAuthenticationTokenFilter;
import io.cronapp.bpm.identity.plugin.filter.CronRestAuthenticationFilter;
import io.cronapp.bpm.identity.plugin.support.util.Utils;
import org.camunda.bpm.engine.impl.cfg.CompositeProcessEnginePlugin;
import org.camunda.bpm.engine.impl.cfg.ProcessEngineConfigurationImpl;
import org.camunda.bpm.engine.impl.cfg.ProcessEnginePlugin;
import org.camunda.bpm.engine.impl.plugin.AdministratorAuthorizationPlugin;
import org.camunda.bpm.engine.spring.SpringProcessEngineConfiguration;
import org.camunda.bpm.spring.boot.starter.configuration.CamundaAuthorizationConfiguration;
import org.camunda.bpm.spring.boot.starter.util.CamundaSpringBootUtil;
import org.camunda.bpm.webapp.impl.security.auth.ContainerBasedAuthenticationFilter;
import org.camunda.connect.plugin.impl.ConnectProcessEnginePlugin;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableMBeanExport;
import org.springframework.jdbc.datasource.lookup.JndiDataSourceLookup;
import org.springframework.jmx.support.RegistrationPolicy;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Configuration
@EnableMBeanExport(registration = RegistrationPolicy.IGNORE_EXISTING)
public class BpmConfiguration {

    @Bean
    @Qualifier(Constants.DATASOURCE_QUALIFIER)
    public DataSource getBpmDataSource() {
        JndiDataSourceLookup lookup = new JndiDataSourceLookup();
        return lookup.getDataSource(Constants.DATASOURCE_NAME);
    }

    @Bean
    public ProcessEngineConfigurationImpl processEngineConfigurationImpl(List<ProcessEnginePlugin> processEnginePlugins) {
        final List<ProcessEnginePlugin> plugins = composeProcessEnginePlugins(processEnginePlugins);

        final SpringProcessEngineConfiguration configuration = CamundaSpringBootUtil.springProcessEngineConfiguration();

        configuration.getProcessEnginePlugins().add(new CompositeProcessEnginePlugin(plugins));

        return configuration;
    }

    @Bean
    public static CamundaAuthorizationConfiguration camundaAuthorizationConfiguration() {
        return new CronAuthorizationConfiguration();
    }

    @Bean
    public FilterRegistrationBean<CronAuthenticationTokenFilter> cronAuthenticationTokenFilterFilter() {
        FilterRegistrationBean<CronAuthenticationTokenFilter> filterRegistration = new FilterRegistrationBean<>();
        filterRegistration.setFilter(new CronAuthenticationTokenFilter());
        filterRegistration.setOrder(100);
        return filterRegistration;
    }

    @Bean
    public FilterRegistrationBean<ContainerBasedAuthenticationFilter> containerBasedAuthenticationFilter() {
        FilterRegistrationBean<ContainerBasedAuthenticationFilter> filterRegistration = new FilterRegistrationBean<>();
        filterRegistration.setFilter(new ContainerBasedAuthenticationFilter());
        filterRegistration.setInitParameters(Collections.singletonMap(Constants.AUTHENTICATION_PROVIDER, Constants.AUTHENTICATION_PROVIDER_CLASS));
        filterRegistration.setOrder(101);
        filterRegistration.addUrlPatterns(getAppUrlPatterns());
        return filterRegistration;
    }

    @Bean
    public FilterRegistrationBean<CronRestAuthenticationFilter> cronRestAuthenticationFilter() {
        FilterRegistrationBean<CronRestAuthenticationFilter> filterRegistration = new FilterRegistrationBean<>();
        filterRegistration.setFilter(new CronRestAuthenticationFilter());
        filterRegistration.setOrder(102);
        filterRegistration.addUrlPatterns("/rest/*", "/api/engine/*");
        return filterRegistration;
    }

    private List<ProcessEnginePlugin> composeProcessEnginePlugins(List<ProcessEnginePlugin> processEnginePlugins) {
        final List<ProcessEnginePlugin> plugins = new ArrayList<>(processEnginePlugins);

        AdministratorAuthorizationPlugin authorizationPlugin = new AdministratorAuthorizationPlugin();
        authorizationPlugin.setAdministratorGroupName(Utils.getAdminRoleName());

        CronIdentityProviderPlugin cronIdentityProviderPlugin = new CronIdentityProviderPlugin();

        ConnectProcessEnginePlugin connectProcessEnginePlugin = new ConnectProcessEnginePlugin();

        FilterPlugin filterPlugin = new FilterPlugin();

        Collections.addAll(plugins, authorizationPlugin, cronIdentityProviderPlugin, connectProcessEnginePlugin, filterPlugin);

        return plugins;
    }

    private String[] getAppUrlPatterns() {
        return new String[]{
                "/app/admin/default/*",
                "/app/welcome/default/*",
                "/app/cockpit/default/*",
                "/app/tasklist/default/*"
        };
    }
}