package cronapi.workflow;

import cronapi.CronapiMetaData;
import org.camunda.bpm.engine.*;
import org.camunda.bpm.engine.authorization.Permissions;
import org.camunda.bpm.engine.authorization.Resources;
import org.camunda.bpm.engine.batch.Batch;
import org.camunda.bpm.engine.batch.history.HistoricBatch;
import org.camunda.bpm.engine.batch.history.HistoricBatchQuery;
import org.camunda.bpm.engine.history.*;
import org.camunda.bpm.engine.impl.cfg.ProcessEngineConfigurationImpl;
import org.camunda.bpm.engine.impl.context.Context;
import org.camunda.bpm.engine.repository.CaseDefinition;
import org.camunda.bpm.engine.repository.DecisionDefinition;
import org.camunda.bpm.engine.repository.ProcessDefinition;
import org.camunda.bpm.engine.runtime.Job;

import java.util.List;

@CronapiMetaData(categoryName = "Workflow History Operations",
        categoryTags = {"Workflow", "History", "Operations"})
public class WorkflowHistoryOperations {

    private static HistoryService getHistoryService() {
        return Context.getProcessEngineConfiguration().getHistoryService();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricProcessInstance}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricProcessInstanceQuery}}",
            description = "{{createHistoricProcessInstanceQueryDescription}}"
    )
    public static HistoricProcessInstanceQuery createHistoricProcessInstanceQuery() {
        return getHistoryService().createHistoricProcessInstanceQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricActivityInstance}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricActivityInstanceQuery}}",
            description = "{{createHistoricActivityInstanceQueryDescription}}"
    )
    public static HistoricActivityInstanceQuery createHistoricActivityInstanceQuery() {
        return getHistoryService().createHistoricActivityInstanceQuery();
    }

    /**
     * Query for the number of historic activity instances aggregated by activities of a single process definition.
     *
     * @param processDefinitionId
     */
    @CronapiMetaData(
            name = "{{createHistoricActivityStatisticsQuery}}",
            description = "{{createHistoricActivityStatisticsQueryDescription}}"
    )
    public static HistoricActivityStatisticsQuery createHistoricActivityStatisticsQuery(String processDefinitionId) {
        return getHistoryService().createHistoricActivityStatisticsQuery(processDefinitionId);
    }

    /**
     * Query for the number of historic case activity instances aggregated by case activities of a single case definition.
     *
     * @param caseDefinitionId
     */
    @CronapiMetaData(
            name = "{{createHistoricCaseActivityStatisticsQuery}}",
            description = "{{createHistoricCaseActivityStatisticsQueryDescription}}"
    )
    public static HistoricCaseActivityStatisticsQuery createHistoricCaseActivityStatisticsQuery(String caseDefinitionId) {
        return getHistoryService().createHistoricCaseActivityStatisticsQuery(caseDefinitionId);
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricTaskInstance}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricTaskInstanceQuery}}",
            description = "{{createHistoricTaskInstanceQueryDescription}}"
    )
    public static HistoricTaskInstanceQuery createHistoricTaskInstanceQuery() {
        return getHistoryService().createHistoricTaskInstanceQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricDetail}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricDetailQuery}}",
            description = "{{createHistoricDetailQueryDescription}}"
    )
    public static HistoricDetailQuery createHistoricDetailQuery() {
        return getHistoryService().createHistoricDetailQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricVariableInstance}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricVariableInstanceQuery}}",
            description = "{{createHistoricVariableInstanceQueryDescription}}"
    )
    public static HistoricVariableInstanceQuery createHistoricVariableInstanceQuery() {
        return getHistoryService().createHistoricVariableInstanceQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link UserOperationLogEntry} instances.
     */
    @CronapiMetaData(
            name = "{{createUserOperationLogQuery}}",
            description = "{{createUserOperationLogQueryDescription}}"
    )
    public static UserOperationLogQuery createUserOperationLogQuery() {
        return getHistoryService().createUserOperationLogQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricIncident historic incidents}.
     */
    @CronapiMetaData(
            name = "{{createHistoricIncidentQuery}}",
            description = "{{createHistoricIncidentQueryDescription}}"
    )
    public static HistoricIncidentQuery createHistoricIncidentQuery() {
        return getHistoryService().createHistoricIncidentQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricIdentityLinkLog historic identity links}.
     */
    @CronapiMetaData(
            name = "{{createHistoricIdentityLinkLogQuery}}",
            description = "{{createHistoricIdentityLinkLogQueryDescription}}"
    )
    public static HistoricIdentityLinkLogQuery createHistoricIdentityLinkLogQuery() {
        return getHistoryService().createHistoricIdentityLinkLogQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricCaseInstance}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricCaseInstanceQuery}}",
            description = "{{createHistoricCaseInstanceQueryDescription}}"
    )
    public static HistoricCaseInstanceQuery createHistoricCaseInstanceQuery() {
        return getHistoryService().createHistoricCaseInstanceQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricCaseActivityInstance}s.
     */
    @CronapiMetaData(
            name = "{{createHistoricCaseActivityInstanceQuery}}",
            description = "{{createHistoricCaseActivityInstanceQueryDescription}}"
    )
    public static HistoricCaseActivityInstanceQuery createHistoricCaseActivityInstanceQuery() {
        return getHistoryService().createHistoricCaseActivityInstanceQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricDecisionInstance}s.
     * <p>
     * If the user has no {@link Permissions#READ_HISTORY} permission on {@link Resources#DECISION_DEFINITION}
     * then the result of the query is empty.
     */
    @CronapiMetaData(
            name = "{{createHistoricDecisionInstanceQuery}}",
            description = "{{createHistoricDecisionInstanceQueryDescription}}"
    )
    public static HistoricDecisionInstanceQuery createHistoricDecisionInstanceQuery() {
        return getHistoryService().createHistoricDecisionInstanceQuery();
    }

    /**
     * Deletes historic task instance.  This might be useful for tasks that are
     * {@link TaskService#newTask() dynamically created} and then {@link TaskService#complete(String) completed}.
     * If the historic task instance doesn't exist, no exception is thrown and the
     * method returns normal.
     *
     * @param taskId
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricTaskInstance}}",
            description = "{{deleteHistoricTaskInstanceDescription}}"
    )
    public static void deleteHistoricTaskInstance(String taskId) {
        getHistoryService().deleteHistoricTaskInstance(taskId);
    }

    /**
     * Deletes historic process instance. All historic activities, historic task and
     * historic details (variable updates, form properties) are deleted as well.
     *
     * @param processInstanceId
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricProcessInstance}}",
            description = "{{deleteHistoricProcessInstanceDescription}}"
    )
    public static void deleteHistoricProcessInstance(String processInstanceId) {
        getHistoryService().deleteHistoricProcessInstance(processInstanceId);
    }

    /**
     * Deletes historic process instances. All historic activities, historic task and
     * historic details (variable updates, form properties) are deleted as well.
     *
     * @param processInstanceIds
     * @throws BadUserRequestException when no process instances are found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricProcessInstances}}",
            description = "{{deleteHistoricProcessInstancesDescription}}"
    )
    public static void deleteHistoricProcessInstances(List<String> processInstanceIds) {
        getHistoryService().deleteHistoricProcessInstances(processInstanceIds);
    }

    /**
     * Deletes historic process instances and all related historic data in bulk manner. DELETE SQL statement will be created for each entity type. They will have list
     * of given process instance ids in IN clause. Therefore, DB limitation for number of values in IN clause must be taken into account.
     *
     * @param processInstanceIds list of process instance ids for removal
     * @throws BadUserRequestException when no process instances are found with the given ids or ids are null or when some of the process instances are not finished yet
     * @throws AuthorizationException  If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricProcessInstancesBulk}}",
            description = "{{deleteHistoricProcessInstancesBulkDescription}}"
    )
    public static void deleteHistoricProcessInstancesBulk(List<String> processInstanceIds) {
        getHistoryService().deleteHistoricProcessInstancesBulk(processInstanceIds);
    }

    /**
     * Schedules history cleanup job at batch window start time. The job will delete historic data for
     * finished process, decision and case instances, and batch operations taking into account {@link ProcessDefinition#getHistoryTimeToLive()},
     * {@link DecisionDefinition#getHistoryTimeToLive()}, {@link CaseDefinition#getHistoryTimeToLive()}, {@link ProcessEngineConfigurationImpl#getBatchOperationHistoryTimeToLive()}
     * and {@link ProcessEngineConfigurationImpl#getBatchOperationsForHistoryCleanup()} values.
     *
     * @return history cleanup job. NB! As of v. 7.9.0, method does not guarantee to return a job. Use {@link #findHistoryCleanupJobs()} instead.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}
     */
    @CronapiMetaData(
            name = "{{cleanUpHistoryAsync}}",
            description = "{{cleanUpHistoryAsyncDescription}}"
    )
    public static Job cleanUpHistoryAsync() {
        return getHistoryService().cleanUpHistoryAsync();
    }

    /**
     * Schedules history cleanup job at batch window start time. The job will delete historic data for
     * finished process, decision and case instances, and batch operations taking into account {@link ProcessDefinition#getHistoryTimeToLive()},
     * {@link DecisionDefinition#getHistoryTimeToLive()}, {@link CaseDefinition#getHistoryTimeToLive()}, {@link ProcessEngineConfigurationImpl#getBatchOperationHistoryTimeToLive()}
     * and {@link ProcessEngineConfigurationImpl#getBatchOperationsForHistoryCleanup()} values.
     *
     * @param immediatelyDue must be true if cleanup must be scheduled at once, otherwise is will be scheduled according to configured batch window
     * @return history cleanup job. Job id can be used to check job logs, incident etc.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}
     */
    @CronapiMetaData(
            name = "{{cleanUpHistoryAsync}}",
            description = "{{cleanUpHistoryAsyncDescription}}"
    )
    public static Job cleanUpHistoryAsync(boolean immediatelyDue) {
        return getHistoryService().cleanUpHistoryAsync(immediatelyDue);
    }

    /**
     * Finds history cleanup job, if present.
     *
     * @return job entity
     * @deprecated As of v. 7.9.0, because there can be more than one history cleanup job at once, use {@link #findHistoryCleanupJobs} instead.
     */
    @CronapiMetaData(
            name = "{{findHistoryCleanupJob}}",
            description = "{{findHistoryCleanupJobDescription}}"
    )
    public static Job findHistoryCleanupJob() {
        return getHistoryService().findHistoryCleanupJob();
    }

    /**
     * Finds history cleanup job if present.
     *
     * @return job entity
     */
    @CronapiMetaData(
            name = "{{findHistoryCleanupJobs}}",
            description = "{{findHistoryCleanupJobsDescription}}"
    )
    public static List<Job> findHistoryCleanupJobs() {
        return getHistoryService().findHistoryCleanupJobs();
    }

    /**
     * Deletes historic process instances asynchronously. All historic activities, historic task and
     * historic details (variable updates, form properties) are deleted as well.
     *
     * @param processInstanceIds
     * @param deleteReason
     * @throws BadUserRequestException when no process instances is found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#CREATE} permission on {@link Resources#BATCH}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricProcessInstancesAsync}}",
            description = "{{deleteHistoricProcessInstancesAsyncDescription}}"
    )
    public static Batch deleteHistoricProcessInstancesAsync(List<String> processInstanceIds, String deleteReason) {
        return getHistoryService().deleteHistoricProcessInstancesAsync(processInstanceIds, deleteReason);
    }

    /**
     * Deletes historic process instances asynchronously based on query. All historic activities, historic task and
     * historic details (variable updates, form properties) are deleted as well.
     *
     * @param query
     * @param deleteReason
     * @throws BadUserRequestException when no process instances is found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#CREATE} permission on {@link Resources#BATCH}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricProcessInstancesAsync}}",
            description = "{{deleteHistoricProcessInstancesAsyncDescription}}"
    )
    public static Batch deleteHistoricProcessInstancesAsync(HistoricProcessInstanceQuery query, String deleteReason) {
        return getHistoryService().deleteHistoricProcessInstancesAsync(query, deleteReason);
    }

    /**
     * Deletes historic process instances asynchronously based on query and a list of process instances. Query result and
     * list of ids will be merged.
     * All historic activities, historic task and historic details (variable updates, form properties) are deleted as well.
     *
     * @param processInstanceIds
     * @param query
     * @param deleteReason
     * @throws BadUserRequestException when no process instances is found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#CREATE} permission on {@link Resources#BATCH}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricProcessInstancesAsync}}",
            description = "{{deleteHistoricProcessInstancesAsyncDescription}}"
    )
    public static Batch deleteHistoricProcessInstancesAsync(List<String> processInstanceIds, HistoricProcessInstanceQuery query, String deleteReason) {
        return getHistoryService().deleteHistoricProcessInstancesAsync(processInstanceIds, query, deleteReason);
    }

    /**
     * Deletes a user operation log entry. Does not cascade to any related entities.
     *
     * @param entryId
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteUserOperationLogEntry}}",
            description = "{{deleteUserOperationLogEntryDescription}}"
    )
    public static void deleteUserOperationLogEntry(String entryId) {
        getHistoryService().deleteUserOperationLogEntry(entryId);
    }

    /**
     * Deletes historic case instance. All historic case activities, historic task and
     * historic details are deleted as well.
     *
     * @param caseInstanceId
     */
    @CronapiMetaData(
            name = "{{deleteHistoricCaseInstance}}",
            description = "{{deleteHistoricCaseInstanceDescription}}"
    )
    public static void deleteHistoricCaseInstance(String caseInstanceId) {
        getHistoryService().deleteHistoricCaseInstance(caseInstanceId);
    }

    /**
     * Deletes historic case instances and all related historic data in bulk manner. DELETE SQL statement will be created for each entity type. They will have list
     * of given case instance ids in IN clause. Therefore, DB limitation for number of values in IN clause must be taken into account.
     *
     * @param caseInstanceIds list of case instance ids for removal
     */
    @CronapiMetaData(
            name = "{{deleteHistoricCaseInstancesBulk}}",
            description = "{{deleteHistoricCaseInstancesBulkDescription}}"
    )
    public static void deleteHistoricCaseInstancesBulk(List<String> caseInstanceIds) {
        getHistoryService().deleteHistoricCaseInstancesBulk(caseInstanceIds);
    }

    /**
     * Deletes historic decision instances of a decision definition. All historic
     * decision inputs and outputs are deleted as well.
     *
     * @param decisionDefinitionId the id of the decision definition
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#DECISION_DEFINITION}.
     * @deprecated Note that this method name is not expressive enough, because it is also possible to delete the historic
     * decision instance by the instance id. Therefore use {@link #deleteHistoricDecisionInstanceByDefinitionId} instead
     * to delete the historic decision instance by the definition id.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstance}}",
            description = "{{deleteHistoricDecisionInstanceDescription}}"
    )
    public static void deleteHistoricDecisionInstance(String decisionDefinitionId) {
        getHistoryService().deleteHistoricDecisionInstance(decisionDefinitionId);
    }

    /**
     * Deletes decision instances and all related historic data in bulk manner. DELETE SQL statement will be created for each entity type. They will have list
     * of given decision instance ids in IN clause. Therefore, DB limitation for number of values in IN clause must be taken into account.
     *
     * @param decisionInstanceIds list of decision instance ids for removal.
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstancesBulk}}",
            description = "{{deleteHistoricDecisionInstancesBulkDescription}}"
    )
    public static void deleteHistoricDecisionInstancesBulk(List<String> decisionInstanceIds) {
        getHistoryService().deleteHistoricDecisionInstancesBulk(decisionInstanceIds);
    }

    /**
     * Deletes historic decision instances of a decision definition. All historic
     * decision inputs and outputs are deleted as well.
     *
     * @param decisionDefinitionId the id of the decision definition
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstanceByDefinitionId}}",
            description = "{{deleteHistoricDecisionInstanceByDefinitionIdDescription}}"
    )
    public static void deleteHistoricDecisionInstanceByDefinitionId(String decisionDefinitionId) {
        getHistoryService().deleteHistoricDecisionInstanceByDefinitionId(decisionDefinitionId);
    }

    /**
     * Deletes historic decision instances by its id. All historic
     * decision inputs and outputs are deleted as well.
     *
     * @param historicDecisionInstanceId the id of the historic decision instance
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE_HISTORY} permission on {@link Resources#DECISION_DEFINITION}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstanceByInstanceId}}",
            description = "{{deleteHistoricDecisionInstanceByInstanceIdDescription}}"
    )
    public static void deleteHistoricDecisionInstanceByInstanceId(String historicDecisionInstanceId) {
        getHistoryService().deleteHistoricDecisionInstanceByInstanceId(historicDecisionInstanceId);
    }

    /**
     * Deletes historic decision instances asynchronously based on a list of decision instances.
     *
     * @param decisionInstanceIds
     * @param deleteReason
     * @throws BadUserRequestException when no decision instances are found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#CREATE} permission on {@link Resources#BATCH}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstancesAsync}}",
            description = "{{deleteHistoricDecisionInstancesAsyncDescription}}"
    )
    public static Batch deleteHistoricDecisionInstancesAsync(List<String> decisionInstanceIds, String deleteReason) {
        return getHistoryService().deleteHistoricDecisionInstancesAsync(decisionInstanceIds, deleteReason);
    }

    /**
     * Deletes historic decision instances asynchronously based on query of decision instances.
     *
     * @param query
     * @param deleteReason
     * @throws BadUserRequestException when no decision instances are found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#CREATE} permission on {@link Resources#BATCH}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstancesAsync}}",
            description = "{{deleteHistoricDecisionInstancesAsyncDescription}}"
    )
    public static Batch deleteHistoricDecisionInstancesAsync(HistoricDecisionInstanceQuery query, String deleteReason) {
        return getHistoryService().deleteHistoricDecisionInstancesAsync(query, deleteReason);
    }

    /**
     * Deletes historic decision instances asynchronously based on query and a list of decision instances, whereby query result and
     * list of ids will be merged.
     *
     * @param decisionInstanceIds
     * @param query
     * @param deleteReason
     * @throws BadUserRequestException when no decision instances are found with the given ids or ids are null.
     * @throws AuthorizationException  If the user has no {@link Permissions#CREATE} permission on {@link Resources#BATCH}.
     */
    @CronapiMetaData(
            name = "{{deleteHistoricDecisionInstancesAsync}}",
            description = "{{deleteHistoricDecisionInstancesAsyncDescription}}"
    )
    public static Batch deleteHistoricDecisionInstancesAsync(List<String> decisionInstanceIds, HistoricDecisionInstanceQuery query, String deleteReason) {
        return getHistoryService().deleteHistoricDecisionInstancesAsync(decisionInstanceIds, query, deleteReason);
    }

    /**
     * creates a native query to search for {@link HistoricProcessInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricProcessInstanceQuery}}",
            description = "{{createNativeHistoricProcessInstanceQueryDescription}}"
    )
    public static NativeHistoricProcessInstanceQuery createNativeHistoricProcessInstanceQuery() {
        return getHistoryService().createNativeHistoricProcessInstanceQuery();
    }

    /**
     * creates a native query to search for {@link HistoricTaskInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricTaskInstanceQuery}}",
            description = "{{createNativeHistoricTaskInstanceQueryDescription}}"
    )
    public static NativeHistoricTaskInstanceQuery createNativeHistoricTaskInstanceQuery() {
        return getHistoryService().createNativeHistoricTaskInstanceQuery();
    }

    /**
     * creates a native query to search for {@link HistoricActivityInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricActivityInstanceQuery}}",
            description = "{{createNativeHistoricActivityInstanceQueryDescription}}"
    )
    public static NativeHistoricActivityInstanceQuery createNativeHistoricActivityInstanceQuery() {
        return getHistoryService().createNativeHistoricActivityInstanceQuery();
    }

    /**
     * creates a native query to search for {@link HistoricCaseInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricCaseInstanceQuery}}",
            description = "{{createNativeHistoricCaseInstanceQueryDescription}}"
    )
    public static NativeHistoricCaseInstanceQuery createNativeHistoricCaseInstanceQuery() {
        return getHistoryService().createNativeHistoricCaseInstanceQuery();
    }

    /**
     * creates a native query to search for {@link HistoricCaseActivityInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricCaseActivityInstanceQuery}}",
            description = "{{createNativeHistoricCaseActivityInstanceQueryDescription}}"
    )
    public static NativeHistoricCaseActivityInstanceQuery createNativeHistoricCaseActivityInstanceQuery() {
        return getHistoryService().createNativeHistoricCaseActivityInstanceQuery();
    }

    /**
     * creates a native query to search for {@link HistoricDecisionInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricDecisionInstanceQuery}}",
            description = "{{createNativeHistoricDecisionInstanceQueryDescription}}"
    )
    public static NativeHistoricDecisionInstanceQuery createNativeHistoricDecisionInstanceQuery() {
        return getHistoryService().createNativeHistoricDecisionInstanceQuery();
    }

    /**
     * creates a native query to search for {@link HistoricVariableInstance}s via SQL
     */
    @CronapiMetaData(
            name = "{{createNativeHistoricVariableInstanceQuery}}",
            description = "{{createNativeHistoricVariableInstanceQueryDescription}}"
    )
    public static NativeHistoricVariableInstanceQuery createNativeHistoricVariableInstanceQuery() {
        return getHistoryService().createNativeHistoricVariableInstanceQuery();
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricJobLog historic job logs}.
     *
     * @since 7.3
     */
    @CronapiMetaData(
            name = "{{createHistoricJobLogQuery}}",
            description = "{{createHistoricJobLogQueryDescription}}"
    )
    public static HistoricJobLogQuery createHistoricJobLogQuery() {
        return getHistoryService().createHistoricJobLogQuery();
    }

    /**
     * Returns the full stacktrace of the exception that occurs when the
     * historic job log with the given id was last executed. Returns null
     * when the historic job log has no exception stacktrace.
     *
     * @param historicJobLogId id of the historic job log, cannot be null.
     * @throws ProcessEngineException when no historic job log exists with the given id.
     * @throws AuthorizationException If the user has no {@link Permissions#READ_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     * @since 7.3
     */
    @CronapiMetaData(
            name = "{{getHistoricJobLogExceptionStacktrace}}",
            description = "{{getHistoricJobLogExceptionStacktraceDescription}}"
    )
    public static String getHistoricJobLogExceptionStacktrace(String historicJobLogId) {
        return getHistoryService().getHistoricJobLogExceptionStacktrace(historicJobLogId);
    }

    /**
     * Creates a new programmatic query to create a historic process instance report.
     *
     * @since 7.5
     */
    @CronapiMetaData(
            name = "{{createHistoricProcessInstanceReport}}",
            description = "{{createHistoricProcessInstanceReportDescription}}"
    )
    public static HistoricProcessInstanceReport createHistoricProcessInstanceReport() {
        return getHistoryService().createHistoricProcessInstanceReport();
    }

    /**
     * Creates a new programmatic query to create a historic task instance report.
     *
     * @since 7.6
     */
    @CronapiMetaData(
            name = "{{createHistoricTaskInstanceReport}}",
            description = "{{createHistoricTaskInstanceReportDescription}}"
    )
    public static HistoricTaskInstanceReport createHistoricTaskInstanceReport() {
        return getHistoryService().createHistoricTaskInstanceReport();
    }

    /**
     * Creates a new programmatic query to create a cleanable historic process instance report.
     *
     * @since 7.8
     */
    @CronapiMetaData(
            name = "{{createCleanableHistoricProcessInstanceReport}}",
            description = "{{createCleanableHistoricProcessInstanceReportDescription}}"
    )
    public static CleanableHistoricProcessInstanceReport createCleanableHistoricProcessInstanceReport() {
        return getHistoryService().createCleanableHistoricProcessInstanceReport();
    }

    /**
     * Creates a new programmatic query to create a cleanable historic decision instance report.
     *
     * @since 7.8
     */
    @CronapiMetaData(
            name = "{{createCleanableHistoricDecisionInstanceReport}}",
            description = "{{createCleanableHistoricDecisionInstanceReportDescription}}"
    )
    public static CleanableHistoricDecisionInstanceReport createCleanableHistoricDecisionInstanceReport() {
        return getHistoryService().createCleanableHistoricDecisionInstanceReport();
    }

    /**
     * Creates a new programmatic query to create a cleanable historic case instance report.
     *
     * @since 7.8
     */
    @CronapiMetaData(
            name = "{{createCleanableHistoricCaseInstanceReport}}",
            description = "{{createCleanableHistoricCaseInstanceReportDescription}}"
    )
    public static CleanableHistoricCaseInstanceReport createCleanableHistoricCaseInstanceReport() {
        return getHistoryService().createCleanableHistoricCaseInstanceReport();
    }

    /**
     * Creates a new programmatic query to create a cleanable historic batch report.
     *
     * @since 7.8
     */
    @CronapiMetaData(
            name = "{{createCleanableHistoricBatchReport}}",
            description = "{{createCleanableHistoricBatchReportDescription}}"
    )
    public static CleanableHistoricBatchReport createCleanableHistoricBatchReport() {
        return getHistoryService().createCleanableHistoricBatchReport();
    }

    /**
     * Creates a query to search for {@link HistoricBatch} instances.
     *
     * @since 7.5
     */
    @CronapiMetaData(
            name = "{{createHistoricBatchQuery}}",
            description = "{{createHistoricBatchQueryDescription}}"
    )
    public static HistoricBatchQuery createHistoricBatchQuery() {
        return getHistoryService().createHistoricBatchQuery();
    }

    /**
     * Deletes a historic batch instance. All corresponding historic job logs are deleted as well;
     *
     * @param id
     * @throws AuthorizationException If the user has no {@link Permissions#DELETE} permission on {@link Resources#BATCH}
     * @since 7.5
     */
    @CronapiMetaData(
            name = "{{deleteHistoricBatch}}",
            description = "{{deleteHistoricBatchDescription}}"
    )
    public static void deleteHistoricBatch(String id) {
        getHistoryService().deleteHistoricBatch(id);
    }

    /**
     * Query for the statistics of DRD evaluation.
     *
     * @param decisionRequirementsDefinitionId - id of decision requirement definition
     * @since 7.6
     */
    @CronapiMetaData(
            name = "{{createHistoricDecisionInstanceStatisticsQuery}}",
            description = "{{createHistoricDecisionInstanceStatisticsQueryDescription}}"
    )
    public static HistoricDecisionInstanceStatisticsQuery createHistoricDecisionInstanceStatisticsQuery(String decisionRequirementsDefinitionId) {
        return getHistoryService().createHistoricDecisionInstanceStatisticsQuery(decisionRequirementsDefinitionId);
    }

    /**
     * Creates a new programmatic query to search for {@link HistoricExternalTaskLog historic external task logs}.
     *
     * @since 7.7
     */
    @CronapiMetaData(
            name = "{{createHistoricExternalTaskLogQuery}}",
            description = "{{createHistoricExternalTaskLogQueryvDescription}}"
    )
    public static HistoricExternalTaskLogQuery createHistoricExternalTaskLogQuery() {
        return getHistoryService().createHistoricExternalTaskLogQuery();
    }

    /**
     * Returns the full error details that occurs when the
     * historic external task log with the given id was last executed. Returns null
     * when the historic external task log contains no error details.
     *
     * @param historicExternalTaskLogId id of the historic external task log, cannot be null.
     * @throws ProcessEngineException when no historic external task log exists with the given id.
     * @throws AuthorizationException If the user has no {@link Permissions#READ_HISTORY} permission on {@link Resources#PROCESS_DEFINITION}.
     * @since 7.7
     */
    @CronapiMetaData(
            name = "{{getHistoricExternalTaskLogErrorDetails}}",
            description = "{{getHistoricExternalTaskLogErrorDetailsDescription}}"
    )
    public static String getHistoricExternalTaskLogErrorDetails(String historicExternalTaskLogId) {
        return getHistoryService().getHistoricExternalTaskLogErrorDetails(historicExternalTaskLogId);
    }
}
