package io.cronapp.sap.rfc;

import com.sap.conn.jco.*;
import com.sap.conn.jco.ext.DestinationDataProvider;
import cronapi.CronapiMetaData;
import cronapi.CronapiMetaData.ObjectType;
import cronapi.Var;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

@SuppressWarnings("unchecked")
@CronapiMetaData(categoryName = "SAP RFC", categoryTags = {"SAP", "RFC"})
public class Api {
  private static Logger LOGGER = Logger.getLogger(Api.class.getName());

  private static final String DESTINATION_NAME = "ABAP_AS";

  @CronapiMetaData(name = "Conectar ao SAP RFC", nameTags = {"conectar"}, description = "Função para se conectar ao SAP RFC",
      params = {"Cliente", "Usuário", "Senha", "Linguagem", "Endereço do servidor de aplicação SAP RFC",
          "Número do sistema do servidor de aplicativos SAP RFC", "Rastreamento", "Capacidade do pool de conexões",
          "Máximo de conexões ativas"},
      paramsType = {ObjectType.STRING, ObjectType.STRING, ObjectType.STRING, ObjectType.STRING, ObjectType.STRING,
          ObjectType.STRING, ObjectType.STRING, ObjectType.STRING, ObjectType.STRING},
      returnType = ObjectType.OBJECT)
  public static Var connect(Var client, Var user, Var passwd, Var lang, Var ashost, Var sysnr, Var trace, Var poolCapacity, Var peakLimit) throws IOException, JCoException {
    Properties connectProperties = new Properties();
    connectProperties.setProperty(DestinationDataProvider.JCO_LANG, lang.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_CLIENT, client.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_PASSWD, passwd.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_USER, user.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_SYSNR, sysnr.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_ASHOST, ashost.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_TRACE, trace.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_POOL_CAPACITY, poolCapacity.getObjectAsString());
    connectProperties.setProperty(DestinationDataProvider.JCO_PEAK_LIMIT, peakLimit.getObjectAsString());

    createDestinationDataFile(connectProperties);

    JCoDestination destination = JCoDestinationManager.getDestination(DESTINATION_NAME);
    return new Var(destination);
  }

  @CronapiMetaData(name = "Executar função no SAP RFC", nameTags = {"executar"}, description = "Função para executar uma chamada a um destino SAP RFC",
      params = {"Conector SAP RFC", "Nome da função", "Parâmetros da função"},
      paramsType = {ObjectType.OBJECT, ObjectType.STRING, ObjectType.MAP},
      returnType = ObjectType.MAP)
  public static Var call(Var destination, Var functionName, Var input) throws JCoException {
    JCoDestination jcoDestination = destination.getTypedObject(JCoDestination.class);
    JCoFunction function = jcoDestination.getRepository()
        .getFunction(functionName.getObjectAsString());
    toRecord(input.getObjectAsMap(), function.getImportParameterList());
    function.execute(jcoDestination);
    Map<String, Var> result = new LinkedHashMap<>();
    toMap(function.getExportParameterList(), result);
    toMap(function.getTableParameterList(), result);
    return new Var(result);
  }

  private static void createDestinationDataFile(Properties connectProperties) throws IOException {
    File destinationFile = new File(DESTINATION_NAME + ".jcoDestination");
    try (FileOutputStream destinationStream = new FileOutputStream(destinationFile, false)) {
      connectProperties.store(destinationStream, "Generated using CronApp");
    }
  }

  private static void toRecord(Map<String, Var> source, JCoRecord destination) {
    destination.forEach(field -> {
      Var value = source.get(field.getName());
      switch (field.getType()) {
        case JCoMetaData.TYPE_CHAR:
        case JCoMetaData.TYPE_STRING:
          field.setValue(value.getObjectAsString());
          break;
        case JCoMetaData.TYPE_DATE:
          field.setValue(new SimpleDateFormat("yyyyMMdd").format(value.getObjectAsDateTime()));
          break;
        case JCoMetaData.TYPE_BCD:
        case JCoMetaData.TYPE_NUM:
        case JCoMetaData.TYPE_FLOAT:
        case JCoMetaData.TYPE_DECF16:
        case JCoMetaData.TYPE_DECF34:
          field.setValue(value.getObjectAsDouble());
          break;
        case JCoMetaData.TYPE_TIME:
          field.setValue(new SimpleDateFormat("HHmmss").format(value.getObjectAsDateTime()));
          break;
        case JCoMetaData.TYPE_BYTE:
        case JCoMetaData.TYPE_XSTRING:
          field.setValue(value.getObjectAsByteArray());
          break;
        case JCoMetaData.TYPE_INT:
        case JCoMetaData.TYPE_INT2:
        case JCoMetaData.TYPE_INT1:
          field.setValue(value.getObjectAsInt());
          break;
        case JCoMetaData.TYPE_STRUCTURE:
          JCoStructure inputStructure = field.getStructure();
          toRecord(value.getObjectAsMap(), inputStructure);
          break;
        case JCoMetaData.TYPE_ITAB:
        case JCoMetaData.TYPE_TABLE:
        case JCoMetaData.TYPE_EXCEPTION:
        case JCoMetaData.TYPE_ABAPOBJECT:
        case JCoMetaData.TYPE_BOX:
        case JCoMetaData.TYPE_GENERIC_BOX:
          LOGGER.log(Level.WARNING, String.format("Unsupported record field type found while converting to record. Name: %s, Type: %d", field.getName(), field.getType()));
          break;
      }
    });
  }

  private static void toMap(JCoRecord source, Map<String, Var> destination) {
    if (source == null) return;
    source.forEach(field -> {
      String fieldName = field.getName();
      switch (field.getType()) {
        case JCoMetaData.TYPE_CHAR:
        case JCoMetaData.TYPE_STRING:
          destination.put(fieldName, new Var(field.getString()));
          break;
        case JCoMetaData.TYPE_DATE:
          destination.put(fieldName, new Var(field.getDate()));
          break;
        case JCoMetaData.TYPE_BCD:
        case JCoMetaData.TYPE_DECF16:
        case JCoMetaData.TYPE_DECF34:
          destination.put(fieldName, new Var(field.getBigDecimal()));
          break;
        case JCoMetaData.TYPE_NUM:
          destination.put(fieldName, new Var(field.getDouble()));
          break;
        case JCoMetaData.TYPE_FLOAT:
          destination.put(fieldName, new Var(field.getFloat()));
          break;
        case JCoMetaData.TYPE_TIME:
          destination.put(fieldName, new Var(field.getTime()));
          break;
        case JCoMetaData.TYPE_BYTE:
        case JCoMetaData.TYPE_XSTRING:
          destination.put(fieldName, new Var(field.getByteArray()));
          break;
        case JCoMetaData.TYPE_INT:
        case JCoMetaData.TYPE_INT2:
        case JCoMetaData.TYPE_INT1:
          destination.put(fieldName, new Var(field.getInt()));
          break;
        case JCoMetaData.TYPE_STRUCTURE:
          Map<String, Var> structure = new LinkedHashMap<>();
          toMap(field.getStructure(), structure);
          destination.put(fieldName, new Var(structure));
          break;
        case JCoMetaData.TYPE_TABLE:
          List<Var> table = new LinkedList<>();
          toList(field.getTable(), table);
          destination.put(fieldName, new Var(table));
          break;
        case JCoMetaData.TYPE_ITAB:
        case JCoMetaData.TYPE_EXCEPTION:
        case JCoMetaData.TYPE_ABAPOBJECT:
        case JCoMetaData.TYPE_BOX:
        case JCoMetaData.TYPE_GENERIC_BOX:
          LOGGER.log(Level.WARNING, String.format("Unsupported record field type found while converting to record. Name: %s, Type: %d", field.getName(), field.getType()));
          break;
      }
    });
  }

  private static void toList(JCoTable source, List<Var> destination) {
    if (source == null || source.isEmpty()) return;
    source.firstRow();
    do {
      Map<String, Var> row = new LinkedHashMap<>();
      toMap(source, row);
      destination.add(new Var(row));
    } while (source.nextRow());
  }
}