package cronapi.swagger;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.google.gson.*;
import cronapi.AppConfig;
import cronapi.ParamMetaData;
import cronapi.i18n.Messages;
import cronapi.rest.security.CronappSecurity;
import org.reflections.Reflections;
import org.reflections.scanners.ResourcesScanner;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Configuration;

import javax.persistence.Column;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Set;

import static cronapi.swagger.CronappOpenApiConsts.*;

@Configuration

public class CronappOpenApiDefinition {

    private String openapi = "3.0.1";
    private final ArrayList<ServerDefinition> servers = new ArrayList<>();
    private InfoDefinition info = new InfoDefinition();
    private LinkedHashMap<String, PathDefinition> paths = new LinkedHashMap();
    private final HashMap<String, HashMap>  components = new HashMap();
    private static final Logger logger = LoggerFactory.getLogger(CronappOpenApiDefinition.class);

    private void fillDefinitions(){
        this.servers.add(this.getDefaultServer());
        this.info.setTitle(AppConfig.getProjectName());
        this.info.setDescription(AppConfig.getProjectName());
        this.info.setVersion("1.0");
        this.fillComponentODataDefaultParams();
        this.fillSecurityDefinition();
        this.fillAuthDefinition();
        this.fillEntitiesOdataDefinition();
        this.fillDatasourcesOdataDefinition();
        this.fillBlocklyDefinition();
    }

    @JsonIgnore
    public JsonElement getJsonData() {

        ObjectWriter ow = new ObjectMapper().writer().withDefaultPrettyPrinter();
        var def = new CronappOpenApiDefinition();
        def.fillDefinitions();
        String json = null;
        try {
            json = ow.writeValueAsString(def);
        } catch (Exception e) {
            logger.error(e.getMessage());
        }

        return new Gson().fromJson(json, JsonElement.class);
    }

    private ServerDefinition getDefaultServer(){
        var server = new ServerDefinition();
        server.setDescription(Messages.getLocale().getCountry());
        server.setUrl("/");
        return server;
    }

    private void fillEntitiesOdataDefinition(){
        if(this.exposeEntities()){

            Reflections reflections = new Reflections(ODATA_PACKAGE);
            Set<Class<?>> allEntities = reflections.getTypesAnnotatedWith(CronappSwagger.class);

            for (Class<?> entity : allEntities) {

                var metaSecurity = entity.getAnnotation(CronappSecurity.class);
                var path = new PathDefinition();
                var tags = entity.getSimpleName().split(",");
                var paramBodyDefinition = new ParameterBodyDefinition(PARAMETER_JSON);
                paramBodyDefinition.setRequired(true);

                for (int j = 0; j < entity.getDeclaredFields().length; j++) {
                    Field field = entity.getDeclaredFields()[j];
                    var columnAnnotation = (Column) field.getAnnotation(Column.class);

                    if(columnAnnotation == null || (columnAnnotation != null && !columnAnnotation.insertable() || !columnAnnotation.updatable()))
                        continue;

                    var param = new LinkedHashMap();
                    param.put("type", this.parseType(field.getType().getName()));
                    param.put("description", columnAnnotation.name());
                    paramBodyDefinition.addProperty(columnAnnotation.name(), param);
                }

                if(!metaSecurity.get().equals(SECURITY_NONE)){
                    var operation = new OperationDefinition();
                    operation.setOperationId(entity.getSimpleName() + ACTION_GET);
                    operation.setTags(tags);
                    if(hasAuthentication()){
                        var securityType = new HashMap<String, ArrayList>();
                        securityType.put(getProjectAuthType(), new ArrayList());
                        var security = new ArrayList();
                        security.add(securityType);
                        operation.setSecurity(security);
                    }

                    operation.setParameters(this.fillODataDefaultParams());

                    operation.getResponses().put("200", this.getDefaultSuccessResponse());
                    operation.getResponses().put("500", this.getDefaultServerErrorResponse());
                    path.getOperations().putIfAbsent(ACTION_GET, operation);
                }

                if(!metaSecurity.post().equals(SECURITY_NONE)){
                    var operation = new OperationDefinition();
                    operation.setOperationId(entity.getSimpleName() + ACTION_POST);
                    operation.setTags(tags);
                    operation.setRequestBody(paramBodyDefinition);
                    if(hasAuthentication()){
                        var securityType = new HashMap<String, ArrayList>();
                        securityType.put(getProjectAuthType(), new ArrayList());
                        var security = new ArrayList();
                        security.add(securityType);
                        operation.setSecurity(security);
                    }
                    operation.getResponses().put("200", this.getDefaultSuccessResponse());
                    operation.getResponses().put("500", this.getDefaultServerErrorResponse());
                    path.getOperations().putIfAbsent(ACTION_POST, operation);
                }

                if(!metaSecurity.put().equals(SECURITY_NONE)){
                    var operation = new OperationDefinition();
                    operation.setOperationId(entity.getName() + ACTION_PUT);
                    operation.setTags(tags);
                    operation.setRequestBody(paramBodyDefinition);
                    if(hasAuthentication()){
                        var securityType = new HashMap<String, ArrayList>();
                        securityType.put(getProjectAuthType(), new ArrayList());
                        var security = new ArrayList();
                        security.add(securityType);
                        operation.setSecurity(security);
                    }
                    operation.getResponses().put("200", this.getDefaultSuccessResponse());
                    operation.getResponses().put("500", this.getDefaultServerErrorResponse());
                    path.getOperations().putIfAbsent(ACTION_PUT, operation);
                }

                this.paths.putIfAbsent(ODATA_REST_PREFIX + "app/" + entity.getSimpleName(), path);
            }
        }
    }

    private void fillDatasourcesOdataDefinition(){
        if(this.exposeDataSources()){
            ClassLoader classLoader = CronappOpenApiDefinition.class.getClassLoader();
            Reflections reflections = new Reflections("META-INF.datasources", new ResourcesScanner());
            Set<String> files = reflections.getResources(DATASOURCE_PATTERN);

            for (String file : files) {
                if (file.endsWith(".datasource.json")) {
                    try (InputStream stream = classLoader.getResourceAsStream(file)) {
                        try (InputStreamReader reader = new InputStreamReader(stream)) {
                            JsonObject datasource = new JsonParser().parse(reader).getAsJsonObject();
                            if(datasource.get("swagger") != null && datasource.get("swagger").getAsBoolean()){
                                JsonObject verbs = datasource.get("verbs").getAsJsonObject();
                                var path = new PathDefinition();
                                var tags = datasource.get("queryName").getAsString().split(",");
                                var datasourceId = datasource.getAsJsonObject().get("customId").getAsString();

                                if(this.isDatasourceVerbEnabled(verbs, ACTION_GET.toUpperCase())){
                                    var operation = new OperationDefinition();
                                    operation.setOperationId(datasourceId + ACTION_GET);
                                    operation.setTags(tags);
                                    if(hasAuthentication()){
                                        var securityType = new HashMap<String, ArrayList>();
                                        securityType.put(getProjectAuthType(), new ArrayList());
                                        var security = new ArrayList();
                                        security.add(securityType);
                                        operation.setSecurity(security);
                                    }

                                    if(datasource.getAsJsonObject().getAsJsonArray("queryParamsValues").size() > 0){
                                        var arrParamsDefinition = new ArrayList<ParameterDefinition>();
                                        JsonArray paramsArr = datasource.getAsJsonObject().getAsJsonArray("queryParamsValues");
                                        for (int i = 0; i < paramsArr.size() ; i++) {
                                            JsonObject param = paramsArr.get(i).getAsJsonObject();
                                            var paramDefinition = new ParameterDefinition();
                                            paramDefinition.setName(param.get("fieldName").getAsString());
                                            paramDefinition.setRequired(false);
                                            paramDefinition.setIn(PARAMETER_QUERY);
                                            arrParamsDefinition.add(paramDefinition);
                                        }
                                        operation.setParameters(arrParamsDefinition);
                                    }

                                    operation.getResponses().put("200", this.getDefaultSuccessResponse());
                                    operation.getResponses().put("500", this.getDefaultServerErrorResponse());
                                    path.getOperations().putIfAbsent(ACTION_GET, operation);
                                }

                                if(this.isDatasourceVerbEnabled(verbs, ACTION_POST.toUpperCase())){
                                    var operation = new OperationDefinition();
                                    operation.setOperationId(datasourceId + ACTION_POST);
                                    operation.setTags(tags);
                                    if(hasAuthentication()){
                                        var securityType = new HashMap<String, ArrayList>();
                                        securityType.put(getProjectAuthType(), new ArrayList());
                                        var security = new ArrayList();
                                        security.add(securityType);
                                        operation.setSecurity(security);
                                    }

                                    if(datasource.getAsJsonObject().getAsJsonArray("queryParamsValues").size() > 0){
                                        var paramBodyDefinition = new ParameterBodyDefinition(PARAMETER_JSON);
                                        paramBodyDefinition.setRequired(true);

                                        JsonArray paramsArr = datasource.getAsJsonObject().getAsJsonArray("queryParamsValues");
                                        for (int i = 0; i < paramsArr.size() ; i++) {
                                            JsonObject paramJson = paramsArr.get(i).getAsJsonObject();
                                            var param = new LinkedHashMap();
                                            param.put("type", "string");
                                            param.put("description", paramJson.get("fieldName").getAsString());
                                            paramBodyDefinition.addProperty(paramJson.get("fieldName").getAsString(), param);
                                        }
                                        operation.setRequestBody(paramBodyDefinition);
                                    }

                                    operation.getResponses().put("200", this.getDefaultSuccessResponse());
                                    operation.getResponses().put("500", this.getDefaultServerErrorResponse());
                                    path.getOperations().putIfAbsent(ACTION_POST, operation);
                                }

                                this.paths.putIfAbsent(ODATA_REST_PREFIX + "app/" + datasourceId, path);
                            }
                        }
                    } catch (Exception e) {
                        System.out.println(e.getMessage());
                    }
                }
            }
        }
    }

    private void fillBlocklyDefinition(){

        if(this.exposeBlocklys()) {
            Reflections reflections = new Reflections(BLOCKLY_PACKAGE);
            Set<Class<?>> allBlocklys = reflections.getTypesAnnotatedWith(CronappSwagger.class);

            for (Class<?> blockly : allBlocklys) {

                var metaSecurity = blockly.getAnnotation(CronappSecurity.class);

                for (int i = 0; i < blockly.getMethods().length; i++) {
                    var path = new PathDefinition();
                    Method method = blockly.getMethods()[i];

                    if (SKIP_METHODS.toLowerCase().contains(method.getName().toLowerCase()))
                        continue;

                    var arrParamsDefinition = new ArrayList<ParameterDefinition>();
                    var pathParams = new StringBuilder();

                    for (int j = 0; j < method.getParameterAnnotations().length; j++) {
                        if (method.getParameterAnnotations()[j].length < 1)
                            continue;
                        var paramAnnotation = (ParamMetaData) method.getParameterAnnotations()[j][0];
                        var paramDefinition = new ParameterDefinition();
                        paramDefinition.setName(paramAnnotation.description());
                        paramDefinition.setRequired(true);
                        paramDefinition.setIn(PARAMETER_PATH);
                        arrParamsDefinition.add(paramDefinition);
                        pathParams.append("/{" + paramAnnotation.description() + "}");
                    }
                    if (!metaSecurity.get().equals(SECURITY_NONE)) {
                        var operation = new OperationDefinition();
                        operation.setOperationId(method.getName() + ACTION_GET);
                        operation.setParameters(arrParamsDefinition);
                        operation.setTags(blockly.getName().split(","));
                        if (hasAuthentication()) {
                            var securityType = new HashMap<String, ArrayList>();
                            securityType.put(getProjectAuthType(), new ArrayList());
                            var security = new ArrayList();
                            security.add(securityType);
                            operation.setSecurity(security);
                        }
                        operation.getResponses().put("200", this.getDefaultSuccessResponse());
                        operation.getResponses().put("400", this.getDefaultBadRequestResponse());
                        path.getOperations().putIfAbsent(ACTION_GET, operation);
                    }
                    if (!metaSecurity.post().equals(SECURITY_NONE)) {
                        var operation = new OperationDefinition();
                        operation.setOperationId(method.getName() + ACTION_POST);
                        operation.setParameters(arrParamsDefinition);
                        operation.setTags(blockly.getName().split(","));
                        if (hasAuthentication()) {
                            var securityType = new HashMap<String, ArrayList>();
                            securityType.put(getProjectAuthType(), new ArrayList());
                            var security = new ArrayList();
                            security.add(securityType);
                            operation.setSecurity(security);
                        }
                        operation.getResponses().put("200", this.getDefaultSuccessResponse());
                        operation.getResponses().put("400", this.getDefaultBadRequestResponse());
                        path.getOperations().putIfAbsent(ACTION_POST, operation);
                    }
                    if (!metaSecurity.put().equals(SECURITY_NONE)) {
                        var operation = new OperationDefinition();
                        operation.setOperationId(method.getName() + ACTION_PUT);
                        operation.setParameters(arrParamsDefinition);
                        operation.setTags(blockly.getName().split(","));
                        if (hasAuthentication()) {
                            var securityType = new HashMap<String, ArrayList>();
                            securityType.put(getProjectAuthType(), new ArrayList());
                            var security = new ArrayList();
                            security.add(securityType);
                            operation.setSecurity(security);
                        }
                        operation.getResponses().put("200", this.getDefaultSuccessResponse());
                        operation.getResponses().put("400", this.getDefaultBadRequestResponse());
                        path.getOperations().putIfAbsent(ACTION_PUT, operation);
                    }
                    if (!metaSecurity.delete().equals(SECURITY_NONE)) {
                        var operation = new OperationDefinition();
                        operation.setOperationId(method.getName() + ACTION_DELETE);
                        operation.setParameters(arrParamsDefinition);
                        operation.setTags(blockly.getName().split(","));
                        if (hasAuthentication()) {
                            var securityType = new HashMap<String, ArrayList>();
                            securityType.put(getProjectAuthType(), new ArrayList());
                            var security = new ArrayList();
                            security.add(securityType);
                            operation.setSecurity(security);
                        }
                        operation.getResponses().put("200", this.getDefaultSuccessResponse());
                        operation.getResponses().put("400", this.getDefaultBadRequestResponse());
                        path.getOperations().putIfAbsent(ACTION_DELETE, operation);
                    }
                    this.paths.putIfAbsent(BLOCKLY_REST_PREFIX + blockly.getSimpleName() + ":" + method.getName() + pathParams.toString(), path);
                }
            }
        }
    }

    private void fillSecurityDefinition(){
        if(getProjectAuthType().equals(SECURITY_AUTH_TOKEN)){
            var apiKey = new HashMap<String, SecurityDefinition>();
            var securityDefinitions = new SecurityDefinition();
            securityDefinitions.setType(SECURITY_SCHEME_AUTH_APIKEY);
            securityDefinitions.setDescription(Messages.getString("swaggerAuthMessage"));
            securityDefinitions.setName(SECURITY_HEADER_NAME);
            securityDefinitions.setIn(PARAMETER_HEADER);
            apiKey.put(AppConfig.getJSON().get("auth").getAsJsonObject().get("type").getAsString(), securityDefinitions);
            this.components.put("securitySchemes", apiKey);
        }
    }

    private void fillComponentODataDefaultParams(){

        var oDataDefault = new HashMap<String, ParameterDefinition>();

        var pTop = new ParameterDefinition();
        pTop.setIn(PARAMETER_QUERY);
        pTop.setDescription(Messages.getString("swaggerTopParamDescription"));
        pTop.setName("top");
        oDataDefault.put("top", pTop);

        var pSkip = new ParameterDefinition();
        pSkip.setIn(PARAMETER_QUERY);
        pSkip.setDescription(Messages.getString("swaggerSkipParamDescription"));
        pSkip.setName("skip");
        oDataDefault.put("skip", pSkip);

        var pSearch = new ParameterDefinition();
        pSearch.setIn(PARAMETER_QUERY);
        pSearch.setDescription(Messages.getString("swaggerSearchParamDescription"));
        pSearch.setName("search");
        oDataDefault.put("search", pSearch);

        var pCount = new ParameterDefinition();
        pCount.setIn(PARAMETER_QUERY);
        pCount.setDescription(Messages.getString("swaggerCountParamDescription"));
        pCount.setName("count");
        oDataDefault.put("count", pCount);

        this.components.put("parameters", oDataDefault);
    }

    private ArrayList<ParameterDefinition> fillODataDefaultParams(){

        var arrParamsDefinition = new ArrayList<ParameterDefinition>();
        var pTop = new ParameterDefinition();
        pTop.setParamByRef("#/components/parameters/top");
        arrParamsDefinition.add(pTop);

        var pSkip = new ParameterDefinition();
        pSkip.setParamByRef("#/components/parameters/skip");
        arrParamsDefinition.add(pSkip);

        var pSearch = new ParameterDefinition();
        pSearch.setParamByRef("#/components/parameters/search");
        arrParamsDefinition.add(pSearch);

        var pCount = new ParameterDefinition();
        pCount.setParamByRef("#/components/parameters/count");
        arrParamsDefinition.add(pCount);

        return arrParamsDefinition;
    }

    private void fillAuthDefinition(){
        var pathAuth = new PathDefinition();
        var oprAuth = new OperationDefinition();
        oprAuth.setOperationId("00-cronapp-auth");
        oprAuth.setTags(Messages.getString("swaggerAuthorization").split(","));

        var paramBodyDefinition = new ParameterBodyDefinition(MULTIPART_FORMDATA);
        paramBodyDefinition.setRequired(true);
        var paramLogin = new LinkedHashMap();
        paramLogin.put("type", "string");
        paramLogin.put("description", Messages.getString("login"));
        paramBodyDefinition.addProperty("username", paramLogin);

        var paramPwd = new LinkedHashMap();
        paramPwd.put("type", "string");
        paramPwd.put("description", Messages.getString("password"));
        paramBodyDefinition.addProperty("password", paramPwd);
        oprAuth.setRequestBody(paramBodyDefinition);

        var response200 = new ResponseDefinition();
        var content = new HashMap();
        content.put("application/json", new HashMap());
        response200.setDescription(Messages.getString("swaggerLoginSuccess"));
        response200.setContent(content);
        oprAuth.getResponses().putIfAbsent("200", response200);

        var response401 = new ResponseDefinition();
        content = new HashMap();
        content.put("application/json", new HashMap());
        response401.setDescription(Messages.getString("loginPasswordInvalid"));
        response401.setContent(content);
        oprAuth.getResponses().putIfAbsent("401", response401);

        pathAuth.getOperations().putIfAbsent(ACTION_POST, oprAuth);
        this.paths.putIfAbsent("/auth", pathAuth);

        var pathRefresh = new PathDefinition();
        var oprRefresh = new OperationDefinition();
        oprRefresh.setOperationId("01-cronapp-auth");
        oprRefresh.setTags(Messages.getString("swaggerAuthorization").split(","));
        oprRefresh.getResponses().put("200", this.getDefaultSuccessResponse());
        oprRefresh.getResponses().put("400", this.getDefaultBadRequestResponse());
        pathRefresh.getOperations().putIfAbsent(ACTION_GET, oprRefresh);
        this.paths.putIfAbsent("/auth/refresh", pathRefresh);
    }

    private boolean hasAuthentication(){
        return !AppConfig.getJSON().get("auth").getAsJsonObject().get("type").getAsString().equals(SECURITY_AUTH_NONE);
    }

    private boolean isDatasourceVerbEnabled(JsonElement verbs, String verb){
        JsonElement verbToCheck = verbs.getAsJsonObject().get(verb);
        JsonElement verbAuthorities = verbs.getAsJsonObject().get(verb + "Authorities");
        JsonArray verbAuthoritiesList = verbAuthorities.isJsonNull() ? new JsonArray() : verbAuthorities.getAsJsonArray();
        return  verbToCheck.getAsBoolean()
                &&
                (
                  (verbAuthoritiesList.size() == 0)
                  ||
                  (verbAuthoritiesList.size() > 0 && !verbAuthoritiesList.get(0).getAsString().equals(SECURITY_NONE))
                );
    }

    private boolean exposeEntities(){
        if(AppConfig.getJSON().get("openApi") == null){
            return false;
        }
        return AppConfig.getJSON().get("openApi").getAsJsonObject().get("exposeEntities").getAsBoolean();
    }

    private boolean exposeDataSources(){
        if(AppConfig.getJSON().get("openApi") == null){
            return false;
        }
        return AppConfig.getJSON().get("openApi").getAsJsonObject().get("exposeDataSources").getAsBoolean();
    }

    private boolean exposeBlocklys(){
        if(AppConfig.getJSON().get("openApi") == null){
            return false;
        }
        return AppConfig.getJSON().get("openApi").getAsJsonObject().get("exposeBlocklys").getAsBoolean();
    }

    @JsonIgnore
    public boolean exposeEndpoints(){
        if(AppConfig.getJSON().get("openApi") == null){
            return false;
        }
        return AppConfig.getJSON().get("openApi").getAsJsonObject().get("exposeEndpoints").getAsBoolean();
    }

    private String getProjectAuthType(){
       return AppConfig.getJSON().get("auth").getAsJsonObject().get("type").getAsString();
    }

    private ResponseDefinition getDefaultSuccessResponse(){
        var response200 = new ResponseDefinition();
        response200.setDescription(Messages.getString("swaggerResponseSuccess"));
        return response200;
    }

    private ResponseDefinition getDefaultBadRequestResponse(){
        var response400 = new ResponseDefinition();
        response400.setDescription(Messages.getString("swaggerResponseError"));
        return response400;
    }

    private ResponseDefinition getDefaultServerErrorResponse(){
        var response500 = new ResponseDefinition();
        response500.setDescription(Messages.getString("errorNotSpecified"));
        return response500;
    }

    private String parseType(String type){
        switch (type){
            case "java.lang.Boolean":
                type = "boolean";
                break;
            case "java.lang.Integer":
                type = "integer";
                break;
            case "java.lang.Long":
                type = "integer";
                break;
            case "java.lang.Number":
                type = "number";
                break;
            default:
                type = "string";
                break;
        }
        return type;
    }

    public ArrayList<ServerDefinition> getServers() {
        return servers;
    }

    public void setOpenAPi(String openAPi) {
        this.openapi = openAPi;
    }

    public String getOpenapi() {
        return openapi;
    }

    public InfoDefinition getInfo() {
        return info;
    }

    public void setInfo(InfoDefinition info) {
        this.info = info;
    }

    public LinkedHashMap<String, PathDefinition> getPaths() {
        return paths;
    }

    public void setPaths(LinkedHashMap<String, PathDefinition> paths) {
        this.paths = paths;
    }

    public HashMap getComponents() {
        return components;
    }
}
