package cronapi.rest;

import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import cronapi.ErrorResponse;
import cronapi.QueryManager;
import cronapi.RestClient;
import cronapi.dashboard.DashboardFront;
import cronapi.dashboard.DashboardService;
import cronapi.i18n.Messages;
import cronapp.reports.commons.ReportFront;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

@RestController
@RequestMapping("/api/rest")
public class DashboardREST {

  private static final Logger log = LoggerFactory.getLogger(DashboardREST.class);

  private DashboardService dashboardService;

  @Autowired
  public DashboardREST(DashboardService dashboardService) {
    this.dashboardService = dashboardService;
  }

  @RequestMapping(value = "/dashboard", method = RequestMethod.POST, produces = MediaType.APPLICATION_JSON_VALUE)
  public ResponseEntity<DashboardFront> getReport(@RequestBody DashboardFront dashboardFront) {
    if (dashboardFront == null) {
      return ResponseEntity.badRequest().header("Error", "Dashboard is null").body(new DashboardFront());
    }
    log.debug("Get dashboard [" + dashboardFront + "].");
    DashboardFront Result = dashboardService.getDashboard(dashboardFront.getDashboardName());
    return ResponseEntity.ok().body(Result);
  }

  @RequestMapping(value = "/dashboard/contentasstring", method = RequestMethod.POST, produces = MediaType.TEXT_PLAIN_VALUE)
  public ResponseEntity<String> getContentAsString(@RequestBody DashboardFront dashboardFront, HttpServletResponse response) {
    if (dashboardFront == null) {
      return ResponseEntity.badRequest().header("Error", "Dashboard is null").body("Error read content file");
    }
    String dashboardName = dashboardFront.getDashboardName();
    log.info("contentasstring");
    log.debug("Print dashboard [" + dashboardName + "]");
    response.setHeader("Content-Disposition", "inline; filename=" + dashboardName);
    response.setContentType("application/plain");
    String dashboardResult = dashboardService.getContentDashboard(dashboardName);
    ResponseEntity error = checkSecurity(dashboardResult);
    if (error != null) {
      return error;
    }
    return ResponseEntity.ok().body(dashboardResult);
  }

  private ResponseEntity checkSecurity(String dashboardName)  {
    JsonObject reportsJson = (JsonObject) new JsonParser().parse(dashboardName);
    JsonElement reportsConfig = reportsJson.get("reportConfig");
    if (!QueryManager.isNull(reportsConfig) && !QueryManager.isNull(reportsConfig.getAsJsonObject().get("restSecurity"))) {
      JsonElement getValue = reportsConfig.getAsJsonObject().get("restSecurity").getAsJsonObject().get("get");
      if (!QueryManager.isNull(getValue)) {
        String security = getValue.getAsJsonPrimitive().getAsString();
        if (StringUtils.isNotEmpty(security)) {
          boolean authorized = false;
          String[] roles = security.split(",");
          for (String role : roles) {
            for (GrantedAuthority authority : RestClient.getRestClient().getAuthorities()) {
              if (role.trim().equalsIgnoreCase(authority.getAuthority())) {
                authorized = true;
                break;
              }
            }
          }
          if (!authorized) {
            JsonObject error = new JsonObject();
            error.addProperty("error", Messages.getString("notAllowed"));
            return ResponseEntity.status(403).body(error.toString());
          }
        }
      }
    }

    return null;
  }

  @ExceptionHandler(Throwable.class)
  @ResponseBody
  ResponseEntity<ErrorResponse> handleControllerException(HttpServletRequest req, Throwable ex) {
    ErrorResponse errorResponse = new ErrorResponse(HttpStatus.INTERNAL_SERVER_ERROR.value(), ex, req.getMethod());
    return new ResponseEntity<>(errorResponse, HttpStatus.INTERNAL_SERVER_ERROR);
  }
}