package cronapi.odata.server;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.PathNotFoundException;
import cronapi.QueryManager;
import cronapi.RestClient;
import cronapi.TokenUtils;
import cronapi.Var;
import cronapi.util.Functions;
import cronapi.util.JsonUtil;
import cronapi.util.Operations;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URIBuilder;
import org.apache.olingo.odata2.api.edm.EdmLiteralKind;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.core.edm.AbstractSimpleType;
import org.apache.olingo.odata2.core.edm.provider.EdmSimplePropertyImplProv;
import org.apache.olingo.odata2.jpa.processor.core.ODataJPAConfig;
import org.apache.olingo.odata2.jpa.processor.core.model.JPAEdmMappingImpl;

import javax.persistence.*;
import java.lang.reflect.Field;
import java.util.*;

public class WebServicesQuery extends ODataQuery implements Query, ODataQueryInterface {

  public static ThreadLocal<WebServicesQuery> CURRENT_REST_QUERY = new ThreadLocal<>();

  private Var lastResult;
  private Long count;
  private boolean isOData;
  private boolean isSOAP;
  private final ObjectMapper mapper;

  public WebServicesQuery(JsonObject query, String method, String type, String queryStatement, UriInfo uriInfo) {
    super(query, method, type, queryStatement, uriInfo);
    this.isOData = query.get("endpoint").getAsJsonObject().get("url").getAsString().contains("/api/cronapi/odata/");
    this.isSOAP = query.get("endpoint").getAsJsonObject().get("url").getAsString().toLowerCase().contains("wsdl");
    this.mapper = new ObjectMapper();
    mapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
  }

  public static boolean isNull(JsonElement value) {
    return value == null || value.isJsonNull();
  }

  private Map<String, Var> getHeaders(Object entity) {
    Map<String, Var> headers = new LinkedHashMap<>();
    if (!isNull(query.get("headers"))) {
      JsonObject paramValues = query.get("headers").getAsJsonObject();
      for (Map.Entry<String, JsonElement> entry : paramValues.entrySet()) {
        headers.put(entry.getKey(), QueryManager.parseExpressionValue(entity, entry.getValue(), getCustomValues()));
      }
    }

    if (isOData) {
      if (!headers.containsKey(TokenUtils.AUTH_HEADER_NAME) || headers.get(TokenUtils.AUTH_HEADER_NAME).isEmptyOrNull()) {
        headers.put(TokenUtils.AUTH_HEADER_NAME, Var.valueOf(RestClient.getRestClient().getToken()));
      }
      if (!headers.containsKey("Accept") || headers.get("Accept").isEmptyOrNull()) {
        headers.put("Accept", Var.valueOf("application/json"));
      }
      if (!headers.containsKey("Content-Type") || headers.get("Content-Type").isEmptyOrNull()) {
        headers.put("Content-Type", Var.valueOf("application/json"));
      }
    }

    return headers;
  }

  private String getURL(Object entity) {
    try {
      Map<String, Var> customValues = getCustomValues();

      String url = query.get("endpoint").getAsJsonObject().get("url").getAsString();

      Var varEntity = Var.valueOf(entity);

      String id = "";

      JsonElement params = query.get("endpoint").getAsJsonObject().get(method).getAsJsonObject().get("parameters");
      if (!isSOAP) {
        if (entity != null) {
          for (String name : uriInfo.getTargetEntitySet().getEntityType().getPropertyNames()) {
            EdmSimplePropertyImplProv property = (EdmSimplePropertyImplProv) uriInfo.getTargetEntitySet().getEntityType().getProperty(name);
            if (property.getProperty().isOriginalId()) {
              Var value = Var.valueOf(varEntity.get(property.getName()));
              if (isOData) {
                if (id.length() > 0) {
                  id += ",";
                }
                id += property.getName() + "=";
                if (value.isNumber() || value.isBoolean()) {
                  id += value.getObjectAsString();
                } else {
                  id += "'" + value.getObjectAsString() + "'";
                }
              } else {
                if (!id.isEmpty()) {
                  id += "/";
                }
                id += value.getObjectAsString();
              }
            }
          }

          customValues.put("primaryKeys", Var.valueOf(id));
          customValues.put("primaryKey", Var.valueOf(id));

          if (!id.isEmpty()) {
            if (isOData) {
              id = "(" + id + ")";
            } else {
              id = "/" + id;
            }
          }


          if (!JsonUtil.isNull(params) && !JsonUtil.isNull(params.getAsJsonObject().get("$url"))) {
            Var urlPath = QueryManager.parseExpressionValue(query, entity, params.getAsJsonObject().get("$url"), customValues);
            url += urlPath.getObjectAsString();
          } else {
            if (method.equals("PUT") || method.equals("DELETE")) {
              url += id;
            }
          }
        }
      }

      List<String> tokens = Functions.parseTokens(url);

      tokens.sort(Comparator.reverseOrder());

      for (String token : tokens) {
        if (token.startsWith(":")) {
          String name = token.substring(1);
          url = url.replaceFirst(":" + name, QueryManager.getParameterValue(query, name, entity, customValues).getObjectAsString());
        }
      }

      if (!JsonUtil.isNull(params) && !isSOAP) {
        URIBuilder uri = new URIBuilder(url);
        for (Map.Entry<String, JsonElement> entry : params.getAsJsonObject().entrySet()) {
          if (!entry.getKey().equals("$datapath") && !entry.getKey().equals("$url")) {
            Var value = QueryManager.parseExpressionValue(query, entity, entry.getValue(), customValues);
            if (!value.isEmptyOrNull()) {
              String key = entry.getKey();
              if (key.startsWith("$") && key.endsWith("$")) {
                key = key.substring(1, key.length() - 1);
                Var keyValue = QueryManager.parseExpressionValue(query, entity, key, customValues);
                key = keyValue.getObjectAsString();
              }
              if (StringUtils.isNotBlank(key)) {
                uri.addParameter(key, value.getObjectAsString());
              }
            }
          }
        }
        url = uri.toString();
      }

      String queryString = RestClient.getRestClient().getRequest().getQueryString();

      if (isOData && !isSOAP && StringUtils.isNotEmpty(queryString)) {
        URIBuilder uri = new URIBuilder(url);
        URIBuilder uriQs = new URIBuilder("http://empty?" + queryString);
        for (NameValuePair entry : uriQs.getQueryParams()) {
          uri.addParameter(entry.getName(), entry.getValue());
        }
        url = uri.toString();
      }

      return url;
    } catch (Throwable e) {
      throw new RuntimeException(e);
    }
  }


  public void checkError(DocumentContext element) {
    try {
      Object message = element.read("$.error.message.value");
      throw new RuntimeException(message.toString());
    } catch (PathNotFoundException e) {
      try {
        Object message = element.read("$.error.error");
        throw new RuntimeException(message.toString());
      } catch (PathNotFoundException e2) {

      }
    }
  }

  private Var callSoap(Object ds) {
    SOAPUtil soap = new SOAPUtil();
    JsonElement path = query.get("endpoint").getAsJsonObject().get(method);
    if (!QueryManager.isNull(path) && !JsonUtil.isNull(path.getAsJsonObject().get("name"))) {
      String function = path.getAsJsonObject().get("name").getAsString();
      SOAPUtil.ServiceMetadata metadata = soap.getSOAPMetadata(getURL(ds), function);
      JsonObject parameters = path.getAsJsonObject().get("parameters").getAsJsonObject();
      if (metadata.functions.size() > 0) {
        Object[] params = new Object[metadata.functions.get(0).inputs.length];
        int i = 0;
        for (Field field : metadata.functions.get(0).inputs) {
          for (Map.Entry<String, JsonElement> entry : parameters.entrySet()) {
            if (field.getName().equals(entry.getKey())) {
              Var value = QueryManager.parseExpressionValue(query, ds, entry.getValue(), getCustomValues());
              params[i] = value.getObject(field.getType());
            }
          }
          i++;
        }

        return Var.valueOf(soap.call(function, getHeaders(ds), params));
      }
    }

    return Var.VAR_NULL;
  }

  private JsonObject getQueryParameters(String method) {
    JsonElement parameters = query.get("endpoint").getAsJsonObject().get(method).getAsJsonObject().get("parameters");
    if (parameters == null) {
      return new JsonObject();
    }
    return parameters.getAsJsonObject();
  }

  @Override
  public List getResultList() {

    try {
      CURRENT_REST_QUERY.set(this);

      Var result = null;

      if (lastResult != null) {
        result = lastResult;
      }

      Map<String, Var> custom = getCustomValues();

      if (result == null) {

        DocumentContext context = null;
        if (isSOAP) {
          result = callSoap(null);
        } else {
          try {
            Map<String, Var> headers = getHeaders(null);

            String content = Operations.getContentFromURL(Var.valueOf(method.equals("FILTER") ? "GET" : method),
                Var.valueOf(headers.get("Content-Type")),
                Var.valueOf(getURL(null)),
                Var.VAR_NULL,
                Var.valueOf(headers),
                Var.VAR_NULL,
                Var.VAR_NULL
            ).getObjectAsString();

            if (JsonUtil.isJson(content)) {
              context = JsonPath.parse(content);

              checkError(context);
            } else {
              JsonObject object = new JsonObject();
              object.addProperty("content", content);
              context = JsonPath.parse(object.toString());
            }
          } catch (Exception e) {
            throw new RuntimeException(e);
          }

          JsonObject parameters = getQueryParameters(method);

          JsonElement path = parameters.get("$datapath");

          if (isOData && JsonUtil.isNullOrBlank(path) && (method.equals("GET") || method.equals("FILTER"))) {
            path = new JsonParser().parse("\"\\\"$.d.results\\\"\"");
          }
          if (!JsonUtil.isNullOrBlank(path)) {
            result = Var.valueOf(context.read(QueryManager.parseExpressionValue(query, null, path, custom).getObjectAsString()));
          } else {
            result = Var.valueOf(context.read("$"));
          }

          JsonElement countPath = getQueryParameters("COUNT").get("$datapath");
          if (isOData && JsonUtil.isNullOrBlank(countPath)) {
            countPath = new JsonParser().parse("\"\\\"$.d.__count\\\"\"");
          }
          if (!JsonUtil.isNullOrBlank(countPath)) {
            try {
              this.count = Var.valueOf(context.read(QueryManager.parseExpressionValue(query, null, countPath, custom).getObjectAsString())).getObjectAsLong();
            } catch (Exception e) {
              this.count = (long) result.size();
            }
          } else {
            this.count = (long) result.size();
          }
        }

      }

      addFields(null, result, custom);

      if (result.isNull()) {
        return Collections.emptyList();
      }

      if (!QueryManager.isNull(query.get("baseEntity")) && lastResult == null) {
        try {
          parameters.put("baseEntity", query.get("baseEntity").getAsString());
          lastResult = result;
          Class classType = Class.forName(query.get("baseEntity").getAsString());
          List items = new LinkedList();
          for (Object o : result.getObjectAsList()) {
            if (o instanceof Map) {
              convertMap((Map) o);
            }
            Object item = mapper.convertValue(o, classType);
            items.add(item);
          }

          return items;
        } catch (Exception e) {
          throw new RuntimeException(e);
        }
      }

      lastResult = result;
      if (result.getObject() instanceof Map) {
        convertMap((Map) result.getObject());
        List<Var> list = new LinkedList<>();
        list.add(result);

        if (count == null) {
          count = 1L;
        }
        return list;
      }
      Collection list = result.getObjectAsVarList(LinkedList.class);
      if (count == null) {
        count = (long) list.size();
      }

      for (Object object : list) {
        if (object instanceof Map) {
          convertMap((Map) object);
        } else if (object instanceof Var && ((Var) object).getObject() instanceof Map) {
          convertMap((Map) ((Var) object).getObject());
        }
      }

      return (LinkedList) list;
    } finally {
      CURRENT_REST_QUERY.remove();
    }
  }

  private Map convertMap(Map map) {
    try {
      for (String name : uriInfo.getTargetEntitySet().getEntityType().getPropertyNames()) {
        EdmSimplePropertyImplProv property = (EdmSimplePropertyImplProv) uriInfo.getTargetEntitySet().getEntityType().getProperty(name);
        if (map.containsKey(name)) {
          Object value = map.get(name);
          if (value instanceof String) {
            Object valueObj = ((AbstractSimpleType) property.getType()).valueOfString(value.toString(), EdmLiteralKind.JSON, property.getFacets(), ((JPAEdmMappingImpl) property.getMapping()).getOriginaType());
            map.put(name, valueObj);
          }
        }
      }
    } catch (Exception e) {
      e.printStackTrace();
    }

    return map;
  }

  private void addFields(Object ds, Var result, Map<String, Var> custom) {
    JsonObject fields = query.get("defaultValuesProperties").getAsJsonObject();
    for (Map.Entry<String, JsonElement> entry : fields.entrySet()) {
      if (!JsonUtil.isNull(entry.getValue().getAsJsonObject().get("path"))) {
        String fieldPath = QueryManager.parseExpressionValue(query, ds, entry.getValue().getAsJsonObject().get("path"), custom).getObjectAsString();
        if (StringUtils.isNotBlank(fieldPath)) {
          addField(result.getObject(), fieldPath, entry.getKey());
        }
      }
    }
  }

  private void addField(Object object, String path, String field) {
    if (object instanceof List) {
      for (Object o : (List) object) {
        addField(o, path, field);
      }
    } else if (object instanceof Map) {
      try {
        DocumentContext context = JsonPath.parse(object);
        Object value = context.read(path);
        ((Map) object).put(field, value);
      } catch (Exception e) {
        //NoCommand
      }
    }
  }

  public boolean delete(Object entity) {
    try {
      Map<String, Var> headers = getHeaders(entity);

      if (isSOAP) {
        callSoap(entity);
      } else {
        DocumentContext context = JsonPath.parse(
            Operations.getContentFromURL(Var.valueOf(RestClient.getRestClient().getMethod()),
                Var.valueOf(headers.get("Content-Type")),
                Var.valueOf(getURL(entity)),
                Var.VAR_NULL,
                Var.valueOf(headers),
                Var.VAR_NULL,
                Var.VAR_NULL
            ));
        checkError(context);
      }

      return true;
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  public Object save(Object entity) {
    try {

      Map<String, Var> headers = getHeaders(entity);

      JsonObject json = new JsonObject();
      Var varEntity = Var.valueOf(entity);

      JsonObject fields = query.get("defaultValuesProperties").getAsJsonObject();
      JsonObject calcFields = query.get("calcFields").getAsJsonObject();
      boolean hasPath = false;
      boolean hasCalc = false;
      Boolean hasObjectKey = false;
      for (String name : fields.keySet()) {
        if (name.equals(ODataJPAConfig.COMPOSITE_KEY_NAME)) {
          hasObjectKey = true;
          break;
        }
      }
      LinkedList<String> keySet = varEntity.keySet();

      for (String name : uriInfo.getTargetEntitySet().getEntityType().getPropertyNames()) {
        JsonElement value = fields.get(name);
        if (!JsonUtil.isNull(value)) {
          if (!JsonUtil.isNull(value.getAsJsonObject().get("path"))) {
            hasPath = true;
            continue;
          }
        }

        if (!JsonUtil.isNull(calcFields)) {
          JsonElement calc = calcFields.get(name);
          if (!JsonUtil.isNull(calc)) {
            hasCalc = true;
            continue;
          }
        }

        if (hasObjectKey || !name.equals(ODataJPAConfig.COMPOSITE_KEY_NAME)) {
          if (isOData) {
            //O VirtualClass guarda os ids como lowercase
            if (keySet.contains(name) || keySet.contains(name.toLowerCase())) {
              json.add(name, Var.valueOf(varEntity.get(name)).getObjectAsJsonElement());
            }
          } else {
            json.add(name, Var.valueOf(varEntity.get(name)).getObjectAsJsonElement());
          }
        }
      }
      Var result = null;

      if (isSOAP) {
        result = callSoap(entity);
      } else {

        result = Operations.getContentFromURL(
            Var.valueOf(method),
            Var.valueOf(headers.get("Content-Type")),
            Var.valueOf(getURL(entity)),
            Var.VAR_NULL,
            Var.valueOf(headers),
            Var.VAR_NULL,
            Var.valueOf(json.toString())
        );

        DocumentContext context = JsonPath.parse(result.getObjectAsString());

        checkError(context);

        JsonElement path = getQueryParameters(method).get("$datapath");
        if (isOData && JsonUtil.isNullOrBlank(path) && (method.equals("POST") || method.equals("PUT"))) {
          path = new JsonParser().parse("\"\\\"$.d\\\"\"");
        }
        if (!JsonUtil.isNullOrBlank(path)) {
          result = Var.valueOf(context.read(QueryManager.parseExpressionValue(query, entity, path, getCustomValues()).getObjectAsString()));
        } else {
          result = Var.valueOf(context.read("$"));
        }
      }

      boolean hasEntity = false;
      for (String name : uriInfo.getTargetEntitySet().getEntityType().getPropertyNames()) {
        EdmSimplePropertyImplProv property = (EdmSimplePropertyImplProv) uriInfo.getTargetEntitySet().getEntityType().getProperty(name);
        if (property.getProperty().isOriginalId()) {
          if (!Var.valueOf(result.get(property.getName())).isNull()) {
            hasEntity = true;
            break;
          }
        }
      }

      if (!hasEntity || hasPath || hasCalc) {
        result = Var.valueOf(entity);
      }

      if (result.getObject() instanceof Map) {
        convertMap((Map) result.getObject());
      }

      if (!QueryManager.isNull(query.get("baseEntity"))) {
        try {
          parameters.put("baseEntity", query.get("baseEntity").getAsString());
          Class classType = Class.forName(query.get("baseEntity").getAsString());
          Object item = mapper.convertValue(result, classType);

          return item;
        } catch (Exception e) {
          throw new RuntimeException(e);
        }
      }

      if (result.isNull()) {
        return Var.valueOf(entity);
      }
      return Var.valueOf(result);
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  public void setUriInfo(UriInfo uriInfo) {
    this.uriInfo = uriInfo;
  }

  public Var getLastResult() {
    return lastResult;
  }

  @Override
  public Long count() {
    return count;
  }

  public void setLastResult(Var value) {
    this.lastResult = value;
  }

  @Override
  public Object getSingleResult() {
    return getResultList().get(0);
  }

  @Override
  public int executeUpdate() {
    return 0;
  }

  @Override
  public Query setMaxResults(int maxResult) {
    parameters.put("MaxResults", maxResult);
    return this;
  }

  @Override
  public int getMaxResults() {
    return parameters.containsKey("MaxResults") ? (int) parameters.get("MaxResults") : -1;
  }

  @Override
  public Query setFirstResult(int startPosition) {
    parameters.put("FirstResult", startPosition);
    return this;
  }

  @Override
  public int getFirstResult() {
    return parameters.containsKey("FirstResult") ? (int) parameters.get("FirstResult") : -1;
  }

  @Override
  public Query setHint(String hintName, Object value) {
    parameters.put("hintName", value);
    return this;
  }

  @Override
  public Map<String, Object> getHints() {
    return null;
  }

  private void putParameter(int index, Object value) {
    parameters.put(String.valueOf(index), value);
  }

  @Override
  public <T> Query setParameter(Parameter<T> param, T value) {
    putParameter(param.getPosition(), value);
    return this;
  }

  @Override
  public Query setParameter(Parameter<Calendar> param, Calendar value, TemporalType temporalType) {
    putParameter(param.getPosition(), value);
    return this;
  }

  @Override
  public Query setParameter(Parameter<Date> param, Date value, TemporalType temporalType) {
    putParameter(param.getPosition(), value);
    return this;
  }

  @Override
  public Query setParameter(String name, Object value) {
    parameters.put(name, value);
    return this;
  }

  @Override
  public Query setParameter(String name, Calendar value, TemporalType temporalType) {
    parameters.put(name, value);
    return this;
  }

  @Override
  public Query setParameter(String name, Date value, TemporalType temporalType) {
    parameters.put(name, value);
    return this;
  }

  @Override
  public Query setParameter(int position, Object value) {
    putParameter(position, value);
    return this;
  }

  @Override
  public Query setParameter(int position, Calendar value, TemporalType temporalType) {
    putParameter(position, value);
    return this;
  }

  @Override
  public Query setParameter(int position, Date value, TemporalType temporalType) {
    putParameter(position, value);
    return this;
  }

  @Override
  public Set<Parameter<?>> getParameters() {
    return null;
  }

  @Override
  public Parameter<?> getParameter(String name) {
    return null;
  }

  @Override
  public <T> Parameter<T> getParameter(String name, Class<T> type) {
    return null;
  }

  @Override
  public Parameter<?> getParameter(int position) {
    return null;
  }

  @Override
  public <T> Parameter<T> getParameter(int position, Class<T> type) {
    return null;
  }

  @Override
  public boolean isBound(Parameter<?> param) {
    return false;
  }

  @Override
  public <T> T getParameterValue(Parameter<T> param) {
    return null;
  }

  @Override
  public Object getParameterValue(String name) {
    return parameters.get(name);
  }

  @Override
  public Object getParameterValue(int position) {
    return null;
  }

  @Override
  public Query setFlushMode(FlushModeType flushMode) {
    return null;
  }

  @Override
  public FlushModeType getFlushMode() {
    return null;
  }

  @Override
  public Query setLockMode(LockModeType lockMode) {
    return null;
  }

  @Override
  public LockModeType getLockMode() {
    return null;
  }

  @Override
  public <T> T unwrap(Class<T> cls) {
    return null;
  }

  public UriInfo getUriInfo() {
    return uriInfo;
  }
}
