package cronapi.odata.server;

import com.google.gson.JsonObject;
import cronapi.Var;
import cronapi.util.Functions;
import org.apache.olingo.odata2.api.uri.UriInfo;
import org.apache.olingo.odata2.api.uri.expression.BinaryExpression;
import org.apache.olingo.odata2.api.uri.expression.MethodExpression;

import java.util.LinkedHashMap;
import java.util.Map;

public class ODataQuery {

  protected JsonObject query;
  protected String method;
  protected String queryStatement;
  protected Map<String, Object> parameters = new LinkedHashMap<>();
  protected String type;
  protected String originalFilter;
  protected String entityName;
  protected UriInfo uriInfo;

  public ODataQuery(JsonObject query, String method, String type, String queryStatement, UriInfo uriInfo) {
    this.type = type;
    this.originalFilter = originalFilter;
    this.entityName = entityName;
    this.parameters.put("queryType", type);
    this.parameters.put("queryStatement", queryStatement);
    this.parameters.put("queryFilter", originalFilter);
    this.query = query;
    this.method = method;
    this.queryStatement = queryStatement;
    this.uriInfo = uriInfo;
    this.originalFilter = (uriInfo.getFilter() != null ? uriInfo.getFilter().getExpressionString() : "");
    try {
      this.entityName = uriInfo.getTargetEntitySet().getName();
    } catch (Throwable e) {
      throw new RuntimeException(e);
    }
  }

  public Map<String, Var> getCustomValues() {
    Map<String, Var> customValues = new LinkedHashMap<>();
    customValues.put("entityName", Var.valueOf(this.entityName));

    customValues.put("queryType", Var.valueOf(type));
    customValues.put("queryStatement", Var.valueOf(queryStatement));
    customValues.put("queryFilter", Var.valueOf(originalFilter));
    customValues.put("queryParameters", Var.valueOf(parameters));
    customValues.put("querySort", Var.VAR_NULL);
    customValues.put("querySortOrder", Var.VAR_NULL);
    if (uriInfo.getOrderBy() != null && !uriInfo.getOrderBy().getOrders().isEmpty()) {
      customValues.put("querySort", Var.valueOf(uriInfo.getOrderBy().getOrders().get(0).getExpression().getUriLiteral()));
      customValues.put("querySortOrder", Var.valueOf(uriInfo.getOrderBy().getOrders().get(0).getSortOrder().toString()));
    }

    customValues.put("simpleFilter", Var.VAR_NULL);
    customValues.put("simpleFilterField", Var.VAR_NULL);

    if (uriInfo.getFilter() != null && !uriInfo.getFilter().getExpressionString().isEmpty()) {
      if (uriInfo.getFilter().getExpression() instanceof BinaryExpression) {
        BinaryExpression binaryExpression = (BinaryExpression) uriInfo.getFilter().getExpression();
        customValues.put("simpleFilter", Var.valueOf(Functions.unquote(binaryExpression.getRightOperand().getUriLiteral())));
        customValues.put("simpleFilterField", Var.valueOf(binaryExpression.getLeftOperand().getUriLiteral()));
      }

      if (uriInfo.getFilter().getExpression() instanceof MethodExpression) {
        MethodExpression binaryExpression = (MethodExpression) uriInfo.getFilter().getExpression();
        customValues.put("simpleFilter", Var.valueOf(Functions.unquote(binaryExpression.getParameters().get(0).getUriLiteral())));
        customValues.put("simpleFilterField", Var.valueOf(binaryExpression.getParameters().get(1).getUriLiteral()));
      }
    }

    Integer firstResult = (Integer) parameters.get("FirstResult");
    Integer maxResult = (Integer) parameters.get("MaxResults");

    customValues.put("page", Var.valueOf(0));

    if (firstResult != null && maxResult != null) {
      Integer page = (firstResult / maxResult) + 1;
      customValues.put("page", Var.valueOf(page));
    }
    customValues.put("top", Var.valueOf(maxResult));
    customValues.put("skip", Var.valueOf(firstResult));
    customValues.put("limit", Var.valueOf(maxResult));
    customValues.put("offset", Var.valueOf(firstResult));
    customValues.put("per_page", Var.valueOf(maxResult));

    return customValues;
  }

  public JsonObject getQuery() {
    return query;
  }

  public void setQuery(JsonObject query) {
    this.query = query;
  }

  public String getMethod() {
    return method;
  }

  public void setMethod(String method) {
    this.method = method;
  }

  public String getQueryStatement() {
    return queryStatement;
  }

  public void setQueryStatement(String queryStatement) {
    this.queryStatement = queryStatement;
  }

  public String getType() {
    return type;
  }

  public void setType(String type) {
    this.type = type;
  }

  public String getOriginalFilter() {
    return originalFilter;
  }

  public void setOriginalFilter(String originalFilter) {
    this.originalFilter = originalFilter;
  }

  public String getEntityName() {
    return entityName;
  }

  public void setEntityName(String entityName) {
    this.entityName = entityName;
  }

  public UriInfo getUriInfo() {
    return uriInfo;
  }

  public void setUriInfo(UriInfo uriInfo) {
    this.uriInfo = uriInfo;
  }
}
