/*
 * Decompiled with CFR 0.152.
 */
package net.joelinn.quartz.jobstore;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.Set;
import java.util.UUID;
import net.joelinn.quartz.jobstore.RedisJobStoreSchema;
import net.joelinn.quartz.jobstore.RedisTriggerState;
import org.quartz.Calendar;
import org.quartz.CronTrigger;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.Job;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.JobPersistenceException;
import org.quartz.ObjectAlreadyExistsException;
import org.quartz.PersistJobDataAfterExecution;
import org.quartz.SimpleTrigger;
import org.quartz.Trigger;
import org.quartz.TriggerKey;
import org.quartz.impl.JobDetailImpl;
import org.quartz.impl.matchers.GroupMatcher;
import org.quartz.impl.matchers.StringMatcher;
import org.quartz.spi.OperableTrigger;
import org.quartz.spi.SchedulerSignaler;
import org.quartz.spi.TriggerFiredBundle;
import org.quartz.spi.TriggerFiredResult;
import org.quartz.utils.ClassUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Jedis;
import redis.clients.jedis.Pipeline;
import redis.clients.jedis.Response;
import redis.clients.jedis.Tuple;

public class RedisStorage {
    private static final Logger logger = LoggerFactory.getLogger(RedisStorage.class);
    protected static final String TRIGGER_CLASS = "trigger_class";
    protected static final String CALENDAR_CLASS = "calendar_class";
    protected static final String CALENDAR_JSON = "calendar_json";
    protected static final String TRIGGER_NEXT_FIRE_TIME = "nextFireTime";
    protected final RedisJobStoreSchema redisSchema;
    protected final ObjectMapper mapper;
    protected final SchedulerSignaler signaler;
    protected final String schedulerInstanceId;
    protected final int lockTimeout;
    protected final int TRIGGER_LOCK_TIMEOUT = 600000;
    protected int misfireThreshold = 60000;
    protected UUID lockValue;

    public RedisStorage(RedisJobStoreSchema redisSchema, ObjectMapper mapper, SchedulerSignaler signaler, String schedulerInstanceId, int lockTimeout) {
        this.signaler = signaler;
        this.schedulerInstanceId = schedulerInstanceId;
        this.redisSchema = redisSchema;
        this.mapper = mapper;
        this.lockTimeout = lockTimeout;
    }

    public boolean lock(Jedis jedis) {
        boolean lockAcquired;
        UUID lockId = UUID.randomUUID();
        String setResponse = jedis.set(this.redisSchema.lockKey(), lockId.toString(), "NX", "PX", this.lockTimeout);
        boolean bl = lockAcquired = !Strings.isNullOrEmpty((String)setResponse) && setResponse.equals("OK");
        if (lockAcquired) {
            this.lockValue = lockId;
        }
        return lockAcquired;
    }

    public void waitForLock(Jedis jedis) {
        while (!this.lock(jedis)) {
            try {
                logger.debug("Waiting for Redis lock.");
                Thread.sleep(this.randomInt(75, 125));
            }
            catch (InterruptedException e) {
                logger.error("Interrupted while waiting for lock.", (Throwable)e);
            }
        }
    }

    public boolean unlock(Jedis jedis) {
        String currentLock = jedis.get(this.redisSchema.lockKey());
        if (!Strings.isNullOrEmpty((String)currentLock) && UUID.fromString(currentLock).equals(this.lockValue)) {
            jedis.del(this.redisSchema.lockKey());
            return true;
        }
        return false;
    }

    protected int randomInt(int min, int max) {
        return new Random().nextInt(max - min + 1) + min;
    }

    public void clearAllSchedulingData(Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        for (String jobHashKey : jedis.smembers(this.redisSchema.jobsSet())) {
            this.removeJob(this.redisSchema.jobKey(jobHashKey), jedis);
        }
        for (String triggerHashKey : jedis.smembers(this.redisSchema.triggersSet())) {
            this.removeTrigger(this.redisSchema.triggerKey(triggerHashKey), jedis);
        }
        for (String calendarHashKey : jedis.smembers(this.redisSchema.calendarsSet())) {
            this.removeCalendar(this.redisSchema.calendarName(calendarHashKey), jedis);
        }
    }

    public void storeJob(JobDetail jobDetail, boolean replaceExisting, Jedis jedis) throws ObjectAlreadyExistsException {
        String jobHashKey = this.redisSchema.jobHashKey(jobDetail.getKey());
        String jobDataMapHashKey = this.redisSchema.jobDataMapHashKey(jobDetail.getKey());
        String jobGroupSetKey = this.redisSchema.jobGroupSetKey(jobDetail.getKey());
        if (!replaceExisting && jedis.exists(jobHashKey).booleanValue()) {
            throw new ObjectAlreadyExistsException(jobDetail);
        }
        Pipeline pipe = jedis.pipelined();
        pipe.hmset(jobHashKey, (Map)this.mapper.convertValue((Object)jobDetail, (TypeReference)new TypeReference<HashMap<String, String>>(){}));
        if (jobDetail.getJobDataMap() != null && !jobDetail.getJobDataMap().isEmpty()) {
            pipe.hmset(jobDataMapHashKey, this.getStringDataMap(jobDetail.getJobDataMap()));
        }
        pipe.sadd(this.redisSchema.jobsSet(), new String[]{jobHashKey});
        pipe.sadd(this.redisSchema.jobGroupsSet(), new String[]{jobGroupSetKey});
        pipe.sadd(jobGroupSetKey, new String[]{jobHashKey});
        pipe.sync();
    }

    protected Map<String, String> getStringDataMap(JobDataMap jobDataMap) {
        HashMap<String, String> stringMap = new HashMap<String, String>();
        for (Map.Entry entry : jobDataMap.entrySet()) {
            if (entry.getValue() == null) continue;
            stringMap.put((String)entry.getKey(), entry.getValue().toString());
        }
        return stringMap;
    }

    public JobDetail retrieveJob(JobKey jobKey, Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        String jobHashKey = this.redisSchema.jobHashKey(jobKey);
        String jobDataMapHashKey = this.redisSchema.jobDataMapHashKey(jobKey);
        Map jobDetailMap = jedis.hgetAll(jobHashKey);
        if (jobDetailMap == null || jobDetailMap.size() == 0) {
            return null;
        }
        JobDetailImpl jobDetail = (JobDetailImpl)this.mapper.convertValue((Object)jobDetailMap, JobDetailImpl.class);
        jobDetail.setKey(jobKey);
        Map jobData = jedis.hgetAll(jobDataMapHashKey);
        if (jobData != null && !jobData.isEmpty()) {
            JobDataMap jobDataMap = new JobDataMap();
            jobDataMap.putAll(jobData);
            jobDetail.setJobDataMap(jobDataMap);
        }
        return jobDetail;
    }

    public boolean removeJob(JobKey jobKey, Jedis jedis) throws JobPersistenceException {
        String jobHashKey = this.redisSchema.jobHashKey(jobKey);
        String jobDataMapHashKey = this.redisSchema.jobDataMapHashKey(jobKey);
        String jobGroupSetKey = this.redisSchema.jobGroupSetKey(jobKey);
        String jobTriggerSetKey = this.redisSchema.jobTriggersSetKey(jobKey);
        Pipeline pipe = jedis.pipelined();
        Response delJobHashKeyResponse = pipe.del(jobHashKey);
        pipe.del(jobDataMapHashKey);
        pipe.srem(this.redisSchema.jobsSet(), new String[]{jobHashKey});
        pipe.srem(jobGroupSetKey, new String[]{jobHashKey});
        Response jobTriggerSetResponse = pipe.smembers(jobTriggerSetKey);
        pipe.del(jobTriggerSetKey);
        Response jobGroupSetSizeResponse = pipe.scard(jobGroupSetKey);
        pipe.sync();
        if ((Long)jobGroupSetSizeResponse.get() == 0L) {
            jedis.srem(this.redisSchema.jobGroupsSet(), new String[]{jobGroupSetKey});
        }
        pipe = jedis.pipelined();
        for (String triggerHashKey : (Set)jobTriggerSetResponse.get()) {
            TriggerKey triggerKey = this.redisSchema.triggerKey(triggerHashKey);
            String triggerGroupSetKey = this.redisSchema.triggerGroupSetKey(triggerKey);
            this.unsetTriggerState(triggerHashKey, jedis);
            pipe.srem(this.redisSchema.triggersSet(), new String[]{triggerHashKey});
            pipe.srem(this.redisSchema.triggerGroupsSet(), new String[]{triggerGroupSetKey});
            pipe.srem(triggerGroupSetKey, new String[]{triggerHashKey});
            pipe.del(triggerHashKey);
        }
        pipe.sync();
        return (Long)delJobHashKeyResponse.get() == 1L;
    }

    public void storeTrigger(OperableTrigger trigger, boolean replaceExisting, Jedis jedis) throws JobPersistenceException {
        String triggerHashKey = this.redisSchema.triggerHashKey(trigger.getKey());
        String triggerGroupSetKey = this.redisSchema.triggerGroupSetKey(trigger.getKey());
        String jobTriggerSetKey = this.redisSchema.jobTriggersSetKey(trigger.getJobKey());
        if (!(trigger instanceof SimpleTrigger) && !(trigger instanceof CronTrigger)) {
            throw new UnsupportedOperationException("Only SimpleTrigger and CronTrigger are supported.");
        }
        boolean exists = jedis.exists(triggerHashKey);
        if (exists && !replaceExisting) {
            throw new ObjectAlreadyExistsException((Trigger)trigger);
        }
        Map triggerMap = (Map)this.mapper.convertValue((Object)trigger, (TypeReference)new TypeReference<HashMap<String, String>>(){});
        triggerMap.put(TRIGGER_CLASS, trigger.getClass().getName());
        Pipeline pipe = jedis.pipelined();
        pipe.hmset(triggerHashKey, triggerMap);
        pipe.sadd(this.redisSchema.triggersSet(), new String[]{triggerHashKey});
        pipe.sadd(this.redisSchema.triggerGroupsSet(), new String[]{triggerGroupSetKey});
        pipe.sadd(triggerGroupSetKey, new String[]{triggerHashKey});
        pipe.sadd(jobTriggerSetKey, new String[]{triggerHashKey});
        if (trigger.getCalendarName() != null && !trigger.getCalendarName().isEmpty()) {
            String calendarTriggersSetKey = this.redisSchema.calendarTriggersSetKey(trigger.getCalendarName());
            pipe.sadd(calendarTriggersSetKey, new String[]{triggerHashKey});
        }
        pipe.sync();
        if (exists) {
            this.unsetTriggerState(triggerHashKey, jedis);
        }
        pipe = jedis.pipelined();
        Response triggerPausedResponse = pipe.sismember(this.redisSchema.pausedTriggerGroupsSet(), triggerGroupSetKey);
        Response jobPausedResponse = pipe.sismember(this.redisSchema.pausedJobGroupsSet(), this.redisSchema.jobGroupSetKey(trigger.getJobKey()));
        pipe.sync();
        if (((Boolean)triggerPausedResponse.get()).booleanValue() || ((Boolean)jobPausedResponse.get()).booleanValue()) {
            long nextFireTime = trigger.getNextFireTime() != null ? trigger.getNextFireTime().getTime() : -1L;
            String jobHashKey = this.redisSchema.jobHashKey(trigger.getJobKey());
            if (jedis.sismember(this.redisSchema.blockedJobsSet(), jobHashKey).booleanValue()) {
                this.setTriggerState(RedisTriggerState.PAUSED_BLOCKED, nextFireTime, triggerHashKey, jedis);
            } else {
                this.setTriggerState(RedisTriggerState.PAUSED, nextFireTime, triggerHashKey, jedis);
            }
        } else if (trigger.getNextFireTime() != null) {
            this.setTriggerState(RedisTriggerState.WAITING, trigger.getNextFireTime().getTime(), triggerHashKey, jedis);
        }
    }

    public boolean removeTrigger(TriggerKey triggerKey, Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        JobDetail job;
        String triggerHashKey = this.redisSchema.triggerHashKey(triggerKey);
        String triggerGroupSetKey = this.redisSchema.triggerGroupSetKey(triggerKey);
        if (!jedis.exists(triggerHashKey).booleanValue()) {
            return false;
        }
        OperableTrigger trigger = this.retrieveTrigger(triggerKey, jedis);
        String jobHashKey = this.redisSchema.jobHashKey(trigger.getJobKey());
        String jobTriggerSetKey = this.redisSchema.jobTriggersSetKey(trigger.getJobKey());
        Pipeline pipe = jedis.pipelined();
        pipe.srem(this.redisSchema.triggersSet(), new String[]{triggerHashKey});
        pipe.srem(triggerGroupSetKey, new String[]{triggerHashKey});
        pipe.srem(jobTriggerSetKey, new String[]{triggerHashKey});
        pipe.sync();
        if (jedis.scard(triggerGroupSetKey) == 0L) {
            jedis.srem(this.redisSchema.triggerGroupsSet(), new String[]{triggerGroupSetKey});
        }
        pipe = jedis.pipelined();
        Response jobTriggerSetKeySizeResponse = pipe.scard(jobTriggerSetKey);
        Response jobExistsResponse = pipe.exists(jobHashKey);
        pipe.sync();
        if ((Long)jobTriggerSetKeySizeResponse.get() == 0L && ((Boolean)jobExistsResponse.get()).booleanValue() && !(job = this.retrieveJob(trigger.getJobKey(), jedis)).isDurable()) {
            this.removeJob(job.getKey(), jedis);
            this.signaler.notifySchedulerListenersJobDeleted(job.getKey());
        }
        if (Strings.isNullOrEmpty((String)trigger.getCalendarName())) {
            jedis.srem(this.redisSchema.calendarTriggersSetKey(trigger.getCalendarName()), new String[]{triggerHashKey});
        }
        this.unsetTriggerState(triggerHashKey, jedis);
        jedis.del(triggerHashKey);
        return true;
    }

    public boolean replaceTrigger(TriggerKey triggerKey, OperableTrigger newTrigger, Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        boolean found;
        OperableTrigger oldTrigger = this.retrieveTrigger(triggerKey, jedis);
        boolean bl = found = oldTrigger != null;
        if (found) {
            if (!oldTrigger.getJobKey().equals((Object)newTrigger.getJobKey())) {
                throw new JobPersistenceException("New trigger is not related to the same job as the old trigger.");
            }
            this.removeTrigger(triggerKey, jedis);
            this.storeTrigger(newTrigger, false, jedis);
        }
        return found;
    }

    public OperableTrigger retrieveTrigger(TriggerKey triggerKey, Jedis jedis) throws JobPersistenceException {
        Class<?> triggerClass;
        String triggerHashKey = this.redisSchema.triggerHashKey(triggerKey);
        Map triggerMap = jedis.hgetAll(triggerHashKey);
        if (triggerMap == null || triggerMap.isEmpty()) {
            logger.debug(String.format("No trigger exists for key %s", triggerHashKey));
            return null;
        }
        try {
            triggerClass = Class.forName((String)triggerMap.get(TRIGGER_CLASS));
        }
        catch (ClassNotFoundException e) {
            throw new JobPersistenceException(String.format("Could not find class %s for trigger.", triggerMap.get(TRIGGER_CLASS)), (Throwable)e);
        }
        triggerMap.remove(TRIGGER_CLASS);
        return (OperableTrigger)this.mapper.convertValue((Object)triggerMap, triggerClass);
    }

    public List<OperableTrigger> getTriggersForJob(JobKey jobKey, Jedis jedis) throws JobPersistenceException {
        String jobTriggerSetKey = this.redisSchema.jobTriggersSetKey(jobKey);
        Set triggerHashKeys = jedis.smembers(jobTriggerSetKey);
        ArrayList<OperableTrigger> triggers = new ArrayList<OperableTrigger>();
        for (String triggerHashKey : triggerHashKeys) {
            triggers.add(this.retrieveTrigger(this.redisSchema.triggerKey(triggerHashKey), jedis));
        }
        return triggers;
    }

    public boolean unsetTriggerState(String triggerHashKey, Jedis jedis) throws JobPersistenceException {
        boolean removed = false;
        Pipeline pipe = jedis.pipelined();
        ArrayList<Response> responses = new ArrayList<Response>(RedisTriggerState.values().length);
        for (RedisTriggerState state : RedisTriggerState.values()) {
            responses.add(pipe.zrem(this.redisSchema.triggerStateKey(state), new String[]{triggerHashKey}));
        }
        pipe.sync();
        for (Response response : responses) {
            removed = (Long)response.get() == 1L;
            if (!removed) continue;
            jedis.del(this.redisSchema.triggerLockKey(this.redisSchema.triggerKey(triggerHashKey)));
            break;
        }
        return removed;
    }

    public boolean setTriggerState(RedisTriggerState state, double score, String triggerHashKey, Jedis jedis) throws JobPersistenceException {
        boolean success = false;
        if (state != null) {
            this.unsetTriggerState(triggerHashKey, jedis);
            success = jedis.zadd(this.redisSchema.triggerStateKey(state), score, triggerHashKey) == 1L;
        }
        return success;
    }

    public boolean checkExists(JobKey jobKey, Jedis jedis) {
        return jedis.exists(this.redisSchema.jobHashKey(jobKey));
    }

    public boolean checkExists(TriggerKey triggerKey, Jedis jedis) {
        return jedis.exists(this.redisSchema.triggerHashKey(triggerKey));
    }

    public void storeCalendar(String name, Calendar calendar, boolean replaceExisting, boolean updateTriggers, Jedis jedis) throws JobPersistenceException {
        String calendarHashKey = this.redisSchema.calendarHashKey(name);
        if (!replaceExisting && jedis.exists(calendarHashKey).booleanValue()) {
            throw new ObjectAlreadyExistsException(String.format("Calendar with key %s already exists.", calendarHashKey));
        }
        HashMap<String, String> calendarMap = new HashMap<String, String>();
        calendarMap.put(CALENDAR_CLASS, calendar.getClass().getName());
        try {
            calendarMap.put(CALENDAR_JSON, this.mapper.writeValueAsString((Object)calendar));
        }
        catch (JsonProcessingException e) {
            throw new JobPersistenceException("Unable to serialize calendar.", (Throwable)e);
        }
        Pipeline pipe = jedis.pipelined();
        pipe.hmset(calendarHashKey, calendarMap);
        pipe.sadd(this.redisSchema.calendarsSet(), new String[]{calendarHashKey});
        pipe.sync();
        if (updateTriggers) {
            String calendarTriggersSetKey = this.redisSchema.calendarTriggersSetKey(name);
            Set triggerHashKeys = jedis.smembers(calendarTriggersSetKey);
            for (String triggerHashKey : triggerHashKeys) {
                OperableTrigger trigger = this.retrieveTrigger(this.redisSchema.triggerKey(triggerHashKey), jedis);
                long removed = jedis.zrem(this.redisSchema.triggerStateKey(RedisTriggerState.WAITING), new String[]{triggerHashKey});
                trigger.updateWithNewCalendar(calendar, (long)this.misfireThreshold);
                if (removed != 1L) continue;
                this.setTriggerState(RedisTriggerState.WAITING, trigger.getNextFireTime().getTime(), triggerHashKey, jedis);
            }
        }
    }

    public boolean removeCalendar(String calendarName, Jedis jedis) throws JobPersistenceException {
        String calendarTriggersSetKey = this.redisSchema.calendarTriggersSetKey(calendarName);
        if (jedis.scard(calendarTriggersSetKey) > 0L) {
            throw new JobPersistenceException(String.format("There are triggers pointing to calendar %s, so it cannot be removed.", calendarName));
        }
        String calendarHashKey = this.redisSchema.calendarHashKey(calendarName);
        Pipeline pipe = jedis.pipelined();
        Response deleteResponse = pipe.del(calendarHashKey);
        pipe.srem(this.redisSchema.calendarsSet(), new String[]{calendarHashKey});
        pipe.sync();
        return (Long)deleteResponse.get() == 1L;
    }

    public Calendar retrieveCalendar(String name, Jedis jedis) throws JobPersistenceException {
        Calendar calendar;
        String calendarHashKey = this.redisSchema.calendarHashKey(name);
        try {
            Map calendarMap = jedis.hgetAll(calendarHashKey);
            if (calendarMap == null || calendarMap.isEmpty()) {
                return null;
            }
            Class<?> calendarClass = Class.forName((String)calendarMap.get(CALENDAR_CLASS));
            calendar = (Calendar)this.mapper.readValue((String)calendarMap.get(CALENDAR_JSON), calendarClass);
        }
        catch (ClassNotFoundException e) {
            logger.error("Class not found for calendar " + name);
            throw new JobPersistenceException(e.getMessage(), (Throwable)e);
        }
        catch (IOException e) {
            logger.error("Unable to deserialize calendar json for calendar " + name);
            throw new JobPersistenceException(e.getMessage(), (Throwable)e);
        }
        return calendar;
    }

    public int getNumberOfJobs(Jedis jedis) {
        return jedis.scard(this.redisSchema.jobsSet()).intValue();
    }

    public int getNumberOfTriggers(Jedis jedis) {
        return jedis.scard(this.redisSchema.triggersSet()).intValue();
    }

    public int getNumberOfCalendars(Jedis jedis) {
        return jedis.scard(this.redisSchema.calendarsSet()).intValue();
    }

    public Set<JobKey> getJobKeys(GroupMatcher<JobKey> matcher, Jedis jedis) {
        HashSet<JobKey> jobKeys;
        block5: {
            block4: {
                jobKeys = new HashSet<JobKey>();
                if (matcher.getCompareWithOperator() != StringMatcher.StringOperatorName.EQUALS) break block4;
                String jobGroupSetKey = this.redisSchema.jobGroupSetKey(new JobKey("", matcher.getCompareToValue()));
                Set jobs = jedis.smembers(jobGroupSetKey);
                if (jobs == null) break block5;
                for (String job : jobs) {
                    jobKeys.add(this.redisSchema.jobKey(job));
                }
                break block5;
            }
            ArrayList<Response> jobGroups = new ArrayList<Response>();
            Pipeline pipe = jedis.pipelined();
            for (String jobGroupSetKey : jedis.smembers(this.redisSchema.jobGroupsSet())) {
                if (!matcher.getCompareWithOperator().evaluate(this.redisSchema.jobGroup(jobGroupSetKey), matcher.getCompareToValue())) continue;
                jobGroups.add(pipe.smembers(jobGroupSetKey));
            }
            pipe.sync();
            for (Response jobGroup : jobGroups) {
                if (jobGroup.get() == null) continue;
                for (String job : (Set)jobGroup.get()) {
                    jobKeys.add(this.redisSchema.jobKey(job));
                }
            }
        }
        return jobKeys;
    }

    public Set<TriggerKey> getTriggerKeys(GroupMatcher<TriggerKey> matcher, Jedis jedis) {
        HashSet<TriggerKey> triggerKeys;
        block5: {
            block4: {
                triggerKeys = new HashSet<TriggerKey>();
                if (matcher.getCompareWithOperator() != StringMatcher.StringOperatorName.EQUALS) break block4;
                String triggerGroupSetKey = this.redisSchema.triggerGroupSetKey(new TriggerKey("", matcher.getCompareToValue()));
                Set triggers = jedis.smembers(triggerGroupSetKey);
                if (triggers == null) break block5;
                for (String trigger : triggers) {
                    triggerKeys.add(this.redisSchema.triggerKey(trigger));
                }
                break block5;
            }
            ArrayList<Response> triggerGroups = new ArrayList<Response>();
            Pipeline pipe = jedis.pipelined();
            for (String triggerGroupSetKey : jedis.smembers(this.redisSchema.triggerGroupsSet())) {
                if (!matcher.getCompareWithOperator().evaluate(this.redisSchema.triggerGroup(triggerGroupSetKey), matcher.getCompareToValue())) continue;
                triggerGroups.add(pipe.smembers(triggerGroupSetKey));
            }
            pipe.sync();
            for (Response triggerGroup : triggerGroups) {
                if (triggerGroup.get() == null) continue;
                for (String trigger : (Set)triggerGroup.get()) {
                    triggerKeys.add(this.redisSchema.triggerKey(trigger));
                }
            }
        }
        return triggerKeys;
    }

    public List<String> getJobGroupNames(Jedis jedis) {
        Set groupsSet = jedis.smembers(this.redisSchema.jobGroupsSet());
        ArrayList<String> groups = new ArrayList<String>(groupsSet.size());
        for (String group : groupsSet) {
            groups.add(this.redisSchema.jobGroup(group));
        }
        return groups;
    }

    public List<String> getTriggerGroupNames(Jedis jedis) {
        Set groupsSet = jedis.smembers(this.redisSchema.triggerGroupsSet());
        ArrayList<String> groups = new ArrayList<String>(groupsSet.size());
        for (String group : groupsSet) {
            groups.add(this.redisSchema.triggerGroup(group));
        }
        return groups;
    }

    public List<String> getCalendarNames(Jedis jedis) {
        Set calendarsSet = jedis.smembers(this.redisSchema.calendarsSet());
        ArrayList<String> calendars = new ArrayList<String>(calendarsSet.size());
        for (String group : calendarsSet) {
            calendars.add(this.redisSchema.calendarName(group));
        }
        return calendars;
    }

    public Trigger.TriggerState getTriggerState(TriggerKey triggerKey, Jedis jedis) {
        String triggerHashKey = this.redisSchema.triggerHashKey(triggerKey);
        Pipeline pipe = jedis.pipelined();
        HashMap<RedisTriggerState, Response> scores = new HashMap<RedisTriggerState, Response>(RedisTriggerState.values().length);
        for (RedisTriggerState redisTriggerState : RedisTriggerState.values()) {
            scores.put(redisTriggerState, pipe.zscore(this.redisSchema.triggerStateKey(redisTriggerState), triggerHashKey));
        }
        pipe.sync();
        for (Map.Entry entry : scores.entrySet()) {
            if (((Response)entry.getValue()).get() == null) continue;
            return ((RedisTriggerState)((Object)entry.getKey())).getTriggerState();
        }
        return Trigger.TriggerState.NONE;
    }

    public void pauseTrigger(TriggerKey triggerKey, Jedis jedis) throws JobPersistenceException {
        long nextFireTime;
        String triggerHashKey = this.redisSchema.triggerHashKey(triggerKey);
        Pipeline pipe = jedis.pipelined();
        Response exists = pipe.exists(triggerHashKey);
        Response completedScore = pipe.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.COMPLETED), triggerHashKey);
        Response nextFireTimeResponse = pipe.hget(triggerHashKey, TRIGGER_NEXT_FIRE_TIME);
        Response blockedScore = pipe.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.BLOCKED), triggerHashKey);
        pipe.sync();
        if (!((Boolean)exists.get()).booleanValue()) {
            return;
        }
        if (completedScore.get() != null) {
            return;
        }
        long l = nextFireTime = nextFireTimeResponse.get() == null || ((String)nextFireTimeResponse.get()).isEmpty() ? -1L : Long.parseLong((String)nextFireTimeResponse.get());
        if (blockedScore.get() != null) {
            this.setTriggerState(RedisTriggerState.PAUSED_BLOCKED, nextFireTime, triggerHashKey, jedis);
        } else {
            this.setTriggerState(RedisTriggerState.PAUSED, nextFireTime, triggerHashKey, jedis);
        }
    }

    public Collection<String> pauseTriggers(GroupMatcher<TriggerKey> matcher, Jedis jedis) throws JobPersistenceException {
        HashSet<String> pausedTriggerGroups = new HashSet<String>();
        if (matcher.getCompareWithOperator() == StringMatcher.StringOperatorName.EQUALS) {
            String triggerGroupSetKey = this.redisSchema.triggerGroupSetKey(new TriggerKey("", matcher.getCompareToValue()));
            long addResult = jedis.sadd(this.redisSchema.pausedTriggerGroupsSet(), new String[]{triggerGroupSetKey});
            if (addResult > 0L) {
                for (String trigger : jedis.smembers(triggerGroupSetKey)) {
                    this.pauseTrigger(this.redisSchema.triggerKey(trigger), jedis);
                }
                pausedTriggerGroups.add(this.redisSchema.triggerGroup(triggerGroupSetKey));
            }
        } else {
            HashMap<String, Response> triggerGroups = new HashMap<String, Response>();
            Pipeline pipe = jedis.pipelined();
            for (String string : jedis.smembers(this.redisSchema.triggerGroupsSet())) {
                if (!matcher.getCompareWithOperator().evaluate(this.redisSchema.triggerGroup(string), matcher.getCompareToValue())) continue;
                triggerGroups.put(string, pipe.smembers(string));
            }
            pipe.sync();
            for (Map.Entry entry : triggerGroups.entrySet()) {
                if (jedis.sadd(this.redisSchema.pausedJobGroupsSet(), new String[]{(String)entry.getKey()}) <= 0L) continue;
                pausedTriggerGroups.add(this.redisSchema.triggerGroup((String)entry.getKey()));
                for (String triggerHashKey : (Set)((Response)entry.getValue()).get()) {
                    this.pauseTrigger(this.redisSchema.triggerKey(triggerHashKey), jedis);
                }
            }
        }
        return pausedTriggerGroups;
    }

    public void pauseJob(JobKey jobKey, Jedis jedis) throws JobPersistenceException {
        for (OperableTrigger trigger : this.getTriggersForJob(jobKey, jedis)) {
            this.pauseTrigger(trigger.getKey(), jedis);
        }
    }

    public Collection<String> pauseJobs(GroupMatcher<JobKey> groupMatcher, Jedis jedis) throws JobPersistenceException {
        HashSet<String> pausedJobGroups;
        block5: {
            block4: {
                pausedJobGroups = new HashSet<String>();
                if (groupMatcher.getCompareWithOperator() != StringMatcher.StringOperatorName.EQUALS) break block4;
                String jobGroupSetKey = this.redisSchema.jobGroupSetKey(new JobKey("", groupMatcher.getCompareToValue()));
                if (jedis.sadd(this.redisSchema.pausedJobGroupsSet(), new String[]{jobGroupSetKey}) <= 0L) break block5;
                pausedJobGroups.add(this.redisSchema.jobGroup(jobGroupSetKey));
                for (String job : jedis.smembers(jobGroupSetKey)) {
                    this.pauseJob(this.redisSchema.jobKey(job), jedis);
                }
                break block5;
            }
            HashMap<String, Response> jobGroups = new HashMap<String, Response>();
            Pipeline pipe = jedis.pipelined();
            for (String string : jedis.smembers(this.redisSchema.jobGroupsSet())) {
                if (!groupMatcher.getCompareWithOperator().evaluate(this.redisSchema.jobGroup(string), groupMatcher.getCompareToValue())) continue;
                jobGroups.put(string, pipe.smembers(string));
            }
            pipe.sync();
            for (Map.Entry entry : jobGroups.entrySet()) {
                if (jedis.sadd(this.redisSchema.pausedJobGroupsSet(), new String[]{(String)entry.getKey()}) <= 0L) continue;
                pausedJobGroups.add(this.redisSchema.jobGroup((String)entry.getKey()));
                for (String jobHashKey : (Set)((Response)entry.getValue()).get()) {
                    this.pauseJob(this.redisSchema.jobKey(jobHashKey), jedis);
                }
            }
        }
        return pausedJobGroups;
    }

    public void resumeTrigger(TriggerKey triggerKey, Jedis jedis) throws JobPersistenceException {
        String triggerHashKey = this.redisSchema.triggerHashKey(triggerKey);
        Pipeline pipe = jedis.pipelined();
        Response exists = pipe.sismember(this.redisSchema.triggersSet(), triggerHashKey);
        Response isPaused = pipe.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.PAUSED), triggerHashKey);
        Response isPausedBlocked = pipe.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.PAUSED_BLOCKED), triggerHashKey);
        pipe.sync();
        if (!((Boolean)exists.get()).booleanValue()) {
            return;
        }
        if (isPaused.get() == null && isPausedBlocked.get() == null) {
            return;
        }
        OperableTrigger trigger = this.retrieveTrigger(triggerKey, jedis);
        String jobHashKey = this.redisSchema.jobHashKey(trigger.getJobKey());
        Date nextFireTime = trigger.getNextFireTime();
        if (nextFireTime != null) {
            if (jedis.sismember(this.redisSchema.blockedJobsSet(), jobHashKey).booleanValue()) {
                this.setTriggerState(RedisTriggerState.BLOCKED, nextFireTime.getTime(), triggerHashKey, jedis);
            } else {
                this.setTriggerState(RedisTriggerState.WAITING, nextFireTime.getTime(), triggerHashKey, jedis);
            }
        }
        this.applyMisfire(trigger, jedis);
    }

    protected boolean applyMisfire(OperableTrigger trigger, Jedis jedis) throws JobPersistenceException {
        Date nextFireTime;
        long misfireTime = System.currentTimeMillis();
        if (this.misfireThreshold > 0) {
            misfireTime -= (long)this.misfireThreshold;
        }
        if ((nextFireTime = trigger.getNextFireTime()) == null || nextFireTime.getTime() > misfireTime || trigger.getMisfireInstruction() == -1) {
            return false;
        }
        Calendar calendar = null;
        if (trigger.getCalendarName() != null) {
            calendar = this.retrieveCalendar(trigger.getCalendarName(), jedis);
        }
        this.signaler.notifyTriggerListenersMisfired((Trigger)((OperableTrigger)trigger.clone()));
        trigger.updateAfterMisfire(calendar);
        this.storeTrigger(trigger, true, jedis);
        if (trigger.getNextFireTime() == null) {
            this.setTriggerState(RedisTriggerState.COMPLETED, System.currentTimeMillis(), this.redisSchema.triggerHashKey(trigger.getKey()), jedis);
            this.signaler.notifySchedulerListenersFinalized((Trigger)trigger);
        } else if (nextFireTime.equals(trigger.getNextFireTime())) {
            return false;
        }
        return true;
    }

    public Collection<String> resumeTriggers(GroupMatcher<TriggerKey> matcher, Jedis jedis) throws JobPersistenceException {
        HashSet<String> resumedTriggerGroups = new HashSet<String>();
        if (matcher.getCompareWithOperator() == StringMatcher.StringOperatorName.EQUALS) {
            String triggerGroupSetKey = this.redisSchema.triggerGroupSetKey(new TriggerKey("", matcher.getCompareToValue()));
            Pipeline pipe = jedis.pipelined();
            pipe.srem(this.redisSchema.pausedJobGroupsSet(), new String[]{triggerGroupSetKey});
            Response triggerHashKeysResponse = pipe.smembers(triggerGroupSetKey);
            pipe.sync();
            for (String triggerHashKey : (Set)triggerHashKeysResponse.get()) {
                OperableTrigger trigger = this.retrieveTrigger(this.redisSchema.triggerKey(triggerHashKey), jedis);
                this.resumeTrigger(trigger.getKey(), jedis);
                resumedTriggerGroups.add(trigger.getKey().getGroup());
            }
        } else {
            for (String triggerGroupSetKey : jedis.smembers(this.redisSchema.triggerGroupsSet())) {
                if (!matcher.getCompareWithOperator().evaluate(this.redisSchema.triggerGroup(triggerGroupSetKey), matcher.getCompareToValue())) continue;
                resumedTriggerGroups.addAll(this.resumeTriggers((GroupMatcher<TriggerKey>)GroupMatcher.triggerGroupEquals((String)this.redisSchema.triggerGroup(triggerGroupSetKey)), jedis));
            }
        }
        return resumedTriggerGroups;
    }

    public Set<String> getPausedTriggerGroups(Jedis jedis) {
        Set triggerGroupSetKeys = jedis.smembers(this.redisSchema.pausedTriggerGroupsSet());
        HashSet<String> names = new HashSet<String>(triggerGroupSetKeys.size());
        for (String triggerGroupSetKey : triggerGroupSetKeys) {
            names.add(this.redisSchema.triggerGroup(triggerGroupSetKey));
        }
        return names;
    }

    public void resumeJob(JobKey jobKey, Jedis jedis) throws JobPersistenceException {
        for (OperableTrigger trigger : this.getTriggersForJob(jobKey, jedis)) {
            this.resumeTrigger(trigger.getKey(), jedis);
        }
    }

    public Collection<String> resumeJobs(GroupMatcher<JobKey> matcher, Jedis jedis) throws JobPersistenceException {
        HashSet<String> resumedJobGroups = new HashSet<String>();
        if (matcher.getCompareWithOperator() == StringMatcher.StringOperatorName.EQUALS) {
            String jobGroupSetKey = this.redisSchema.jobGroupSetKey(new JobKey("", matcher.getCompareToValue()));
            Pipeline pipe = jedis.pipelined();
            Response unpauseResponse = pipe.srem(this.redisSchema.pausedJobGroupsSet(), new String[]{jobGroupSetKey});
            Response jobsResponse = pipe.smembers(jobGroupSetKey);
            pipe.sync();
            if ((Long)unpauseResponse.get() > 0L) {
                resumedJobGroups.add(this.redisSchema.jobGroup(jobGroupSetKey));
            }
            for (String job : (Set)jobsResponse.get()) {
                this.resumeJob(this.redisSchema.jobKey(job), jedis);
            }
        } else {
            for (String jobGroupSetKey : jedis.smembers(this.redisSchema.jobGroupsSet())) {
                if (!matcher.getCompareWithOperator().evaluate(this.redisSchema.jobGroup(jobGroupSetKey), matcher.getCompareToValue())) continue;
                resumedJobGroups.addAll(this.resumeJobs((GroupMatcher<JobKey>)GroupMatcher.jobGroupEquals((String)this.redisSchema.jobGroup(jobGroupSetKey)), jedis));
            }
        }
        return resumedJobGroups;
    }

    public void pauseAll(Jedis jedis) throws JobPersistenceException {
        for (String triggerGroupSet : jedis.smembers(this.redisSchema.triggerGroupsSet())) {
            this.pauseTriggers((GroupMatcher<TriggerKey>)GroupMatcher.triggerGroupEquals((String)this.redisSchema.triggerGroup(triggerGroupSet)), jedis);
        }
    }

    public void resumeAll(Jedis jedis) throws JobPersistenceException {
        for (String triggerGroupSet : jedis.smembers(this.redisSchema.triggerGroupsSet())) {
            this.resumeTriggers((GroupMatcher<TriggerKey>)GroupMatcher.triggerGroupEquals((String)this.redisSchema.triggerGroup(triggerGroupSet)), jedis);
        }
    }

    protected void releaseOrphanedTriggers(RedisTriggerState currentState, RedisTriggerState newState, Jedis jedis) throws JobPersistenceException {
        for (Tuple triggerTuple : jedis.zrangeWithScores(this.redisSchema.triggerStateKey(currentState), 0L, -1L)) {
            String lockId = jedis.get(this.redisSchema.triggerLockKey(this.redisSchema.triggerKey(triggerTuple.getElement())));
            if (!Strings.isNullOrEmpty((String)lockId)) continue;
            logger.debug(String.format("Changing state of orphaned trigger %s from %s to %s.", new Object[]{triggerTuple.getElement(), currentState, newState}));
            this.setTriggerState(newState, triggerTuple.getScore(), triggerTuple.getElement(), jedis);
        }
    }

    protected void releaseTriggersCron(Jedis jedis) throws JobPersistenceException {
        if (System.currentTimeMillis() - this.getLastTriggersReleaseTime(jedis) > 600000L) {
            this.releaseOrphanedTriggers(RedisTriggerState.ACQUIRED, RedisTriggerState.WAITING, jedis);
            this.releaseOrphanedTriggers(RedisTriggerState.BLOCKED, RedisTriggerState.WAITING, jedis);
            this.releaseOrphanedTriggers(RedisTriggerState.PAUSED_BLOCKED, RedisTriggerState.PAUSED, jedis);
            this.settLastTriggerReleaseTime(System.currentTimeMillis(), jedis);
        }
    }

    protected long getLastTriggersReleaseTime(Jedis jedis) {
        String lastReleaseTime = jedis.get(this.redisSchema.lastTriggerReleaseTime());
        if (lastReleaseTime == null) {
            return 0L;
        }
        return Long.parseLong(lastReleaseTime);
    }

    protected void settLastTriggerReleaseTime(long time, Jedis jedis) {
        jedis.set(this.redisSchema.lastTriggerReleaseTime(), Long.toString(time));
    }

    protected boolean isJobConcurrentExecutionDisallowed(Class<? extends Job> jobClass) {
        return ClassUtils.isAnnotationPresent(jobClass, DisallowConcurrentExecution.class);
    }

    protected boolean lockTrigger(TriggerKey triggerKey, Jedis jedis) {
        return jedis.set(this.redisSchema.triggerLockKey(triggerKey), this.schedulerInstanceId, "NX", "PX", 600000).equals("OK");
    }

    public List<OperableTrigger> acquireNextTriggers(long noLaterThan, int maxCount, long timeWindow, Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        boolean retry;
        this.releaseTriggersCron(jedis);
        ArrayList<OperableTrigger> acquiredTriggers = new ArrayList<OperableTrigger>();
        block0: do {
            retry = false;
            HashSet<String> acquiredJobHashKeysForNoConcurrentExec = new HashSet<String>();
            for (Tuple triggerTuple : jedis.zrangeByScoreWithScores(this.redisSchema.triggerStateKey(RedisTriggerState.WAITING), 0.0, (double)(noLaterThan + timeWindow), 0, maxCount)) {
                OperableTrigger trigger = this.retrieveTrigger(this.redisSchema.triggerKey(triggerTuple.getElement()), jedis);
                if (this.applyMisfire(trigger, jedis)) {
                    logger.debug("misfired trigger: " + triggerTuple.getElement());
                    retry = true;
                    continue block0;
                }
                if (trigger.getNextFireTime() == null) {
                    this.unsetTriggerState(triggerTuple.getElement(), jedis);
                    continue;
                }
                String jobHashKey = this.redisSchema.jobHashKey(trigger.getJobKey());
                JobDetail job = this.retrieveJob(trigger.getJobKey(), jedis);
                if (this.isJobConcurrentExecutionDisallowed(job.getJobClass())) {
                    if (acquiredJobHashKeysForNoConcurrentExec.contains(jobHashKey)) continue;
                    acquiredJobHashKeysForNoConcurrentExec.add(jobHashKey);
                }
                this.lockTrigger(trigger.getKey(), jedis);
                this.setTriggerState(RedisTriggerState.ACQUIRED, triggerTuple.getScore(), triggerTuple.getElement(), jedis);
                acquiredTriggers.add(trigger);
                logger.debug(String.format("Trigger %s acquired", triggerTuple.getElement()));
            }
        } while (retry);
        return acquiredTriggers;
    }

    public void releaseAcquiredTrigger(OperableTrigger trigger, Jedis jedis) throws JobPersistenceException {
        String triggerHashKey = this.redisSchema.triggerHashKey(trigger.getKey());
        if (jedis.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.ACQUIRED), triggerHashKey) != null) {
            if (trigger.getNextFireTime() != null) {
                this.setTriggerState(RedisTriggerState.WAITING, trigger.getNextFireTime().getTime(), triggerHashKey, jedis);
            } else {
                this.unsetTriggerState(triggerHashKey, jedis);
            }
        }
    }

    public List<TriggerFiredResult> triggersFired(List<OperableTrigger> triggers, Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        ArrayList<TriggerFiredResult> results = new ArrayList<TriggerFiredResult>();
        for (OperableTrigger trigger : triggers) {
            String triggerHashKey = this.redisSchema.triggerHashKey(trigger.getKey());
            logger.debug(String.format("Trigger %s fired.", triggerHashKey));
            Pipeline pipe = jedis.pipelined();
            Response triggerExistsResponse = pipe.exists(triggerHashKey);
            Response triggerAcquiredResponse = pipe.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.ACQUIRED), triggerHashKey);
            pipe.sync();
            if (!((Boolean)triggerExistsResponse.get()).booleanValue() || triggerAcquiredResponse.get() == null) {
                if (!((Boolean)triggerExistsResponse.get()).booleanValue()) {
                    logger.debug(String.format("Trigger %s does not exist.", triggerHashKey));
                    continue;
                }
                logger.debug(String.format("Trigger %s was not acquired.", triggerHashKey));
                continue;
            }
            Calendar calendar = null;
            String calendarName = trigger.getCalendarName();
            if (calendarName != null && (calendar = this.retrieveCalendar(calendarName, jedis)) == null) continue;
            Date previousFireTime = trigger.getPreviousFireTime();
            trigger.triggered(calendar);
            JobDetail job = this.retrieveJob(trigger.getJobKey(), jedis);
            TriggerFiredBundle triggerFiredBundle = new TriggerFiredBundle(job, trigger, calendar, false, new Date(), previousFireTime, previousFireTime, trigger.getNextFireTime());
            if (this.isJobConcurrentExecutionDisallowed(job.getJobClass())) {
                String jobHashKey = this.redisSchema.jobHashKey(trigger.getJobKey());
                String jobTriggerSetKey = this.redisSchema.jobTriggersSetKey(job.getKey());
                for (String nonConcurrentTriggerHashKey : jedis.smembers(jobTriggerSetKey)) {
                    Double score = jedis.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.WAITING), nonConcurrentTriggerHashKey);
                    if (score != null) {
                        this.setTriggerState(RedisTriggerState.BLOCKED, score, nonConcurrentTriggerHashKey, jedis);
                        continue;
                    }
                    score = jedis.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.PAUSED), nonConcurrentTriggerHashKey);
                    if (score == null) continue;
                    this.setTriggerState(RedisTriggerState.PAUSED_BLOCKED, score, nonConcurrentTriggerHashKey, jedis);
                }
                pipe = jedis.pipelined();
                pipe.set(this.redisSchema.jobBlockedKey(job.getKey()), this.schedulerInstanceId);
                pipe.sadd(this.redisSchema.blockedJobsSet(), new String[]{jobHashKey});
                pipe.sync();
            }
            if (trigger.getNextFireTime() != null) {
                long nextFireTime = trigger.getNextFireTime().getTime();
                jedis.hset(triggerHashKey, TRIGGER_NEXT_FIRE_TIME, Long.toString(nextFireTime));
                logger.debug(String.format("Releasing trigger %s with next fire time %s. Setting state to WAITING.", triggerHashKey, nextFireTime));
                this.setTriggerState(RedisTriggerState.WAITING, nextFireTime, triggerHashKey, jedis);
            } else {
                jedis.hset(triggerHashKey, TRIGGER_NEXT_FIRE_TIME, "");
                this.unsetTriggerState(triggerHashKey, jedis);
            }
            results.add(new TriggerFiredResult(triggerFiredBundle));
        }
        return results;
    }

    protected boolean isPersistJobDataAfterExecution(Class<? extends Job> jobClass) {
        return ClassUtils.isAnnotationPresent(jobClass, PersistJobDataAfterExecution.class);
    }

    public void triggeredJobComplete(OperableTrigger trigger, JobDetail jobDetail, Trigger.CompletedExecutionInstruction triggerInstCode, Jedis jedis) throws JobPersistenceException, ClassNotFoundException {
        String jobHashKey = this.redisSchema.jobHashKey(jobDetail.getKey());
        String jobDataMapHashKey = this.redisSchema.jobDataMapHashKey(jobDetail.getKey());
        String triggerHashKey = this.redisSchema.triggerHashKey(trigger.getKey());
        logger.debug(String.format("Job %s completed.", jobHashKey));
        if (jedis.exists(jobHashKey).booleanValue()) {
            Pipeline pipe;
            if (this.isPersistJobDataAfterExecution(jobDetail.getJobClass())) {
                JobDataMap jobDataMap = jobDetail.getJobDataMap();
                pipe = jedis.pipelined();
                pipe.del(jobDataMapHashKey);
                if (jobDataMap != null && !jobDataMap.isEmpty()) {
                    pipe.hmset(jobDataMapHashKey, this.getStringDataMap(jobDataMap));
                }
                pipe.sync();
            }
            if (this.isJobConcurrentExecutionDisallowed(jobDetail.getJobClass())) {
                pipe = jedis.pipelined();
                pipe.srem(this.redisSchema.blockedJobsSet(), new String[]{jobHashKey});
                pipe.del(this.redisSchema.jobBlockedKey(jobDetail.getKey()));
                pipe.sync();
                String jobTriggersSetKey = this.redisSchema.jobTriggersSetKey(jobDetail.getKey());
                for (String nonConcurrentTriggerHashKey : jedis.smembers(jobTriggersSetKey)) {
                    Double score = jedis.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.BLOCKED), nonConcurrentTriggerHashKey);
                    if (score != null) {
                        this.setTriggerState(RedisTriggerState.WAITING, score, nonConcurrentTriggerHashKey, jedis);
                        continue;
                    }
                    score = jedis.zscore(this.redisSchema.triggerStateKey(RedisTriggerState.PAUSED_BLOCKED), nonConcurrentTriggerHashKey);
                    if (score == null) continue;
                    this.setTriggerState(RedisTriggerState.PAUSED, score, nonConcurrentTriggerHashKey, jedis);
                }
                this.signaler.signalSchedulingChange(0L);
            }
        } else {
            jedis.srem(this.redisSchema.blockedJobsSet(), new String[]{jobHashKey});
        }
        if (jedis.exists(triggerHashKey).booleanValue()) {
            if (triggerInstCode == Trigger.CompletedExecutionInstruction.DELETE_TRIGGER) {
                if (trigger.getNextFireTime() == null) {
                    if (Strings.isNullOrEmpty((String)jedis.hget(triggerHashKey, TRIGGER_NEXT_FIRE_TIME))) {
                        this.removeTrigger(trigger.getKey(), jedis);
                    }
                } else {
                    this.removeTrigger(trigger.getKey(), jedis);
                    this.signaler.signalSchedulingChange(0L);
                }
            } else if (triggerInstCode == Trigger.CompletedExecutionInstruction.SET_TRIGGER_COMPLETE) {
                this.setTriggerState(RedisTriggerState.COMPLETED, System.currentTimeMillis(), triggerHashKey, jedis);
                this.signaler.signalSchedulingChange(0L);
            } else if (triggerInstCode == Trigger.CompletedExecutionInstruction.SET_TRIGGER_ERROR) {
                logger.debug(String.format("Trigger %s set to ERROR state.", triggerHashKey));
                double score = trigger.getNextFireTime() != null ? (double)trigger.getNextFireTime().getTime() : 0.0;
                this.setTriggerState(RedisTriggerState.ERROR, score, triggerHashKey, jedis);
                this.signaler.signalSchedulingChange(0L);
            } else if (triggerInstCode == Trigger.CompletedExecutionInstruction.SET_ALL_JOB_TRIGGERS_ERROR) {
                String jobTriggersSetKey = this.redisSchema.jobTriggersSetKey(jobDetail.getKey());
                for (String errorTriggerHashKey : jedis.smembers(jobTriggersSetKey)) {
                    String nextFireTime = jedis.hget(errorTriggerHashKey, TRIGGER_NEXT_FIRE_TIME);
                    double score = Strings.isNullOrEmpty((String)nextFireTime) ? 0.0 : Double.parseDouble(nextFireTime);
                    this.setTriggerState(RedisTriggerState.ERROR, score, errorTriggerHashKey, jedis);
                }
                this.signaler.signalSchedulingChange(0L);
            } else if (triggerInstCode == Trigger.CompletedExecutionInstruction.SET_ALL_JOB_TRIGGERS_COMPLETE) {
                String jobTriggerSetKey = this.redisSchema.jobTriggersSetKey(jobDetail.getKey());
                for (String completedTriggerHashKey : jedis.smembers(jobTriggerSetKey)) {
                    this.setTriggerState(RedisTriggerState.COMPLETED, System.currentTimeMillis(), completedTriggerHashKey, jedis);
                }
                this.signaler.signalSchedulingChange(0L);
            }
        }
    }
}

