/*
 * *************************************************************************
 *   Copyright (c) 2018-2025, dreamlu.net All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the dreamlu.net developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: chunmeng.lu (qq596392912@gmail.com)
 * *************************************************************************
 */

package net.dreamlu.boot.support;

import net.dreamlu.tool.util.Charsets;
import net.dreamlu.tool.util.URLUtils;
import net.dreamlu.tool.util.WebUtils;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.ResourceRegion;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.io.IOException;

/**
 * 基础 controller
 *
 * @author L.cm
 */
public abstract class BaseController {

	/**
	 * redirect跳转
	 *
	 * @param url 目标url
	 * @return 跳转地址
	 */
	protected String redirect(String url) {
		return "redirect:".concat(url);
	}

	/**
	 * 下载文件
	 *
	 * @param file 文件
	 * @return {ResponseEntity}
	 * @throws IOException io异常
	 */
	protected ResponseEntity<ResourceRegion> download(File file) throws IOException {
		String fileName = file.getName();
		return download(file, fileName);
	}

	/**
	 * 下载
	 *
	 * @param file     文件
	 * @param fileName 生成的文件名
	 * @return {ResponseEntity}
	 * @throws IOException io异常
	 */
	protected ResponseEntity<ResourceRegion> download(File file, String fileName) throws IOException {
		Resource resource = new FileSystemResource(file);
		return download(resource, fileName);
	}

	/**
	 * 下载
	 *
	 * @param resource 资源
	 * @param fileName 生成的文件名
	 * @return {ResponseEntity}
	 * @throws IOException io异常
	 */
	protected ResponseEntity<ResourceRegion> download(Resource resource, String fileName) throws IOException {
		HttpServletRequest request = WebUtils.getRequest();
		String header = request.getHeader("User-Agent");
		// 避免空指针
		header = header == null ? "" : header.toUpperCase();
		HttpStatus status;
		if (header.contains("MSIE") || header.contains("TRIDENT") || header.contains("EDGE")) {
			fileName = URLUtils.encodeURL(fileName, Charsets.UTF_8);
			status = HttpStatus.OK;
		} else {
			fileName = new String(fileName.getBytes(Charsets.UTF_8), Charsets.ISO_8859_1);
			status = HttpStatus.CREATED;
		}
		// 断点续传
		long position = 0;
		long count = resource.contentLength();
		String range = request.getHeader("Range");
		if (null != range) {
			status = HttpStatus.PARTIAL_CONTENT;
			String[] rangeRange = range.replace("bytes=", "").split("-");
			position = Long.parseLong(rangeRange[0]);
			if (rangeRange.length > 1) {
				long end = Long.parseLong(rangeRange[1]);
				count = end - position + 1;
			}
		}
		HttpHeaders headers = new HttpHeaders();
		headers.setContentType(MediaType.APPLICATION_OCTET_STREAM);
		headers.setContentDispositionFormData("attachment", fileName);
		return new ResponseEntity<>(new ResourceRegion(resource, position, count), headers, status);
	}
}
