/*
 * *************************************************************************
 *   Copyright (c) 2018-2025, dreamlu.net All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the dreamlu.net developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: chunmeng.lu (qq596392912@gmail.com)
 * *************************************************************************
 */

package net.dreamlu.boot.config;

import lombok.AllArgsConstructor;
import net.dreamlu.boot.properties.TransactionalProperties;
import org.springframework.aop.aspectj.annotation.AnnotationAwareAspectJAutoProxyCreator;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.interceptor.TransactionInterceptor;

import java.util.*;

@Configuration
@EnableConfigurationProperties(TransactionalProperties.class)
@ConditionalOnClass(name = "org.springframework.transaction.PlatformTransactionManager")
@ConditionalOnProperty(value = "dream.transactional.enabled", havingValue = "true", matchIfMissing = true)
@AllArgsConstructor
public class TransactionalConfiguration {
	private static final String TRANSACTION_INTERCEPTOR_NAME = "transactionAdvice";

	private final TransactionalProperties transactionalProperties;
	private final PlatformTransactionManager transactionManager;

	@Bean(TRANSACTION_INTERCEPTOR_NAME)
	public TransactionInterceptor transactionInterceptor() {
		Properties attributes = new Properties();
		attributes.putAll(transactionalProperties.getTxAttributes());
		attributes.setProperty("find*", "PROPAGATION_REQUIRED,readOnly");
		attributes.setProperty("select*", "PROPAGATION_REQUIRED,readOnly");
		attributes.setProperty("insert*", "PROPAGATION_REQUIRED,-Exception");
		attributes.setProperty("update*", "PROPAGATION_REQUIRED,-Exception");
		attributes.setProperty("delete*", "PROPAGATION_REQUIRED,-Exception");
		attributes.setProperty("*", "PROPAGATION_REQUIRED,-Exception");
		return new TransactionInterceptor(transactionManager, attributes);
	}

	@Bean
	public AnnotationAwareAspectJAutoProxyCreator transactionProxyCreator(){
		AnnotationAwareAspectJAutoProxyCreator c = new AnnotationAwareAspectJAutoProxyCreator();
		c.setInterceptorNames(TRANSACTION_INTERCEPTOR_NAME);
		c.setIncludePatterns(Arrays.asList(transactionalProperties.getExecutionPatterns()));
		c.setProxyTargetClass(true);
		c.setOrder(2);
		return c;
	}

}
