/*
 * *************************************************************************
 *   Copyright (c) 2018-2025, dreamlu.net All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the dreamlu.net developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: chunmeng.lu (qq596392912@gmail.com)
 * *************************************************************************
 */

package net.dreamlu.boot.config;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.dreamlu.boot.func.FmtFunc;
import net.dreamlu.boot.properties.*;
import net.dreamlu.boot.runer.StartEventListener;
import net.dreamlu.boot.template.DreamTemplate;
import net.dreamlu.tool.util.SpringUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.http.MediaType;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.config.annotation.CorsRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import org.springframework.web.servlet.mvc.method.annotation.AbstractJsonpResponseBodyAdvice;

/**
 * 自动配置
 *
 * @author l.cm
 */
@Configuration
@EnableConfigurationProperties({
	DreamProperties.class,
	DreamFmtProperties.class,
	DreamCrossProperties.class,
	DreamSwaggerProperties.class,
	DreamMessagesProperties.class,
	DreamHttpCacheProperties.class
})
@EnableAspectJAutoProxy(proxyTargetClass = true, exposeProxy = true)
@Slf4j
public class DreamBootAutoConfiguration {

	@Bean
	public StartEventListener startEventListener() {
		return new StartEventListener();
	}

	@Bean
	public SpringUtils springUtils() {
		return new SpringUtils();
	}

	@Bean("tpl")
	public DreamTemplate dreamTemplate(DreamProperties properties) {
		return new DreamTemplate(properties);
	}

	@Bean("fmt")
	public FmtFunc fmtFunc(DreamFmtProperties properties) {
		return new FmtFunc(properties);
	}

	/**
	 * jsonp配置
	 *
	 * @author L.cm
	 */
	@ControllerAdvice
	@ConditionalOnProperty(value = "dream.cross.jsonp.enabled", havingValue = "true")
	public static class JsonpResponseAdvice extends AbstractJsonpResponseBodyAdvice implements InitializingBean {

		public JsonpResponseAdvice() {
			super("callback");
		}

		@Override
		protected MediaType getContentType(MediaType contentType, ServerHttpRequest request, ServerHttpResponse response) {
			return new MediaType("application", "javascript", contentType.getCharset());
		}

		@Override
		public void afterPropertiesSet() throws Exception {
			log.info("jsonp cross is enabled.");
		}
	}

	@Configuration
	@ConditionalOnProperty(value = "dream.cross.cors.enabled", havingValue = "true")
	@AllArgsConstructor
	public static class CorsConfiguration extends WebMvcConfigurerAdapter implements InitializingBean {
		private final DreamCrossProperties properties;

		/**
		 * 配置 Cors 跨域
		 *
		 * @param registry CorsRegistry
		 */
		@Override
		public void addCorsMappings(CorsRegistry registry) {
			DreamCrossProperties.Cors cors = properties.getCors();
			registry.addMapping(cors.getMapping())
				.allowedOrigins(cors.getAllowedOrigins().toArray(new String[0]))
				.allowedMethods(cors.getAllowedMethods().toArray(new String[0]))
				.allowedHeaders(cors.getAllowedHeaders().toArray(new String[0]))
				.exposedHeaders(cors.getExposedHeaders().toArray(new String[0]))
				.allowCredentials(cors.getAllowCredentials())
				.maxAge(cors.getMaxAge());
		}

		@Override
		public void afterPropertiesSet() throws Exception {
			log.info("cors cross is enabled.");
		}
	}

}
