/*
 * *************************************************************************
 *   Copyright (c) 2018-2025, dreamlu.net All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the dreamlu.net developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: chunmeng.lu (qq596392912@gmail.com)
 * *************************************************************************
 */

package net.dreamlu.boot.runer;

import net.dreamlu.tool.util.StringUtils;
import org.springframework.boot.builder.SpringApplicationBuilder;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.core.env.*;
import org.springframework.util.Assert;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 项目启动器，搞定环境变量问题
 *
 * @author L.cm
 */
public class DreamApplication {

	/**
	 * Create an application context
	 *
	 * @param appName application name
	 * @param source  The sources
	 * @param args    args the command line arguments
	 * @return an application context created from the current state
	 */
	public static ConfigurableApplicationContext run(String appName, Object source, String... args) {
		Assert.hasText(appName, "[appName]服务名不能为空");
		// 读取环境变量，使用spring boot的规则
		ConfigurableEnvironment environment = new StandardEnvironment();
		MutablePropertySources propertySources = environment.getPropertySources();
		propertySources.addFirst(new SimpleCommandLinePropertySource(args));
		propertySources.addLast(new MapPropertySource(StandardEnvironment.SYSTEM_PROPERTIES_PROPERTY_SOURCE_NAME, environment.getSystemProperties()));
		propertySources.addLast(new SystemEnvironmentPropertySource(StandardEnvironment.SYSTEM_ENVIRONMENT_PROPERTY_SOURCE_NAME, environment.getSystemEnvironment()));
		// 获取配置的环境变量
		String[] activeProfiles = environment.getActiveProfiles();
		// 判断环境:dev、test、prod
		List<String> profiles = Arrays.asList(activeProfiles);
		// 预设的环境
		List<String> presetProfiles = new ArrayList<>(Arrays.asList("dev", "test", "prod"));
		// 交集
		presetProfiles.retainAll(profiles);
		// 当前使用
		List<String> activeProfileList = new ArrayList<>(profiles);
		SpringApplicationBuilder builder = new SpringApplicationBuilder(source);
		String profile;
		if (presetProfiles.isEmpty()) {
			// 默认dev开发
			profile = "dev";
			activeProfileList.add(profile);
			builder.profiles(profile);
		} else if (presetProfiles.size() == 1) {
			profile = presetProfiles.get(0);
		} else {
			// 同时存在dev、test、prod环境时
			throw new RuntimeException("同时存在环境变量:[" + StringUtils.join(activeProfiles) + "]");
		}
		// 添加启动目录打印
		String startJarPath = DreamApplication.class.getResource("/").getPath().split("!")[0];
		String activePros = StringUtils.join(activeProfileList);
		System.err.println(String.format("----启动中，读取到的环境变量:[%s]，jar地址:[%s]----", activePros, startJarPath));
		System.setProperty("dream.env", profile);
		System.setProperty("dream.name", appName);
		System.setProperty("spring.application.name", appName);
		System.setProperty("banner.location", "classpath:dream_banner.txt");
		// 默认日志格式不好查bug
		System.setProperty("logging.pattern.console", "%clr(%d{yyyy-MM-dd HH:mm:ss.SSS}){faint} %clr(%5p) %clr(${PID:- }){magenta} %clr(---){faint} %clr([%15.15t]){faint} %clr(%C{36}.%M:%L){cyan} %clr(:){faint} %m%n%wEx");
		return builder.run(args);
	}

}
