/*
 * *************************************************************************
 *   Copyright (c) 2018-2025, dreamlu.net All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the dreamlu.net developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: chunmeng.lu (qq596392912@gmail.com)
 * *************************************************************************
 */

package net.dreamlu.boot.error;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import net.dreamlu.tool.exception.LocalizedException;
import net.dreamlu.tool.exception.ServiceException;
import net.dreamlu.tool.result.Result;
import net.dreamlu.tool.result.Results;
import net.dreamlu.tool.util.Exceptions;
import net.dreamlu.tool.util.StringUtils;
import org.springframework.context.MessageSource;
import org.springframework.context.NoSuchMessageException;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.MediaType;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.Locale;
import java.util.Set;

/**
 * 处理异常国际化
 *
 * @author L.cm
 */
@Slf4j
@AllArgsConstructor
public class ExceptionTranslator {
	private final MessageSource messageSource;

	/**
	 * 异常处理
	 *
	 * 1. 优先读取国际化
	 * 2. 国际化没有时给预设的异常信息
	 * 3. 原本的异常信息
	 *
	 * @param error Throwable
	 * @return 友好的异常消息
	 */
	public Result<String> handle(Throwable error) {
		Locale locale = LocaleContextHolder.getLocale();
		Throwable _error = Exceptions.unwrap(error);
		return handleException(_error, locale);
	}

	protected Result<String> handleException(Throwable error, Locale locale) {
		if (error instanceof MissingServletRequestParameterException) {
			return handleError((MissingServletRequestParameterException) error, locale);
		} else if (error instanceof MethodArgumentNotValidException) {
			return handleError((MethodArgumentNotValidException) error);
		} else if (error instanceof BindException) {
			return handleError((BindException) error);
		} else if (error instanceof ConstraintViolationException) {
			return handleError((ConstraintViolationException) error);
		} else if (error instanceof NoHandlerFoundException) {
			return handleError((NoHandlerFoundException) error, locale);
		} else if (error instanceof HttpRequestMethodNotSupportedException) {
			return handleError((HttpRequestMethodNotSupportedException) error, locale);
		} else if (error instanceof HttpMediaTypeNotSupportedException) {
			return handleError((HttpMediaTypeNotSupportedException) error, locale);
		} else if (error instanceof LocalizedException) {
			return handleError((LocalizedException) error, locale);
		} else if (error instanceof ServiceException) {
			return handleError((ServiceException) error, locale);
		} else {
			return handleError(error, locale);
		}
	}

	protected Result<String> handleError(MissingServletRequestParameterException e, Locale locale) {
		log.error("缺少请求参数", e);
		String parameterName = e.getParameterName();
		String parameterType = e.getParameterType();
		return getMessage(e, locale, new Object[]{parameterName, parameterType});
	}

	protected Result<String> handleError(MethodArgumentNotValidException e) {
		log.error("参数验证失败", e);
		BindingResult result = e.getBindingResult();
		FieldError error = result.getFieldError();
		return Results.failure(error.getDefaultMessage());
	}

	protected Result<String> handleError(BindException e) {
		log.error("参数绑定失败", e);
		BindingResult result = e.getBindingResult();
		FieldError error = result.getFieldError();
		return Results.failure(error.getDefaultMessage());
	}

	protected Result<String> handleError(ConstraintViolationException e) {
		log.error("参数验证失败", e);
		Set<ConstraintViolation<?>> violations = e.getConstraintViolations();
		ConstraintViolation<?> violation = violations.iterator().next();
		return Results.failure(violation.getMessage());
	}

	protected Result<String> handleError(NoHandlerFoundException e, Locale locale) {
		log.error("404没找到请求", e);
		String httpMethod = e.getHttpMethod();
		String requestURL = e.getRequestURL();
		return getMessage(e, locale, new Object[]{httpMethod, requestURL});
	}

	protected Result<String> handleError(HttpRequestMethodNotSupportedException e, Locale locale) {
		log.error("不支持当前请求方法", e);
		String method = e.getMethod();
		return getMessage(e, locale, new Object[]{method});
	}

	protected Result<String> handleError(HttpMediaTypeNotSupportedException e, Locale locale) {
		log.error("不支持当前媒体类型", e);
		MediaType contentType = e.getContentType();
		return getMessage(e, locale, new Object[]{contentType.getType()});
	}

	protected Result<String> handleError(LocalizedException e, Locale locale) {
		log.error("本地化异常", e);
		return getMessage(e, locale, e.getLocaleArgs());
	}

	protected Result<String> handleError(ServiceException e, Locale locale) {
		log.error("本地化异常", e);
		return e.getResult();
	}

	protected Result<String> handleError(Throwable e, Locale locale) {
		log.error("通用异常", e);
		return getMessage(e, locale, new Object[0]);
	}

	protected Result<String> getMessage(Throwable e, Locale locale, Object[] values) {
		String msgKey = e instanceof LocalizedException ?
			((LocalizedException) e).getLocaleMessage() : e.getClass().getName();
		String message;
		try {
			message = messageSource.getMessage(msgKey, values, locale);
		} catch (NoSuchMessageException ex) {
			log.warn(ex.getMessage());
			message = e.getMessage();
		}
		if (StringUtils.isBlank(message)) {
			message = "系统异常";
		}
		return Results.failure(message);
	}

}
