/*
 * *************************************************************************
 *   Copyright (c) 2018-2025, dreamlu.net All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * Neither the name of the dreamlu.net developer nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * Author: chunmeng.lu (qq596392912@gmail.com)
 * *************************************************************************
 */

package net.dreamlu.boot.error;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.AllArgsConstructor;
import net.dreamlu.boot.error.DreamErrorAttributes;
import net.dreamlu.boot.error.DreamErrorController;
import net.dreamlu.boot.error.ExceptionTranslator;
import net.dreamlu.boot.properties.DreamMessagesProperties;
import net.dreamlu.boot.properties.DreamProperties;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.boot.autoconfigure.web.*;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.web.servlet.DispatcherServlet;

import javax.servlet.Servlet;

/**
 * 统一异常处理
 *
 * @author L.cm
 */
@Configuration
@ConditionalOnWebApplication
@ConditionalOnClass({ Servlet.class, DispatcherServlet.class })
@AutoConfigureBefore(ErrorMvcAutoConfiguration.class)
@AllArgsConstructor
public class DreamErrorMvcAutoConfiguration {
	private final ServerProperties serverProperties;
	private final ObjectMapper objectMapper;
	private final DreamProperties properties;
	private final DreamMessagesProperties messagesProperties;

	@Bean
	@ConditionalOnMissingBean(value = ExceptionTranslator.class, search = SearchStrategy.CURRENT)
	public ExceptionTranslator exceptionTranslator() {
		MessageSource messageSource = getMessageSource(messagesProperties);
		return new ExceptionTranslator(messageSource);
	}

	@Bean
	@ConditionalOnMissingBean(value = ErrorAttributes.class, search = SearchStrategy.CURRENT)
	public DefaultErrorAttributes errorAttributes(ExceptionTranslator translator) {
		return new DreamErrorAttributes(translator, properties);
	}

	@Bean
	@ConditionalOnMissingBean(value = ErrorController.class, search = SearchStrategy.CURRENT)
	public BasicErrorController basicErrorController(ErrorAttributes errorAttributes) {
		return new DreamErrorController(errorAttributes, this.serverProperties.getError(), this.properties, this.objectMapper);
	}

	public MessageSource getMessageSource(DreamMessagesProperties properties) {
		ReloadableResourceBundleMessageSource messageSource = new ReloadableResourceBundleMessageSource();
		messageSource.setBasename(properties.getException().getBasename());
		messageSource.setCacheSeconds(properties.getCacheSeconds());
		messageSource.setDefaultEncoding(properties.getEncoding().name());
		messageSource.setAlwaysUseMessageFormat(properties.isAlwaysUseMessageFormat());
		messageSource.setFallbackToSystemLocale(properties.isFallbackToSystemLocale());
		return messageSource;
	}

}
