package net.dreamlu.boot.properties;

import lombok.Getter;
import lombok.Setter;
import net.dreamlu.tool.util.StringUtils;
import org.springframework.boot.context.properties.ConfigurationProperties;

import java.util.HashMap;
import java.util.Map;

/**
 * 配置文件
 *
 * @author L.cm
 */
@ConfigurationProperties("dream")
public class DreamProperties {
	/**
	 * 环境，方便在代码中获取
	 */
	@Getter
	@Setter
	private String env;
	/**
	 * 装载自定义配置dream.prop.xxx
	 */
	@Getter
	private final Map<String, String> prop = new HashMap<>();

	@Getter
	private final Mvc mvc = new Mvc();

	@Getter
	private final Login login = new Login();

	@Getter
	private final Async async = new Async();

	@Getter
	private final Captcha captcha = new Captcha();

	@Getter
	private final Locale locale = new Locale();

	@Getter
	private final Upload upload = new Upload();

	@Getter
	private final Tpl tpl = new Tpl();

	/**
	 * Mvc配置
	 */
	@Getter
	@Setter
	public static class Mvc {
		/**
		 * contentType，默认：application/json;charset=UTF-8
		 */
		private String contentType = DreamDefaults.Mvc.contentType;
	}

	/**
	 * 登录配置
	 */
	@Getter
	@Setter
	public static class Login {
		/**
		 * 登录重试锁定次数，默认：5
		 */
		private int retryLimit = DreamDefaults.Login.retryLimit;
		/**
		 * 登录重试锁定cache名，默认：retryLimitCache
		 */
		private String retryLimitCacheName = DreamDefaults.Login.retryLimitCacheName;
	}

	/**
	 * 异步线程设置
	 */
	@Getter
	@Setter
	public static class Async {
		/**
		 * 异步核心线程数，默认：2
		 */
		private int corePoolSize = DreamDefaults.Async.corePoolSize;
		/**
		 * 异步最大线程数，默认：50
		 */
		private int maxPoolSize = DreamDefaults.Async.maxPoolSize;
		/**
		 * 队列容量，默认：10000
		 */
		private int queueCapacity = DreamDefaults.Async.queueCapacity;
		/**
		 * 线程存活时间，默认：300
		 */
		private int keepAliveSeconds = DreamDefaults.Async.keepAliveSeconds;
	}

	/**
	 * 验证码设置
	 */
	@Getter
	@Setter
	public static class Captcha {
		/**
		 * 验证码cookie名，默认：dream-captcha
		 */
		private String cookieName = DreamDefaults.Captcha.cookieName;
		/**
		 * 验证码cache名，默认：dreamCaptchaCache
		 */
		private String cacheName = DreamDefaults.Captcha.chacheName;
	}

	/**
	 * 国际化配置
	 */
	@Getter
	@Setter
	public static class Locale {
		/**
		 * 国际化cookie name，默认：DREAM-LANG
		 */
		String cookieName = DreamDefaults.Locale.cookieName;
		/**
		 * 国际化参数名，默认：language
		 */
		String paramName = DreamDefaults.Locale.paramName;
		/**
		 * 默认的语言，默认：zh
		 */
		java.util.Locale defaultLocale = DreamDefaults.Locale.defaultLocale;
	}

	/**
	 * 上传配置，考虑为jar包运行
	 * <p>
	 * 所以需要指定上传目录
	 */
	@Getter
	@Setter
	public static class Upload {
		/**
		 * 所以需要指定上传目录，默认：jar包同级目录
		 */
		String savePath = DreamDefaults.Upload.savePath;
	}

	/**
	 * 内部模板引擎
	 */
	@Getter
	@Setter
	public static class Tpl {
		/**
		 * 模板分隔符开始，默认：{{
		 */
		String open = DreamDefaults.Tpl.open;
		/**
		 * 模板分隔符结束，默认：}}
		 */
		String close = DreamDefaults.Tpl.close;
		/**
		 * 模板前缀，默认：classpath:templates/tpl/
		 */
		String prefix = DreamDefaults.Tpl.prefix;
		/**
		 * 缓存模板，默认：true
		 */
		boolean cache = DreamDefaults.Tpl.cache;
	}

	/**
	 * 获取配置
	 *
	 * @param key key
	 * @return value
	 */
	public String get(String key) {
		return get(key, null);
	}

	/**
	 * 获取配置
	 *
	 * @param key          key
	 * @param defaultValue 默认值
	 * @return value
	 */
	public String get(String key, String defaultValue) {
		String value = prop.get(key);
		if (StringUtils.isBlank(value)) {
			return defaultValue;
		}
		return value;
	}

	/**
	 * 获取配置
	 *
	 * @param key key
	 * @return int value
	 */
	public Integer getInt(String key) {
		return getInt(key, null);
	}

	/**
	 * 获取配置
	 *
	 * @param key          key
	 * @param defaultValue 默认值
	 * @return int value
	 */
	public Integer getInt(String key, Integer defaultValue) {
		String value = prop.get(key);
		if (value != null) {
			return Integer.parseInt(value.trim());
		}
		return defaultValue;
	}

	/**
	 * 获取配置
	 *
	 * @param key key
	 * @return long value
	 */
	public Long getLong(String key) {
		return getLong(key, null);
	}

	/**
	 * 获取配置
	 *
	 * @param key          key
	 * @param defaultValue 默认值
	 * @return long value
	 */
	public Long getLong(String key, Long defaultValue) {
		String value = prop.get(key);
		if (value != null) {
			return Long.parseLong(value.trim());
		}
		return defaultValue;
	}

	/**
	 * 获取配置
	 *
	 * @param key key
	 * @return Boolean value
	 */
	public Boolean getBoolean(String key) {
		return getBoolean(key, null);
	}

	/**
	 * 获取配置
	 *
	 * @param key          key
	 * @param defaultValue 默认值
	 * @return Boolean value
	 */
	public Boolean getBoolean(String key, Boolean defaultValue) {
		String value = prop.get(key);
		if (value != null) {
			value = value.toLowerCase().trim();
			if ("true".equals(value)) {
				return true;
			} else if ("false".equals(value)) {
				return false;
			}
			throw new RuntimeException("The value can not parse to Boolean : " + value);
		}
		return defaultValue;
	}

	/**
	 * 判断是否存在key
	 *
	 * @param key prop key
	 * @return boolean
	 */
	public boolean containsKey(String key) {
		return prop.containsKey(key);
	}

}
