package net.dreamlu.boot.error;


import com.fasterxml.jackson.databind.ObjectMapper;
import net.dreamlu.boot.properties.DreamMessagesProperties;
import net.dreamlu.boot.properties.DreamProperties;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.condition.SearchStrategy;
import org.springframework.boot.autoconfigure.web.*;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.validation.beanvalidation.MethodValidationPostProcessor;
import org.springframework.web.servlet.DispatcherServlet;

import javax.servlet.Servlet;

/**
 * 统一异常处理
 *
 * @author L.cm
 */
@Configuration
@ConditionalOnWebApplication
@ConditionalOnClass({ Servlet.class, DispatcherServlet.class })
@AutoConfigureBefore(ErrorMvcAutoConfiguration.class)
public class DreamErrorMvcAutoConfiguration {
	private final ServerProperties serverProperties;
	private final ObjectMapper objectMapper;
	private final DreamProperties properties;
	private final DreamMessagesProperties messagesProperties;

	public DreamErrorMvcAutoConfiguration(
		ServerProperties serverProperties,
		ObjectMapper objectMapper,
		DreamProperties properties,
		DreamMessagesProperties messagesProperties) {
		this.serverProperties = serverProperties;
		this.objectMapper = objectMapper;
		this.properties = properties;
		this.messagesProperties = messagesProperties;
	}

	@Bean
	@ConditionalOnMissingBean(value = ExceptionTranslator.class, search = SearchStrategy.CURRENT)
	public ExceptionTranslator exceptionTranslator() {
		MessageSource messageSource = getMessageSource(messagesProperties);
		return new ExceptionTranslator(messageSource);
	}

	@Bean
	@ConditionalOnMissingBean(value = ErrorAttributes.class, search = SearchStrategy.CURRENT)
	public DefaultErrorAttributes errorAttributes(ExceptionTranslator translator) {
		return new DreamErrorAttributes(translator, properties);
	}

	@Bean
	@ConditionalOnMissingBean(value = ErrorController.class, search = SearchStrategy.CURRENT)
	public BasicErrorController basicErrorController(ErrorAttributes errorAttributes) {
		return new DreamErrorController(errorAttributes, this.serverProperties.getError(), this.properties, this.objectMapper);
	}

	public MessageSource getMessageSource(DreamMessagesProperties properties) {
		ReloadableResourceBundleMessageSource messageSource = new ReloadableResourceBundleMessageSource();
		messageSource.setBasename(properties.getException().getBasename());
		messageSource.setCacheSeconds(properties.getCacheSeconds());
		messageSource.setDefaultEncoding(properties.getEncoding().name());
		messageSource.setAlwaysUseMessageFormat(properties.isAlwaysUseMessageFormat());
		messageSource.setFallbackToSystemLocale(properties.isFallbackToSystemLocale());
		return messageSource;
	}

	@Bean
	public MethodValidationPostProcessor methodValidationPostProcessor() {
		return new MethodValidationPostProcessor();
	}
}
