package net.dreamlu.boot.error;

import lombok.extern.slf4j.Slf4j;
import net.dreamlu.boot.properties.DreamProperties;
import net.dreamlu.boot.result.Result;
import net.dreamlu.boot.result.Results;
import net.dreamlu.tool.util.BeanUtils;
import net.dreamlu.tool.util.JsonUtils;
import org.springframework.boot.autoconfigure.web.DefaultErrorAttributes;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.servlet.ModelAndView;
import org.springframework.web.servlet.view.json.MappingJackson2JsonView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Map;

/**
 * 全局异常处理
 *
 * @author L.cm
 */
@Slf4j
@SuppressWarnings("unchecked")
public class DreamErrorAttributes extends DefaultErrorAttributes {

	private final ExceptionTranslator translator;
	private final DreamProperties properties;

	public DreamErrorAttributes(ExceptionTranslator translator, DreamProperties properties) {
		this.translator = translator;
		this.properties = properties;
	}

	@Override
	public ModelAndView resolveException(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
		super.resolveException(request, response, handler, ex);
		// 异常翻译
		Result<String> result = translator.handle(ex);
		MappingJackson2JsonView view = new MappingJackson2JsonView();
		view.setObjectMapper(JsonUtils.getInstance());
		view.setContentType(properties.getMvc().getContentType());
		return new ModelAndView(view, BeanUtils.toMap(result));
	}

	@Override
	public Map<String, Object> getErrorAttributes(RequestAttributes requestAttributes, boolean includeStackTrace) {
		Throwable error = getError(requestAttributes);
		Result<String> result;
		if (error == null) {
			Integer status = getAttribute(requestAttributes, "javax.servlet.error.status_code");
			result = Results.failure(String.valueOf(status));
		} else {
			result = translator.handle(error);
		}
		return BeanUtils.toMap(result);
	}

	private <T> T getAttribute(RequestAttributes requestAttributes, String name) {
		return (T) requestAttributes.getAttribute(name, RequestAttributes.SCOPE_REQUEST);
	}
}
