package net.dreamlu.boot.config;

import net.dreamlu.boot.properties.TransactionalProperties;
import org.springframework.aop.aspectj.annotation.AnnotationAwareAspectJAutoProxyCreator;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.interceptor.TransactionInterceptor;

import java.util.*;

@Configuration
@EnableConfigurationProperties(TransactionalProperties.class)
@ConditionalOnClass(name = "org.springframework.transaction.PlatformTransactionManager")
@ConditionalOnProperty(value = "dream.transactional.enabled", havingValue = "true", matchIfMissing = true)
public class TransactionalConfiguration {
	private static final String TRANSACTION_INTERCEPTOR_NAME = "transactionAdvice";

	private final TransactionalProperties transactionalProperties;
	private final PlatformTransactionManager transactionManager;

	public TransactionalConfiguration(TransactionalProperties transactionalProperties, PlatformTransactionManager transactionManager) {
		this.transactionalProperties = transactionalProperties;
		this.transactionManager = transactionManager;
	}

	@Bean(TRANSACTION_INTERCEPTOR_NAME)
	public TransactionInterceptor transactionInterceptor() {
		Properties attributes = new Properties();
		attributes.putAll(transactionalProperties.getTxAttributes());
		attributes.setProperty("find*", "PROPAGATION_REQUIRED,readOnly");
		attributes.setProperty("select*", "PROPAGATION_REQUIRED,readOnly");
		attributes.setProperty("insert*", "PROPAGATION_REQUIRED,-Exception");
		attributes.setProperty("update*", "PROPAGATION_REQUIRED,-Exception");
		attributes.setProperty("delete*", "PROPAGATION_REQUIRED,-Exception");
		attributes.setProperty("*", "PROPAGATION_REQUIRED,-Exception");
		return new TransactionInterceptor(transactionManager, attributes);
	}

	@Bean
	public AnnotationAwareAspectJAutoProxyCreator transactionProxyCreator(){
		AnnotationAwareAspectJAutoProxyCreator c = new AnnotationAwareAspectJAutoProxyCreator();
		c.setInterceptorNames(TRANSACTION_INTERCEPTOR_NAME);
		c.setIncludePatterns(Arrays.asList(transactionalProperties.getExecutionPatterns()));
		c.setProxyTargetClass(true);
		c.setOrder(2);
		return c;
	}

}
