package net.dreamlu.boot.config;

import io.swagger.annotations.Api;
import net.dreamlu.boot.properties.DreamSwaggerProperties;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.env.Environment;
import org.springframework.web.servlet.config.annotation.ResourceHandlerRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;
import springfox.bean.validators.configuration.BeanValidatorPluginsConfiguration;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.*;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spi.service.contexts.SecurityContext;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger.web.ApiKeyVehicle;
import springfox.documentation.swagger2.configuration.Swagger2DocumentationConfiguration;

import java.util.Collections;
import java.util.List;

/**
 * Swagger2配置
 *
 * @author L.cm
 */
@Configuration
@ConditionalOnClass({
	Swagger2DocumentationConfiguration.class,
	BeanValidatorPluginsConfiguration.class
})
@Import({
	Swagger2DocumentationConfiguration.class,
	BeanValidatorPluginsConfiguration.class
})
@AutoConfigureAfter(DreamSwaggerProperties.class)
public class SwaggerConfiguration extends WebMvcConfigurerAdapter implements EnvironmentAware {
	private final DreamSwaggerProperties properties;
	private Environment environment;

	public SwaggerConfiguration(DreamSwaggerProperties properties) {
		this.properties = properties;
	}

	@Bean
	public Docket createRestApi() {
		// 组名为应用名
		String appName = environment.getProperty("spring.application.name", String.class);
		Docket docket = new Docket(DocumentationType.SWAGGER_2)
			.useDefaultResponseMessages(false)
			.groupName(appName)
			.apiInfo(apiInfo()).select()
			.apis(RequestHandlerSelectors.withClassAnnotation(Api.class))
			.paths(PathSelectors.any())
			.build();
		// 如果开启认证
		if (properties.getAuthorization().getEnabled()) {
			docket.securitySchemes(Collections.singletonList(apiKey()));
			docket.securityContexts(Collections.singletonList(securityContext()));
		}
		return docket;
	}

	/**
	 * 配置基于 ApiKey 的鉴权对象
	 *
	 * @return {ApiKey}
	 */
	private ApiKey apiKey() {
		return new ApiKey(properties.getAuthorization().getName(),
			properties.getAuthorization().getKeyName(),
			ApiKeyVehicle.HEADER.getValue());
	}

	/**
	 * 配置默认的全局鉴权策略的开关，以及通过正则表达式进行匹配；默认 ^.*$ 匹配所有URL
	 * 其中 securityReferences 为配置启用的鉴权策略
	 *
	 * @return {SecurityContext}
	 */
	private SecurityContext securityContext() {
		return SecurityContext.builder()
			.securityReferences(defaultAuth())
			.forPaths(PathSelectors.regex(properties.getAuthorization().getAuthRegex()))
			.build();
	}

	/**
	 * 配置默认的全局鉴权策略；其中返回的 SecurityReference 中，reference 即为ApiKey对象里面的name，保持一致才能开启全局鉴权
	 *
	 * @return {List<SecurityReference>}
	 */
	private List<SecurityReference> defaultAuth() {
		AuthorizationScope authorizationScope = new AuthorizationScope("global", "accessEverything");
		AuthorizationScope[] authorizationScopes = new AuthorizationScope[1];
		authorizationScopes[0] = authorizationScope;
		return Collections.singletonList(SecurityReference.builder()
			.reference(properties.getAuthorization().getName())
			.scopes(authorizationScopes).build());
	}

	private ApiInfo apiInfo() {
		return new ApiInfoBuilder()
			.title(properties.getTitle())
			.description(properties.getDescription())
			.version(properties.getVersion())
			.contact(new Contact(properties.getContactUser(), properties.getContactUrl(), properties.getContactEmail()))
			.build();
	}

	@Override
	public void addResourceHandlers(ResourceHandlerRegistry registry) {
		registry.addResourceHandler("/swagger-ui.html")
			.addResourceLocations("classpath:/META-INF/resources/");
		registry.addResourceHandler("/webjars*")
			.addResourceLocations("classpath:/META-INF/resources/webjars/");
	}

	@Override
	public void setEnvironment(Environment environment) {
		this.environment = environment;
	}
}
