package net.dreamlu.boot.template;

import net.dreamlu.boot.exception.LocalizedException;
import net.dreamlu.boot.properties.DreamProperties;
import net.dreamlu.tool.util.IOUtils;
import net.dreamlu.tool.util.SpringUtils;
import net.dreamlu.tool.util.Try;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;
import org.springframework.util.ClassUtils;

import javax.script.*;
import java.io.*;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

/**
 * lay 服务器端模板
 *
 * @author L.cm
 */
public class DreamTemplate implements ApplicationContextAware, InitializingBean {
	private final ConcurrentMap<String, String> tplCache = new ConcurrentHashMap<>();
	private final Function<String, String> tplFunction = Try.of(x -> {
		Resource resource = DreamTemplate.this.getApplicationContext().getResource(x);
		return IOUtils.toString(resource.getInputStream());
	});
	private final DreamProperties dreamProperties;
	private ApplicationContext applicationContext;
	private ScriptEngine engine;
	private DreamConsole console;

	public DreamTemplate(DreamProperties dreamProperties) {
		this.dreamProperties = dreamProperties;
		this.console = new DreamConsole();
	}

	/**
	 * 渲染html字符串
	 * @param tplName 模板名称
	 * @param data 数据模型
	 * @return 渲染后的html
	 */
	public String renderTpl(String tplName, Object data) {
		if (tplName.startsWith("/")) {
			tplName = tplName.substring(1);
		}
		DreamProperties.Tpl tpl = dreamProperties.getTpl();
		final String tplPath = tpl.getPrefix() + tplName;
		try {
			String html = tpl.isCache()
				? tplCache.computeIfAbsent(tplPath, tplFunction)
				: tplFunction.apply(tplPath);
			return _render(html, data);
		} catch (ScriptException e) {
			throw new LocalizedException("error.tpl.Script", e.getMessage());
		} catch (Exception e) {
			throw new LocalizedException("error.tpl.FileNotFound", e.getMessage());
		}
	}

	/**
	 * 渲染html字符串
	 * @param html html字符串
	 * @param data 数据模型
	 * @return 渲染后的html
	 */
	public String render(String html, Object data) {
		try {
			return _render(html, data);
		} catch (ScriptException e) {
			throw new LocalizedException("error.tpl.Script", e.getMessage());
		}
	}

	private String _render(String html, Object data) throws ScriptException {
		// 避免多线程问题
		Bindings bindings = engine.createBindings();
		bindings.put("_html_", html);
		bindings.put("data", data);
		return (String) engine.eval("laytpl(_html_).render(data);", bindings);
	}

	@Override
	public void afterPropertiesSet() throws Exception {
		final ScriptEngineManager engineManager = new ScriptEngineManager();
		final ScriptEngine engine = engineManager.getEngineByMimeType("text/javascript");
		Bindings bindings = engine.createBindings();
		try (
			final InputStream input = DreamTemplate.class.getResourceAsStream("laytpl.js");
			final Reader reader = new BufferedReader(new InputStreamReader(input));
		) {
			Map<String, String> config = new HashMap<>();
			DreamProperties.Tpl tpl = dreamProperties.getTpl();
			config.put("open", tpl.getOpen());
			config.put("close", tpl.getClose());
			bindings.put("console", console);
			bindings.put("dream", new DreamContext(applicationContext));
			bindings.put("_config", config);
			engine.setBindings(bindings, ScriptContext.GLOBAL_SCOPE);
			engine.eval(reader, bindings);
			this.engine = engine;
		}
		this.engine.eval("console.log('DreamTpl init, laytpl version:{}', laytpl.v);");
	}

	@Override
	public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
		this.applicationContext = applicationContext;
	}

	public ApplicationContext getApplicationContext() {
		return applicationContext;
	}
}
