package net.dreamlu.boot.error;

import lombok.extern.slf4j.Slf4j;
import net.dreamlu.boot.exception.LocalizedException;
import net.dreamlu.tool.util.Exceptions;
import org.springframework.context.MessageSource;
import org.springframework.context.NoSuchMessageException;
import org.springframework.context.i18n.LocaleContextHolder;
import org.springframework.http.MediaType;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpMediaTypeNotSupportedException;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.MissingServletRequestParameterException;
import org.springframework.web.servlet.NoHandlerFoundException;

import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import java.util.Locale;
import java.util.Set;

/**
 * 处理异常国际化
 *
 * @author L.cm
 */
@Slf4j
public class ExceptionTranslator {
	private final MessageSource messageSource;

	public ExceptionTranslator(MessageSource messageSource) {
		this.messageSource = messageSource;
	}

	/**
	 * 异常处理
	 *
	 * 1. 优先读取国际化
	 * 2. 国际化没有时给预设的异常信息
	 * 3. 原本的异常信息
	 *
	 * @param error Throwable
	 * @return 友好的异常消息
	 */
	public String handle(Throwable error) {
		Locale locale = LocaleContextHolder.getLocale();
		Throwable _error = Exceptions.unwrap(error);
		return handleException(_error, locale);
	}

	protected String handleException(Throwable error, Locale locale) {
		String message = null;
		if (error instanceof MissingServletRequestParameterException) {
			message = handleError((MissingServletRequestParameterException) error, locale);
		} else if (error instanceof MethodArgumentNotValidException) {
			message = handleError((MethodArgumentNotValidException) error);
		} else if (error instanceof BindException) {
			message = handleError((BindException) error);
		} else if (error instanceof ConstraintViolationException) {
			message = handleError((ConstraintViolationException) error);
		} else if (error instanceof NoHandlerFoundException) {
			message = handleError((NoHandlerFoundException) error, locale);
		} else if (error instanceof HttpRequestMethodNotSupportedException) {
			message = handleError((HttpRequestMethodNotSupportedException) error, locale);
		} else if (error instanceof HttpMediaTypeNotSupportedException) {
			message = handleError((HttpMediaTypeNotSupportedException) error, locale);
		} else if (error instanceof LocalizedException) {
			message = handleError((LocalizedException) error, locale);
		} else {
			message = handleError(error, locale);
		}
		if (null == message) {
			message = error.getClass().getSimpleName();
		}
		return message;
	}

	protected String handleError(MissingServletRequestParameterException e, Locale locale) {
		log.error("缺少请求参数", e);
		String parameterName = e.getParameterName();
		String parameterType = e.getParameterType();
		return getMessage(e, locale, new Object[]{parameterName, parameterType});
	}

	protected String handleError(MethodArgumentNotValidException e) {
		log.error("参数验证失败", e);
		BindingResult result = e.getBindingResult();
		FieldError error = result.getFieldError();
		return error.getDefaultMessage();
	}

	protected String handleError(BindException e) {
		log.error("参数绑定失败", e);
		BindingResult result = e.getBindingResult();
		FieldError error = result.getFieldError();
		return error.getDefaultMessage();
	}

	protected String handleError(ConstraintViolationException e) {
		log.error("参数验证失败", e);
		Set<ConstraintViolation<?>> violations = e.getConstraintViolations();
		ConstraintViolation<?> violation = violations.iterator().next();
		return violation.getMessage();
	}

	protected String handleError(NoHandlerFoundException e, Locale locale) {
		log.error("404没找到请求", e);
		String httpMethod = e.getHttpMethod();
		String requestURL = e.getRequestURL();
		return getMessage(e, locale, new Object[]{httpMethod, requestURL});
	}

	protected String handleError(HttpRequestMethodNotSupportedException e, Locale locale) {
		log.error("不支持当前请求方法", e);
		String method = e.getMethod();
		return getMessage(e, locale, new Object[]{method});
	}

	protected String handleError(HttpMediaTypeNotSupportedException e, Locale locale) {
		log.error("不支持当前媒体类型", e);
		MediaType contentType = e.getContentType();
		return getMessage(e, locale, new Object[]{contentType.getType()});
	}

	protected String handleError(LocalizedException e, Locale locale) {
		log.error("本地化异常", e);
		return getMessage(e, locale, e.getLocaleArgs());
	}

	protected String handleError(Throwable e, Locale locale) {
		log.error("通用异常", e);
		return getMessage(e, locale, new Object[0]);
	}

	protected String getMessage(Throwable e, Locale locale, Object[] values) {
		String msgKey = e instanceof LocalizedException ?
			((LocalizedException) e).getLocaleMessage() : e.getClass().getName();
		try {
			return messageSource.getMessage(msgKey, values, locale);
		} catch (NoSuchMessageException ex) {
			log.error(ex.getMessage());
			return e.getMessage();
		}
	}

}
