package net.dreamlu.boot.cache.http;

import lombok.extern.slf4j.Slf4j;
import net.dreamlu.boot.properties.DreamProperties;
import net.dreamlu.tool.util.ClassUtils;
import net.dreamlu.tool.util.StringUtils;
import net.dreamlu.tool.util.WebUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.Order;
import org.springframework.expression.Expression;
import org.springframework.expression.ExpressionParser;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.expression.spel.support.SimpleEvaluationContext;
import org.springframework.http.HttpMethod;
import org.springframework.util.Assert;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import java.time.Clock;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * 由于时间仓促写了个简版的
 *
 * 尚未处理 etag
 *
 * @author L.cm
 */
@Aspect
@Order
@Slf4j
public class HttpCacheAspect  {
	/**
	 * 表达式处理
	 */
	private static final ExpressionParser elParser = new SpelExpressionParser();

	@Around("@annotation(cacheAble)")
	public Object aroundWxApi(ProceedingJoinPoint point, HttpCacheAble cacheAble) throws Throwable {
		handleHttpCacheAble(point, cacheAble);
		return point.proceed();
	}

	/**
	 * 处理http cache
	 * @param point 切点
	 * @param cacheAble httpCache注解
	 */
	private void handleHttpCacheAble(ProceedingJoinPoint point, HttpCacheAble cacheAble) {
		HttpServletRequest request = WebUtils.getRequest();
		// http cache 针对 HEAD 和 GET 请求
		String method = request.getMethod();
		HttpMethod httpMethod = HttpMethod.resolve(method);
		if (httpMethod == null) {
			return;
		}
		List<HttpMethod> allowList = Arrays.asList(HttpMethod.HEAD, HttpMethod.GET);
		if (allowList.indexOf(httpMethod) == -1) {
			return;
		}
		// 缓存时间,秒
		long maxAge = cacheAble.maxAge();
		if (maxAge < 1) {
			return;
		}

		// 判断表达式
		String condition = cacheAble.condition();
		if (StringUtils.isNotBlank(condition)) {
			Expression expression = elParser.parseExpression(condition);
			SimpleEvaluationContext context = getEvaluationContext(point);
			boolean useHttpCache = expression.getValue(context, Boolean.class);
			if (!useHttpCache) {
				return;
			}
		}
		// 最后修改时间
		long ims = request.getDateHeader("If-Modified-Since");
		long now = Clock.systemUTC().millis();

		// 缓存时间,毫秒
		long maxAgeMicros = TimeUnit.SECONDS.toMillis(maxAge);
		HttpServletResponse response = WebUtils.getResponse();
		// 如果header头没有过期
		long expires, lastModified;
		if (ims + maxAgeMicros > now) {
			response.setStatus(HttpServletResponse.SC_NOT_MODIFIED);
			expires = ims + maxAgeMicros;
			lastModified = ims;
			log.info("{} 304 {}", method, request.getRequestURI());
		} else {
			expires = now + maxAgeMicros;
			lastModified = now;
		}
		response.setHeader("Cache-Control", "max-age=" + maxAge);
		response.addDateHeader("Expires", expires);
		response.addDateHeader("Last-Modified", lastModified);
	}

	/**
	 * 获取方法上的参数
	 * @param point 切点
	 * @return {SimpleEvaluationContext}
	 */
	private SimpleEvaluationContext getEvaluationContext(ProceedingJoinPoint point) {
		SimpleEvaluationContext context = SimpleEvaluationContext.forReadOnlyDataBinding().build();
		MethodSignature ms = (MethodSignature) point.getSignature();
		Method method = ms.getMethod();
		Object[] args = point.getArgs();
		for (int i = 0; i < args.length; i++) {
			MethodParameter methodParam = ClassUtils.getMethodParameter(method, i);
			Object value = args[i];
			context.setVariable(methodParam.getParameterName(), value);
		}
		return context;
	}
}
