/*
 * Copyright 2002-2005 Uwyn bvba/sprl <info[remove] at uwyn dot com>
 * Distributed under the terms of the GNU Lesser General Public
 * License, v2.1 or later
 *
 * $Id: ServerInfo.java 1319 2005-02-13 11:55:44Z gbevin $
 */
package com.uwyn.drone.core;

import com.uwyn.drone.core.exceptions.CoreException;
import com.uwyn.drone.core.exceptions.UnknownServerAddressException;
import com.uwyn.drone.core.exceptions.UnknownServerHostnameException;
import com.uwyn.rife.tools.StringUtils;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.nio.charset.UnsupportedCharsetException;
import java.util.ArrayList;
import java.util.Collection;

class ServerInfo
{
	private static final int	DEFAULT_MAX = 1024;
	private static final int	DEFAULT_AMOUNT = 25;
	private static final int	DEFAULT_INTERVAL = 1000;
	private static final int	DEFAULT_TIMEOUT = 5 * 60 * 1000;
	private static final String	DEFAULT_CHARSET = "UTF-8";
	
	private ArrayList	mAddresses = null;
	private int			mMax = 0;
	private int			mAmount = 0;
	private int			mInterval = 0;
	private int			mTimeout = 0;
	private String		mCharset = null;

	ServerInfo()
	{
		mAddresses = new ArrayList();
		mMax = DEFAULT_MAX;
		mAmount = DEFAULT_AMOUNT;
		mInterval = DEFAULT_INTERVAL;
		mTimeout = DEFAULT_TIMEOUT;
		mCharset = DEFAULT_CHARSET;
	}
	
	Collection getAddresses()
	{
		return mAddresses;
	}
	
	int getMax()
	{
		return mMax;
	}
	
	int getAmount()
	{
		return mAmount;
	}
	
	int getInterval()
	{
		return mInterval;
	}
	
	int getTimeout()
	{
		return mTimeout;
	}
	
	String getCharset()
	{
		return mCharset;
	}
	
	void setOutput(int max, int amount, int interval)
	{
		if (max <= 0)		throw new IllegalArgumentException("max has to be bigger than 0.");
		if (max <= amount)	throw new IllegalArgumentException("max has to be bigger than amount.");
		if (amount <= 0)	throw new IllegalArgumentException("amount has to be bigger than 0.");
		if (interval <= 0)	throw new IllegalArgumentException("interval has to be bigger than 0.");
			
		mMax = max;
		mAmount = amount;
		mInterval = interval;
	}

	void addAddress(String host, int port)
	throws CoreException
	{
		if (null == host)		throw new IllegalArgumentException("host can't be null.");
		if (0 == host.length())	throw new IllegalArgumentException("host can't be empty.");
		
		// detect ip addresses
		byte[] parts = StringUtils.splitToByteArray(host, ".");
		if (4 == parts.length)
		{
			addAddress(parts, port);
			return;
		}

		// no ip address, thus a regular hostname
		InetAddress address = null;
		
		try
		{
			address = InetAddress.getByName(host);
		}
		catch (UnknownHostException e)
		{
			throw new UnknownServerHostnameException(host, e);
		}

		addAddress(address, port);
	}

	void addAddress(byte[] ip, int port)
	throws CoreException
	{
		if (null == ip)	throw new IllegalArgumentException("ip can't be null.");

		InetAddress address = null;
		
		try
		{
			address = InetAddress.getByAddress(ip);
		}
		catch (UnknownHostException e)
		{
			throw new UnknownServerAddressException(ip, e);
		}
		
		addAddress(address, port);
	}
	
	synchronized void addAddress(InetAddress address, int port)
	{
		assert address != null;
		assert port >= 0;
		
		mAddresses.add(new InetSocketAddress(address, port));
	}
	
	void setTimeout(int timeout)
	{
		mTimeout = timeout;
	}
	
	void setCharset(String charset)
	throws UnsupportedCharsetException
	{
		if (!Charset.isSupported(charset))
		{
			throw new UnsupportedCharsetException(charset);
		}
		
		mCharset = charset;
	}
}
