package code.complete.log;

import android.support.annotation.NonNull;
import android.support.annotation.Size;

/**
 * Log Info
 * @author imkarl
 */
public final class LogInfo {
    private int priority;
    private String tag;
    private long time;
    private String threadName;
    private StackTraceElement[] callerStackTrace;

    private LogInfo() {
    }

    @NonNull
    static LogInfo obtain(@LogPriority int priority, String tag, long time, String threadName, StackTraceElement[] callerStackTrace) {
        LogInfo infos = new LogInfo();
        infos.priority = priority;
        infos.tag = tag;
        infos.time = time;
        infos.threadName = threadName;
        infos.callerStackTrace = callerStackTrace;
        return infos;
    }

    /**
     * 获取调用者信息
     */
    @NonNull
    public CharSequence getCallerString() {
        final StackTraceElement element = callerStackTrace[0];
        final String className = element.getClassName();

        StringBuilder content = new StringBuilder(className.substring(className.lastIndexOf(".") + 1));
        content.append('.').append(element.getMethodName());
        if (element.isNativeMethod()) {
            return content.append("(Native Method)");
        }
        if (element.getFileName() != null && element.getLineNumber() >= 0) {
            return content.append('(').append(element.getFileName()).append(':').append(element.getLineNumber()).append(')');
        }
        if (element.getFileName() != null) {
            return content.append('(').append(element.getFileName()).append(')');
        }
        return content.append("(Unknown Source)");
    }

    void recycle() {
        tag = null;
        threadName = null;
        callerStackTrace = null;
    }

    @LogPriority
    public int getPriority() {
        return priority;
    }
    @NonNull
    public String getTag() {
        return tag;
    }
    public long getTime() {
        return time;
    }
    @NonNull
    public String getThreadName() {
        return threadName;
    }
    @NonNull @Size(min=1)
    public StackTraceElement[] getCallerStackTrace() {
        return callerStackTrace;
    }
}
