package cn.zoecloud.core.operation;

import cn.zoecloud.core.ClientException;
import cn.zoecloud.core.ServiceException;
import cn.zoecloud.core.auth.Credential;
import cn.zoecloud.core.model.*;
import cn.zoecloud.core.service.HttpMethod;
import cn.zoecloud.core.service.RequestMessage;
import cn.zoecloud.core.service.ServiceClient;
import com.alibaba.fastjson.JSON;

import static cn.zoecloud.core.internal.ZoeCloudResponseParsers.commandResultListResponseParser;

/**
 * 命令的操作
 * @author Leo
 */
public class ZoeCloudCommandOperation extends ZoeCloudOperation {
    private static final String COMMAND_PATH = "/api/2.0/commands";
    private static final String COMMAND_RESULT_PATH = COMMAND_PATH + "/results";
    private static final String COMMAND_OPEN_SCREEN_PATH = COMMAND_PATH + "/screen/open";
    private static final String COMMAND_CLOSE_SCREEN_PATH = COMMAND_PATH + "/screen/close";
    private static final String COMMAND_FETCH_SCREEN_STATUS_PATH = COMMAND_PATH + "/screen/status";
    private static final String COMMAND_FETCH_PROGRAMS_PATH = COMMAND_PATH + "/programs/fetch";
    private static final String COMMAND_CAPTURE_SCREEN_PATH = COMMAND_PATH + "/screen/capture";

    public ZoeCloudCommandOperation(ServiceClient client, Credential credential, String endpoint) {
        super(client, credential, endpoint);
    }

    /**
     * 获取命令执行结果
     * @param getCommandResultListRequest
     * @return
     * @throws ServiceException
     * @throws ClientException
     */
    public CommandResultListResult getCommandResults(GetCommandResultListRequest getCommandResultListRequest)
            throws ServiceException, ClientException {
        assert (getCommandResultListRequest != null);
        assert (getCommandResultListRequest.getBody() != null);

        RequestMessage request = new RequestMessage(this.getEndpoint(), COMMAND_RESULT_PATH);
        request.setMethod(HttpMethod.POST);
        request.setBody(JSON.toJSONString(getCommandResultListRequest.getBody()));
        return doOperation(request, commandResultListResponseParser);
    }

    /**
     * 打开屏幕
     * @param commandOpenScreenRequest
     * @return
     */
    public CommandResultListResult openScreen(CommandOpenScreenRequest commandOpenScreenRequest) {
        return this.sendSimpleRequest(commandOpenScreenRequest, COMMAND_OPEN_SCREEN_PATH);
    }

    /**
     * 关闭屏幕
     * @param commandCloseScreenRequest
     * @return
     */
    public CommandResultListResult closeScreen(CommandCloseScreenRequest commandCloseScreenRequest) {
        return this.sendSimpleRequest(commandCloseScreenRequest, COMMAND_CLOSE_SCREEN_PATH);
    }

    /**
     * 获取屏幕状态
     * @param commandFetchScreenStatusRequest
     * @return
     */
    public CommandResultListResult fetchScreenStatus(CommandFetchScreenStatusRequest commandFetchScreenStatusRequest) {
        return this.sendSimpleRequest(commandFetchScreenStatusRequest, COMMAND_FETCH_SCREEN_STATUS_PATH);
    }

    /**
     * 获取节目信息
     * @param commandFetchProgramsRequest
     * @return
     */
    public CommandResultListResult fetchPrograms(CommandFetchProgramsRequest commandFetchProgramsRequest) {
        return this.sendSimpleRequest(commandFetchProgramsRequest, COMMAND_FETCH_PROGRAMS_PATH);
    }

    /**
     * 屏幕截图
     * @param commandCaptureScreenRequest
     * @return
     */
    public CommandResultListResult captureScreen(CommandCaptureScreenRequest commandCaptureScreenRequest) {
        return this.sendSimpleRequest(commandCaptureScreenRequest, COMMAND_CAPTURE_SCREEN_PATH);
    }

    /**
     * 发送简单的命令请求
     * @param request
     * @param commandPath
     * @return
     */
    private CommandResultListResult sendSimpleRequest(BaseCommandRequest request, String commandPath) {
        this.checkRequest(request);

        RequestMessage requestMessage = new RequestMessage(this.getEndpoint(), commandPath);
        requestMessage.setMethod(HttpMethod.POST);
        requestMessage.setBody(JSON.toJSONString(request));
        return doOperation(requestMessage, commandResultListResponseParser);
    }

    /**
     * 检查请求信息
     * @param request
     */
    private void checkRequest(BaseCommandRequest request) {
        assert (request != null);

        if (request.getTimeout() == null
                || request.getTimeout() <= 0
                || request.getTimeout() > 5) {
            throw new ClientException("The timeout should be in the range of [1-5].");
        }
    }
}
