/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.yrtietfrestconf.rev20170126.yrtietfrestconf.restconf;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v11.yrtietfrestconf.rev20170126.yrtietfrestconf.restconf.restconf.Data;
import org.onosproject.yang.gen.v11.yrtietfrestconf.rev20170126.yrtietfrestconf.restconf.restconf.Operations;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of restconf.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultRestconf extends InnerModelObject implements Restconf {
    protected String yangLibraryVersion;
    protected BitSet valueLeafFlags = new BitSet();
    protected Data data;
    protected Operations operations;

    @Override
    public String yangLibraryVersion() {
        return yangLibraryVersion;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public Data data() {
        return data;
    }

    @Override
    public Operations operations() {
        return operations;
    }

    @Override
    public void yangLibraryVersion(String yangLibraryVersion) {
        valueLeafFlags.set(LeafIdentifier.YANGLIBRARYVERSION.getLeafIndex());
        this.yangLibraryVersion = yangLibraryVersion;
    }

    @Override
    public void data(Data data) {
        this.data = data;
    }

    @Override
    public void operations(Operations operations) {
        this.operations = operations;
    }

    @Override
    public int hashCode() {
        return Objects.hash(yangLibraryVersion, valueLeafFlags, data, operations);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultRestconf) {
            DefaultRestconf other = (DefaultRestconf) obj;
            return
                Objects.deepEquals(yangLibraryVersion, other.yangLibraryVersion) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(data, other.data) &&
                Objects.deepEquals(operations, other.operations);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("yangLibraryVersion=" + yangLibraryVersion)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("data=" + data)
            .add("operations=" + operations)
            .toString();
    }


    /**
     * Creates an instance of defaultRestconf.
     */
    public DefaultRestconf() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}