/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.tunnelp2pparamsconfig;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.TeNodeId;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.TeTopologyId;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.TeTpId;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of hierarchicalLinkId.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultHierarchicalLinkId extends InnerModelObject implements HierarchicalLinkId {
    protected TeNodeId localTeNodeId;
    protected TeTpId localTeLinkTpId;
    protected TeNodeId remoteTeNodeId;
    protected TeTopologyId teTopologyId;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public TeNodeId localTeNodeId() {
        return localTeNodeId;
    }

    @Override
    public TeTpId localTeLinkTpId() {
        return localTeLinkTpId;
    }

    @Override
    public TeNodeId remoteTeNodeId() {
        return remoteTeNodeId;
    }

    @Override
    public TeTopologyId teTopologyId() {
        return teTopologyId;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void localTeNodeId(TeNodeId localTeNodeId) {
        valueLeafFlags.set(LeafIdentifier.LOCALTENODEID.getLeafIndex());
        this.localTeNodeId = localTeNodeId;
    }

    @Override
    public void localTeLinkTpId(TeTpId localTeLinkTpId) {
        valueLeafFlags.set(LeafIdentifier.LOCALTELINKTPID.getLeafIndex());
        this.localTeLinkTpId = localTeLinkTpId;
    }

    @Override
    public void remoteTeNodeId(TeNodeId remoteTeNodeId) {
        valueLeafFlags.set(LeafIdentifier.REMOTETENODEID.getLeafIndex());
        this.remoteTeNodeId = remoteTeNodeId;
    }

    @Override
    public void teTopologyId(TeTopologyId teTopologyId) {
        valueLeafFlags.set(LeafIdentifier.TETOPOLOGYID.getLeafIndex());
        this.teTopologyId = teTopologyId;
    }

    @Override
    public int hashCode() {
        return Objects.hash(localTeNodeId, localTeLinkTpId, remoteTeNodeId, teTopologyId, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultHierarchicalLinkId) {
            DefaultHierarchicalLinkId other = (DefaultHierarchicalLinkId) obj;
            return
                Objects.deepEquals(localTeNodeId, other.localTeNodeId) &&
                Objects.deepEquals(localTeLinkTpId, other.localTeLinkTpId) &&
                Objects.deepEquals(remoteTeNodeId, other.remoteTeNodeId) &&
                Objects.deepEquals(teTopologyId, other.teTopologyId) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("localTeNodeId=" + localTeNodeId)
            .add("localTeLinkTpId=" + localTeLinkTpId)
            .add("remoteTeNodeId=" + remoteTeNodeId)
            .add("teTopologyId=" + teTopologyId)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultHierarchicalLinkId.
     */
    public DefaultHierarchicalLinkId() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}