/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.tepathproperties;

import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.actnietftetypes.rev20170310.actnietftetypes.TeBandwidth;
import org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.pathaffinities.PathAffinities;
import org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.pathsrlgs.PathSrlgs;
import org.onosproject.yang.gen.v11.actnietfte.rev20170310.actnietfte.tepathproperties.pathproperties.Metric;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of pathProperties.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultPathProperties extends InnerModelObject implements PathProperties {
    protected TeBandwidth bandwidthGeneric;
    protected short setupPriority;
    protected short holdPriority;
    protected boolean ignoreOverload;
    protected BitSet valueLeafFlags = new BitSet();
    protected List<Metric> metric;
    protected PathAffinities pathAffinities;
    protected PathSrlgs pathSrlgs;

    @Override
    public TeBandwidth bandwidthGeneric() {
        return bandwidthGeneric;
    }

    @Override
    public short setupPriority() {
        return setupPriority;
    }

    @Override
    public short holdPriority() {
        return holdPriority;
    }

    @Override
    public boolean ignoreOverload() {
        return ignoreOverload;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public List<Metric> metric() {
        return metric;
    }

    @Override
    public PathAffinities pathAffinities() {
        return pathAffinities;
    }

    @Override
    public PathSrlgs pathSrlgs() {
        return pathSrlgs;
    }

    @Override
    public void bandwidthGeneric(TeBandwidth bandwidthGeneric) {
        valueLeafFlags.set(LeafIdentifier.BANDWIDTHGENERIC.getLeafIndex());
        this.bandwidthGeneric = bandwidthGeneric;
    }

    @Override
    public void setupPriority(short setupPriority) {
        valueLeafFlags.set(LeafIdentifier.SETUPPRIORITY.getLeafIndex());
        this.setupPriority = setupPriority;
    }

    @Override
    public void holdPriority(short holdPriority) {
        valueLeafFlags.set(LeafIdentifier.HOLDPRIORITY.getLeafIndex());
        this.holdPriority = holdPriority;
    }

    @Override
    public void ignoreOverload(boolean ignoreOverload) {
        valueLeafFlags.set(LeafIdentifier.IGNOREOVERLOAD.getLeafIndex());
        this.ignoreOverload = ignoreOverload;
    }

    @Override
    public void metric(List<Metric> metric) {
        this.metric = metric;
    }

    @Override
    public void pathAffinities(PathAffinities pathAffinities) {
        this.pathAffinities = pathAffinities;
    }

    @Override
    public void pathSrlgs(PathSrlgs pathSrlgs) {
        this.pathSrlgs = pathSrlgs;
    }

    @Override
    public void addToMetric(Metric addTo) {
        if (metric == null) {
            metric = new ArrayList<>();
        }
        metric.add(addTo);
    }


    @Override
    public int hashCode() {
        return Objects.hash(bandwidthGeneric, setupPriority, holdPriority, ignoreOverload, valueLeafFlags, metric, pathAffinities, pathSrlgs);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultPathProperties) {
            DefaultPathProperties other = (DefaultPathProperties) obj;
            return
                Objects.deepEquals(bandwidthGeneric, other.bandwidthGeneric) &&
                Objects.deepEquals(setupPriority, other.setupPriority) &&
                Objects.deepEquals(holdPriority, other.holdPriority) &&
                Objects.deepEquals(ignoreOverload, other.ignoreOverload) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(metric, other.metric) &&
                Objects.deepEquals(pathAffinities, other.pathAffinities) &&
                Objects.deepEquals(pathSrlgs, other.pathSrlgs);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("bandwidthGeneric=" + bandwidthGeneric)
            .add("setupPriority=" + setupPriority)
            .add("holdPriority=" + holdPriority)
            .add("ignoreOverload=" + ignoreOverload)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("metric=" + metric)
            .add("pathAffinities=" + pathAffinities)
            .add("pathSrlgs=" + pathSrlgs)
            .toString();
    }


    /**
     * Creates an instance of defaultPathProperties.
     */
    public DefaultPathProperties() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}