/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.receiverinfo.receivers;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.yrtietfinettypes.rev20130715.yrtietfinettypes.Host;
import org.onosproject.yang.gen.v1.yrtietfinettypes.rev20130715.yrtietfinettypes.PortNumber;
import org.onosproject.yang.gen.v1.yrtsubscription.yrtsubscription.TransportProtocol;
import org.onosproject.yang.model.InnerModelObject;
import org.onosproject.yang.model.MultiInstanceObject;

/**
 * Represents the implementation of receiver.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultReceiver extends InnerModelObject 
        implements Receiver, MultiInstanceObject<ReceiverKeys> {
    protected Host address;
    protected PortNumber port;
    protected TransportProtocol protocol;
    protected BitSet valueLeafFlags = new BitSet();

    @Override
    public Host address() {
        return address;
    }

    @Override
    public PortNumber port() {
        return port;
    }

    @Override
    public TransportProtocol protocol() {
        return protocol;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public void address(Host address) {
        valueLeafFlags.set(LeafIdentifier.ADDRESS.getLeafIndex());
        this.address = address;
    }

    @Override
    public void port(PortNumber port) {
        valueLeafFlags.set(LeafIdentifier.PORT.getLeafIndex());
        this.port = port;
    }

    @Override
    public void protocol(TransportProtocol protocol) {
        valueLeafFlags.set(LeafIdentifier.PROTOCOL.getLeafIndex());
        this.protocol = protocol;
    }

    @Override
    public int hashCode() {
        return Objects.hash(address, port, protocol, valueLeafFlags);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultReceiver) {
            DefaultReceiver other = (DefaultReceiver) obj;
            return
                Objects.deepEquals(address, other.address) &&
                Objects.deepEquals(port, other.port) &&
                Objects.deepEquals(protocol, other.protocol) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("address=" + address)
            .add("port=" + port)
            .add("protocol=" + protocol)
            .add("valueLeafFlags=" + valueLeafFlags)
            .toString();
    }


    /**
     * Creates an instance of defaultReceiver.
     */
    public DefaultReceiver() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}