/*
 * Copyright 2018-present Open Networking Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.vpnservicemulticast.multicast.rp;

import java.util.BitSet;
import java.util.Objects;
import java.util.StringJoiner;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.MulticastRpDiscoveryType;
import org.onosproject.yang.gen.v1.ietfl3vpnsvc.rev20160730.ietfl3vpnsvc.vpnservicemulticast.multicast.rp.rpdiscovery.BsrCandidates;
import org.onosproject.yang.model.InnerModelObject;

/**
 * Represents the implementation of rpDiscovery.
 *
 * <p>
 * valueLeafFlags identify the leafs whose value are explicitly set
 * Applicable in protocol edit and query operation.
 * </p>
 */
public class DefaultRpDiscovery extends InnerModelObject implements RpDiscovery {
    protected Class<? extends MulticastRpDiscoveryType> rpDiscoveryType;
    protected BitSet valueLeafFlags = new BitSet();
    protected BsrCandidates bsrCandidates;

    @Override
    public Class<? extends MulticastRpDiscoveryType> rpDiscoveryType() {
        return rpDiscoveryType;
    }

    @Override
    public BitSet valueLeafFlags() {
        return valueLeafFlags;
    }

    @Override
    public BsrCandidates bsrCandidates() {
        return bsrCandidates;
    }

    @Override
    public void rpDiscoveryType(Class<? extends MulticastRpDiscoveryType> rpDiscoveryType) {
        valueLeafFlags.set(LeafIdentifier.RPDISCOVERYTYPE.getLeafIndex());
        this.rpDiscoveryType = rpDiscoveryType;
    }

    @Override
    public void bsrCandidates(BsrCandidates bsrCandidates) {
        this.bsrCandidates = bsrCandidates;
    }

    @Override
    public int hashCode() {
        return Objects.hash(rpDiscoveryType, valueLeafFlags, bsrCandidates);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj instanceof DefaultRpDiscovery) {
            DefaultRpDiscovery other = (DefaultRpDiscovery) obj;
            return
                Objects.deepEquals(rpDiscoveryType, other.rpDiscoveryType) &&
                Objects.deepEquals(valueLeafFlags, other.valueLeafFlags) &&
                Objects.deepEquals(bsrCandidates, other.bsrCandidates);
        }
        return false;
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", getClass().getSimpleName() +"{", "}")
            .add("rpDiscoveryType=" + rpDiscoveryType)
            .add("valueLeafFlags=" + valueLeafFlags)
            .add("bsrCandidates=" + bsrCandidates)
            .toString();
    }


    /**
     * Creates an instance of defaultRpDiscovery.
     */
    public DefaultRpDiscovery() {
    }


    @Override
    public boolean isLeafValueSet(LeafIdentifier leaf) {
        return valueLeafFlags.get(leaf.getLeafIndex());
    }

}